
/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <cpl.h>

#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_framecombination.h"
#include "sph_test_zpl_simulator.h"
#include "sph_zpl_exposure.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"



/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_zpl_utils  Unit test of sph_zpl_utils
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_zpl_utils_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_zpl_utils_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_utils_calculate_master_frame_mean_weight_from_quad_image function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_calculate_master_frame_mean_weight_from_quad_image(void) {
    sph_master_frame* result = NULL;
    cpl_frame* zexpcube_plus = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_parameterlist* plist = NULL;
    sph_quad_image* quadimage = NULL;
    const int nframes = 3;
    double mean = 0.0;
    double rms = 0.0;
    double value1, value2, value3, value4, rmsd1, rmsd2, rmsd3, rmsd4;

    zexpcubes = cpl_frameset_new();

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_plus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_plus);

    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);

    quadimage =
            sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes(
                    zexpcubes, SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( quadimage );
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 170.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 170.0,
            1e-20);
    //sph_quad_image_save( quadimage, "test_sph_framecombination_quad_image.fits.tmp", NULL);

    value1 = 170.0;
    value2 = 30.0;
    value3 = 30.0;
    value4 = 170.0;
    rmsd1 = 2.;
    rmsd2 = 4.;
    rmsd3 = 5.;
    rmsd4 = 3.;

    //modify rms of the quad image in order to test better mean weight
    sph_utils_cpl_image_fill_double(quadimage->zero_image->iframe->rmsmap,
            rmsd1);
    sph_utils_cpl_image_fill_double(quadimage->zero_image->pframe->rmsmap,
            rmsd2);
    sph_utils_cpl_image_fill_double(quadimage->pi_image->iframe->rmsmap, rmsd3);
    sph_utils_cpl_image_fill_double(quadimage->pi_image->pframe->rmsmap, rmsd4);

    mean = (value1 / (rmsd1 * rmsd1) + value2 / (rmsd2 * rmsd2)
            + value3 / (rmsd3 * rmsd3) + value4 / (rmsd4 * rmsd4))
            / (1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2)
                    + 1. / (rmsd3 * rmsd3) + 1. / (rmsd4 * rmsd4));
    rms = sqrt(
            1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2) + 1. / (rmsd3 * rmsd3)
                    + 1. / (rmsd4 * rmsd4));

    result = sph_zpl_utils_calculate_master_frame_mean_weight_from_quad_image(
            quadimage);
    //printf("mean: %4.4f \n", mean);
    //printf("rms: %4.4f \n", rms);
    cpl_test_abs( cpl_image_get_mean( result->image ), mean, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->rmsmap ), rms, 1e-20);

    //sph_master_frame_save( result, "test_sph_zpl_utils_quad2master.fits.tmp", NULL );

    cpl_parameterlist_delete(plist);
    cpl_frameset_delete(zexpcubes);
    sph_quad_image_delete(quadimage);
    sph_master_frame_delete(result);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_calculate_stokes_param_double_image_from_quad_image(void) {
    sph_double_image* di = NULL;
    sph_quad_image* qi = NULL;
    cpl_frameset* quadframes = NULL;
    cpl_frame* qframe = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    quadframes =
            sph_test_simulator_create_quad_image_frames_from_zplexp_frames();

    qframe = cpl_frameset_get_first(quadframes);
    qi = sph_quad_image_load(cpl_frame_get_filename(qframe), 0);
    cpl_test_nonnull( qi );
    di = sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image(qi);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check(di);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            396.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            280.0, 1e-20);
    sph_double_image_save(di,
            "test_zpl_utils_stokes_param_double_image.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);
    cpl_frameset_delete(quadframes);
    sph_quad_image_delete(qi);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_utils_apply_polarization_flat function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_apply_polarization_flat(void) {
    sph_double_image* doubleimage = NULL;
    sph_double_image* polarization_flat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 100;

    doubleimage = sph_test_zpl_simulator_create_double_image(nx, ny, 100.0,
            10.0);
    polarization_flat = sph_test_zpl_simulator_create_double_image(nx, ny,
            200.0, 5.0);
    rerr = sph_zpl_utils_apply_polarization_flat(doubleimage,
            polarization_flat);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_double_image_quality_check(doubleimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( doubleimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            100.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( doubleimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            10.0-100.0*(5.0/200.), 1e-20);
    sph_double_image_delete(doubleimage);
    sph_double_image_delete(polarization_flat);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_utils_apply_modem_efficiency function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_apply_modem_efficiency(void) {
    sph_double_image* doubleimage = NULL;
    sph_master_frame* modem = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 100;

    doubleimage = sph_test_zpl_simulator_create_double_image(nx, ny, 100.0,
            10.0);
    modem = sph_test_zpl_simulator_create_master_frame(nx, ny, 5.0);
    rerr = sph_zpl_utils_apply_modem_efficiency(doubleimage, modem);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_double_image_quality_check(doubleimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( doubleimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            100.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( doubleimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            10.0/5.0, 1e-20);
    sph_double_image_delete(doubleimage);
    sph_master_frame_delete(modem);

    return;
}

#ifdef SPH_TEST_UTILS
static void cutest_test_zpl_utils_preproc(void) {
    cpl_frameset* rawframes;
    cpl_frameset* rframes;
    cpl_frame* frame;
    cpl_frame* cam1_result_frame;
    cpl_frame* cam2_result_frame;
    cpl_imagelist* rimagelist_cam1;
    cpl_imagelist* rimagelist_cam2;
    cpl_image* rimage_mean_cam1;
    cpl_image* rimage_mean_cam2;

    rawframes = cpl_frameset_new();
    frame = cpl_frame_new();
    if (frame) {
        cpl_frame_set_tag(frame, "RAW");
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_filename(frame, "ZIMPOL/data/test_double_ext.fits");
        //cpl_frame_set_filename( frame, "pol250_10sec_nshut.fits" );
    }
    cpl_frameset_insert(rawframes, frame);
    rframes = sph_zpl_utils_preproc(rawframes);
    cpl_test_assert( rframes);

    /* check with the etalons(python) results */
    /* camera-1 */
    // calculated
    cam1_result_frame = cpl_frameset_get_first(rframes);
    rimagelist_cam1 = cpl_imagelist_load(
            cpl_frame_get_filename(cam1_result_frame), CPL_TYPE_FLOAT, 0);
    rimage_mean_cam1 = cpl_imagelist_collapse_create(rimagelist_cam1);

    printf("\n rimage_mean_cam1 = %15.8f\n",
            cpl_image_get_absflux(rimage_mean_cam1));

    //etalon
    /*
     cam1_etalon_frame = cpl_frame_new();
     if ( cam1_etalon_frame ) {
     cpl_frame_set_tag( cam1_etalon_frame, "PREPROC");
     cpl_frame_set_group( cam1_etalon_frame, CPL_FRAME_GROUP_RAW);
     cpl_frame_set_filename( cam1_etalon_frame, "ZIMPOL/data/camera1_cube_zpl_exp_test_double_ext.fits" );
     }
     eimagelist_cam1 = cpl_imagelist_load( cpl_frame_get_filename( cam1_etalon_frame ), CPL_TYPE_FLOAT, 0);
     eimage_mean_cam1 = cpl_imagelist_collapse_create( eimagelist_cam1 );
     cpl_test_abs( cpl_image_get_absflux( rimage_mean_cam1 ),
     cpl_image_get_absflux( eimage_mean_cam1 ), 1e-16);
     //printf("rimage_mean_cam1 = %15.8f\n", cpl_image_get_absflux( rimage_mean_cam1 ));
     //printf("eimage_mean_cam1 = %15.8f\n", cpl_image_get_absflux( eimage_mean_cam1 ));
     *
     */

    // camera-2
    // calculated
    cam2_result_frame = cpl_frameset_get_next(rframes);
    rimagelist_cam2 = cpl_imagelist_load(
            cpl_frame_get_filename(cam2_result_frame), CPL_TYPE_FLOAT, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(rimagelist_cam2);
    cpl_image_delete(rimagelist_cam2);
    rimage_mean_cam2 = cpl_imagelist_collapse_create(rimagelist_cam2);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(rimage_mean_cam2);
    cpl_image_delete(rimage_mean_cam2);

    /*
     cam2_etalon_frame = cpl_frame_new();
     if ( cam2_etalon_frame ) {
     cpl_frame_set_tag( cam2_etalon_frame, "PREPROC");
     cpl_frame_set_group( cam2_etalon_frame, CPL_FRAME_GROUP_RAW);
     cpl_frame_set_filename( cam2_etalon_frame, "ZIMPOL/data/camera2_cube_zpl_exp_test_double_ext.fits" );
     }

     eimagelist_cam2 = cpl_imagelist_load( cpl_frame_get_filename( cam2_etalon_frame ), CPL_TYPE_FLOAT, 0);
     eimage_mean_cam2 = cpl_imagelist_collapse_create( eimagelist_cam2 );
     cpl_test_abs( cpl_image_get_absflux( rimage_mean_cam2 ),
     cpl_image_get_absflux( eimage_mean_cam2 ), 1e-16);
     //sph_filemanager_clean();
     *
     */
}
#endif

static void cutest_zpl_utils_get_frame_center(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    double cx0 = 0.0;
    double cy0 = 0.0;
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube2 = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* zexpcube_frames = NULL;

    const int nz = 5; //nframes
    const int nx = 10;
    const int ny = 10;
    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    double dV = 100.0;

    zexpcube_frames = cpl_frameset_new();

    zexpcube = sph_test_zpl_simulator_frame_template("test_utils_cube_1.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_utils_zplexp_single_frame.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        cpl_frameset_insert(zexpframes, zexpframe);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, zexpcube);
    cpl_frameset_insert(zexpcube_frames, zexpcube);
    //cpl_frame_delete( zexpcube );
    cpl_frameset_delete(zexpframes);

    zexpcube2 = sph_test_zpl_simulator_frame_template("test_utils_cube_2.fits");
    zexpframes = cpl_frameset_new();
    for (int i = nz; i < 2 * nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_utils_zplexp_single_frame.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        cpl_frameset_insert(zexpframes, zexpframe);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, zexpcube2);
    cpl_frameset_insert(zexpcube_frames, zexpcube2);
    //cpl_frame_delete( zexpcube );
    cpl_frameset_delete(zexpframes);

    rerr = sph_zpl_utils_get_frame_center(zexpcube_frames, &cx0, &cy0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(cx0, nx/2);
    cpl_test_eq(cy0, ny);
    cpl_frameset_delete(zexpcube_frames);

    return;
}

static void cutest_zpl_utils_subtract_overscans_zplexp(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zplexp_append = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frameset* subtrframes = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_table* ovsc_table = NULL;
    double a[8] = { 100.0, 1.0, 200.0, 2.0, 300.0, 3.0, 400.0, 4.0 };
    double b[8] = { 110.0, 11.0, 210.0, 12.0, 310.0, 31.0, 410.0, 14.0 };

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);
    cpl_test_nonnull( zplexp );
    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save_open zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame(
            "test_utils_zplexp_cube1_ovsc.fits", "PREPROC",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);

    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, zexpframe, NULL, ovsc_table);

    //printf("ADU1_ZERO_OVSC_MEAN = %f\n", cpl_table_get_double( ovsc_table,  ADU1_ZERO_OVSC_MEAN, 0, NULL ));

    zplexp_append = sph_test_zpl_simulator_create_exposure_4flat_double(100,
            100, 10.0, 20.0, 30.0, 40.0);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(b);
    if (zplexp_append->ovsc)
        cpl_propertylist_delete(zplexp_append->ovsc);
    zplexp_append->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);

    sph_zpl_exposure_save_append(zplexp_append, zexpframe, ovsc_table);
    //sph_cube_finalise_file( cpl_frame_get_filename(zexpframe));

    //printf("ADU1_ZERO_OVSC_MEAN = %f\n", cpl_table_get_double( ovsc_table,  ADU1_ZERO_OVSC_MEAN, 1, NULL ));

    sph_zpl_exposure_finalize_file(zexpframe, ovsc_table);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zplexp_append);

    //load a zpl exposure from the zpl exp frame cube: first plane = 0
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            a[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            a[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            a[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            a[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            a[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            a[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            a[7], 1e-20);

    sph_zpl_exposure_delete(zexpload);
    //load a zpl exposure from the zpl exp frame cube: second plane = 1
    zexpload = sph_zpl_exposure_load(zexpframe, 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            10.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           20.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ), 30.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ), 40.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            b[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            b[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            b[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            b[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            b[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            b[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            b[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            b[7], 1e-20);
    sph_zpl_exposure_delete(zexpload);

    //subtract overscans and verify if it is done correctly
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, zexpframe);
    subtrframes = sph_zpl_utils_subtract_overscans(inframes);
    zexpload = sph_zpl_exposure_load(cpl_frameset_get_first(subtrframes), 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
            ZERO_ODD_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    //printf( " zexpload->image_zero_odd: %f ", cpl_image_get_mean( zexpload->image_zero_odd ) );
    //printf( " ZERO_ODD_DOUBLE - 0.5*(a[0] + a[2]): %f", ZERO_ODD_DOUBLE - 0.5*(a[0] + a[2]));
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_even ),
            ZERO_EVEN_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_pi_odd ),
            PI_ODD_DOUBLE - 0.5*(a[4] + a[6]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_pi_even ),
            PI_EVEN_DOUBLE - 0.5*(a[4] + a[6]), 1e-10);
    sph_zpl_exposure_delete(zexpload);

    zexpload = sph_zpl_exposure_load(cpl_frameset_get_first(subtrframes), 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
            10 - 0.5*(b[0] + b[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_even ),
            20 - 0.5*(b[0] + b[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_pi_odd ),
            30 - 0.5*(b[4] + b[6]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_pi_even ),
            40 - 0.5*(b[4] + b[6]), 1e-10);

    sph_zpl_exposure_delete(zexpload);

    cpl_frameset_delete(inframes);
    cpl_table_delete(ovsc_table);
    cpl_frameset_delete(subtrframes);
    sph_filemanager_clean();

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_util_subtract_overscans_zpl_exposure_imaging function
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_subtract_overscans_zpl_exposure_imaging(void) {
    sph_zpl_exposure_imaging* zplexp = NULL;
    sph_zpl_exposure_imaging* zplexp_append = NULL;
    sph_zpl_exposure_imaging* zexpload = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frameset* subtrframes = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpframe1 = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_table* ovsc_table = NULL;
    double a[8] = { 100.0, 1.0, 200.0, 2.0 };
    double b[8] = { 110.0, 11.0, 210.0, 12.0 };

    zplexp = sph_test_zpl_simulator_create_exposure_imaging(100, 100);
    cpl_test_nonnull( zplexp );

    /* internal test of ph_test_zpl_simulator_create_exposure_imaging & _2flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_even ),
                           ZERO_EVEN_DOUBLE, 1e-20);

    // save open zpl exp imaging frame: plane = 0
    zexpframe = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_finalize_1.fits", "PREPROC IMAGING",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_imaging_ovsc_table_create_empty();
    sph_zpl_exposure_imaging_save_open(zplexp, zexpframe, NULL, ovsc_table);
    //printf("Filename fo zexpframe: %s \n", cpl_frame_get_filename( zexpframe ));
    // save append zpl expimaging : plane = 1

    zplexp_append = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(
            100, 100, 10.0, 20.0);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(b);
    if (zplexp_append->ovsc)
        cpl_propertylist_delete(zplexp_append->ovsc);
    zplexp_append->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    sph_zpl_exposure_imaging_save_append(zplexp_append, zexpframe, ovsc_table);

    sph_zpl_exposure_imaging_finalize_file(zexpframe, ovsc_table);
    //load a zpl exposure imaging from zpl exp imaging frame cube: plane = 0
    zexpload = sph_zpl_exposure_imaging_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_even ),
                           ZERO_EVEN_DOUBLE, 1e-20);
    cpl_test_abs(cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_MEAN ),
                           a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_RMS ), a[1],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_MEAN ), a[2],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_RMS ), a[3],
            1e-20);
    sph_zpl_exposure_imaging_delete(zexpload);

    //load a zpl exposure imaging from zpl exp imaging frame cube: plane = 1
    zexpload = sph_zpl_exposure_imaging_load(zexpframe, 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ), 10.0,
            1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_even ), 20.0, 1e-20);
    cpl_test_abs(cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_MEAN ),
                           b[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_RMS ), b[1],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_MEAN ), b[2],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_RMS ), b[3],
            1e-20);

    sph_zpl_exposure_imaging_delete(zplexp);
    sph_zpl_exposure_imaging_delete(zplexp_append);
    sph_zpl_exposure_imaging_delete(zexpload);
    cpl_table_delete(ovsc_table);

    //subtract overscans from zpl exposure imaging and verify if it is done correctly
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, zexpframe);
    subtrframes = sph_zpl_utils_subtract_overscans(inframes);
    zexpload = sph_zpl_exposure_imaging_load(
            cpl_frameset_get_first(subtrframes), 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            ZERO_ODD_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            ZERO_EVEN_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    sph_zpl_exposure_imaging_delete(zexpload);

    zexpload = sph_zpl_exposure_imaging_load(
            cpl_frameset_get_first(subtrframes), 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            10 - 0.5*(b[0] + b[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            20 - 0.5*(b[0] + b[2]), 1e-10);

    sph_zpl_exposure_imaging_delete(zexpload);
    cpl_frameset_delete(subtrframes);

    //simulate and add a second cube to the inframes in order to test function for frameset of two cubes
    zplexp = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(100,
            100, 105.0, 205.0);
    cpl_test_nonnull( zplexp );

    /* internal test of sph_test_zpl_simulator_create_exposure_imaging & _2flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_odd ), 105, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_even ), 205, 1e-20);

    // save open zpl exp imaging frame: plane = 0
    zexpframe = sph_filemanager_create_temp_frame(
            "test_zplexp_imaging_finalize_2.fits", "PREPROC IMAGING",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_imaging_ovsc_table_create_empty();
    sph_zpl_exposure_imaging_save_open(zplexp, zexpframe, NULL, ovsc_table);
    //printf("Filename fo zexpframe: %s \n", cpl_frame_get_filename( zexpframe ));
    // save append zpl expimaging : plane = 1

    zplexp_append = sph_test_zpl_simulator_create_exposure_imaging_2flat_double(
            100, 100, 14.0, 24.0);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist_imaging(b);
    if (zplexp_append->ovsc)
        cpl_propertylist_delete(zplexp_append->ovsc);
    zplexp_append->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    sph_zpl_exposure_imaging_save_append(zplexp_append, zexpframe, ovsc_table);

    sph_zpl_exposure_imaging_finalize_file(zexpframe, ovsc_table);
    //load a zpl exposure imaging from zpl exp imaging frame cube: plane = 0
    zexpload = sph_zpl_exposure_imaging_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ), 105.0,
                            1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_even ), 205.0, 1e-20);
    cpl_test_abs(cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_MEAN ), a[0],
                           1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_RMS ), a[1],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_MEAN ), a[2],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_RMS ), a[3],
            1e-20);
    sph_zpl_exposure_imaging_delete(zexpload);
    //load a zpl exposure imaging from zpl exp imaging frame cube: plane = 1
    zexpload = sph_zpl_exposure_imaging_load(zexpframe, 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ), 14.0,
                            1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_even ), 24.0, 1e-20);
    cpl_test_abs(cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_MEAN ),
                           b[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_OVSC_RMS ), b[1],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_MEAN ), b[2],
            1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_OVSC_RMS ), b[3],
            1e-20);

    sph_zpl_exposure_imaging_delete(zplexp);
    sph_zpl_exposure_imaging_delete(zplexp_append);
    sph_zpl_exposure_imaging_delete(zexpload);
    cpl_table_delete(ovsc_table);

    //subtract overscans from zpl exposure imaging and verify if it is done correctly
    cpl_frameset_insert(inframes, zexpframe);
    //test for first cube
    subtrframes = sph_zpl_utils_subtract_overscans(inframes);
    zexpload = sph_zpl_exposure_imaging_load(
            cpl_frameset_get_first(subtrframes), 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            ZERO_ODD_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            ZERO_EVEN_DOUBLE - 0.5*(a[0] + a[2]), 1e-10);
    sph_zpl_exposure_imaging_delete(zexpload);
    zexpload = sph_zpl_exposure_imaging_load(
            cpl_frameset_get_first(subtrframes), 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            10.0 - 0.5*(b[0] + b[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            20.0 - 0.5*(b[0] + b[2]), 1e-10);
    sph_zpl_exposure_imaging_delete(zexpload);

    //test for second cube
    zexpframe1 = cpl_frameset_get_next(subtrframes);
    zexpload = sph_zpl_exposure_imaging_load(zexpframe1, 0);
    cpl_test_nonnull( zexpload );
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            105 - 0.5*(a[0] + a[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            205 - 0.5*(a[0] + a[2]), 1e-10);
    sph_zpl_exposure_imaging_delete(zexpload);
    zexpload = sph_zpl_exposure_imaging_load(zexpframe1, 1);
    cpl_test_nonnull( zexpload );
    cpl_test_abs( cpl_image_get_mean( zexpload->image_odd ),
            14.0 - 0.5*(b[0] + b[2]), 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_even ),
            24.0 - 0.5*(b[0] + b[2]), 1e-10);
    sph_zpl_exposure_imaging_delete(zexpload);

    cpl_frameset_delete(subtrframes);
    cpl_frameset_delete(inframes);
    sph_filemanager_clean();

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_zpl_utils_check_format function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_zpl_utils_check_format(void) {
    cpl_frame*        frame       = NULL;
    cpl_frameset*     rawframes   = NULL;
    sph_double_image* doubleimage = NULL;
    cpl_image*        image       = NULL;

    int nx = 100;
    int ny = 100;
    int format;

    //Unsupported data format
    doubleimage = sph_test_zpl_simulator_create_double_image(nx, ny, 100.0,
            10.0);
    frame = sph_filemanager_create_temp_frame("test_doubleimage_check_format", "RAW", CPL_FRAME_GROUP_NONE );
    sph_double_image_save(doubleimage, cpl_frame_get_filename( frame ), NULL, CPL_IO_CREATE);
    rawframes = cpl_frameset_new();
    cpl_frameset_insert(rawframes, frame);

    format = sph_zpl_utils_check_format(rawframes);
    cpl_test_eq(format, -1);

    if (sph_error_get_last_code() != CPL_ERROR_NONE) cpl_error_reset();
    sph_double_image_delete(doubleimage);
    cpl_frameset_delete(rawframes); rawframes = NULL;

    //old style format
    image = sph_test_image_tools_create_flat_image_double(nx, ny, 10.0);
    frame = sph_filemanager_create_temp_frame("test_zpl_raw_oldsyle_check", "RAW", CPL_FRAME_GROUP_NONE );
    cpl_image_save( image, cpl_frame_get_filename(frame), CPL_TYPE_FLOAT, NULL, CPL_IO_CREATE );
    cpl_image_save( image, cpl_frame_get_filename(frame), CPL_TYPE_FLOAT, NULL,  CPL_IO_EXTEND );
    cpl_image_save( image, cpl_frame_get_filename(frame), CPL_TYPE_FLOAT, NULL, CPL_IO_EXTEND );

    rawframes = cpl_frameset_new();
    cpl_frameset_insert(rawframes, frame);

    format = sph_zpl_utils_check_format(rawframes);
    cpl_test_zero(format);
    cpl_frameset_delete(rawframes);
    cpl_image_delete( image );

    sph_filemanager_clean();


    return;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_zpl_utils",
            cutest_init_zpl_utils_testsuite, cutest_clean_zpl_utils_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


#ifdef SPH_TEST_UTILS
    /* Missing FITS input */
     if ( NULL == sph_test_do(pSuite, "Testing pre-processing",
     cutest_test_zpl_utils_preproc) )
     {
     return sph_test_get_error();
     }
#endif


    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image",
                    cutest_zpl_utils_calculate_stokes_param_double_image_from_quad_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_zpl_utils_calculate_master_frame_mean_weight_from_quad_image",
                    cutest_zpl_utils_calculate_master_frame_mean_weight_from_quad_image)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_zpl_utils_apply_polarization_flat",
                    cutest_zpl_utils_apply_polarization_flat)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_zpl_utils_apply_modem_efficiency",
                    cutest_zpl_utils_apply_modem_efficiency)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_zpl_utils_get_frame_center",
                    cutest_zpl_utils_get_frame_center)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_zpl_utils_subtract_overscans_zplexp",
                    cutest_zpl_utils_subtract_overscans_zplexp)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite,
                    "cutest_zpl_utils_subtract_overscans_zpl_exposure_imaging",
                    cutest_zpl_utils_subtract_overscans_zpl_exposure_imaging)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite,
                    "cutest_zpl_utils_check_format: Check zimpol rawdata format",
                    cutest_zpl_utils_check_format)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}
