/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_point_pattern.h"
#include "sph_andromeda_support.h"
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Other Helper Functions
 */
/*----------------------------------------------------------------------------*/
static cpl_imagelist* cutest_sph_andromeda_support_helper_create_imlist(
        int nangs, int nx, int ny, cpl_vector** angles,
        cpl_imagelist** varinlist) {
    cpl_image* imageA = NULL;
    cpl_image* imageB = NULL;
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    double ang = 0.0;
    double dx = 0.0;
    double dy = 0.0;

    if (angles) {
        *angles = cpl_vector_new(nangs);
        cpl_test_nonnull( *angles );
    }
    if (varinlist) {
        *varinlist = cpl_imagelist_new();
        cpl_test_nonnull( *varinlist );
    }
    imlist = cpl_imagelist_new();
    cpl_test_nonnull( imlist );
    imageB = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(imageB, 128.5, 128.5, 1.0, 1.0, 1.0);

    for (ii = 0; ii < nangs; ++ii) {
        ang = ii * 15.7;
        dx = 10.0 * cos(CPL_MATH_RAD_DEG * ang);
        dy = 10.0 * sin(CPL_MATH_RAD_DEG * ang);

        imageA = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

        if (angles)
            cpl_vector_set(*angles, ii, ang);

        cpl_image_fill_gaussian(imageA, 128.5 + dx, 128.5 + dy, 1.0, 1.0, 1.0);
        cpl_image_add(imageA, imageB); //sph_test_image_tools_add_gauss(imageA,128.0,128.0,1.0,1.0);
        cpl_imagelist_set(imlist, imageA, ii);
        if (varinlist) {
            imageA = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);
            cpl_imagelist_set(*varinlist, imageA, ii);
        }
    }
    cpl_image_delete(imageB);
    return imlist;
}

static sph_andromeda_core*
cutest_sph_andromeda_support_helper_setupcore(void) {
    cpl_vector* angles = NULL;
    cpl_image* psfin = NULL;
    cpl_bivector* indices = NULL;
    sph_psfcube* psfcube = NULL;
    sph_andromeda_core* core = NULL;
    cpl_imagelist* imlist = NULL;

    imlist = cutest_sph_andromeda_support_helper_create_imlist(2, 256, 256,
            NULL, NULL);
    cpl_test_nonnull( imlist );
    angles = cpl_vector_new(2);
    cpl_test_nonnull( angles );
    cpl_vector_set(angles, 0, 0.0);
    cpl_vector_set(angles, 1, 90.0);
    indices = cpl_bivector_wrap_vectors(cpl_vector_new(1), cpl_vector_new(1));
    cpl_test_nonnull( indices );
    cpl_vector_set(cpl_bivector_get_x(indices), 0, 0);
    cpl_vector_set(cpl_bivector_get_y(indices), 0, 1);

    psfin = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(psfin, 16.0, 16.0, 1.0, 1.0, 1.0);

    psfcube = sph_andromeda_support_calc_psf_shift_subpix(psfin, 10);
    cpl_test_nonnull( psfcube );

    core = sph_andromeda_support_init_core(imlist, angles, indices, psfcube);

    cpl_image_delete(psfin);

    return core;
}
static
int cutest_sph_andromeda_check_all_in_pairs(cpl_vector* angin,
        cpl_bivector* result) {
    int vv = 0;
    int ii = 0;
    int found = 0;

    for (vv = 0; vv < cpl_vector_get_size(angin); ++vv) {
        found = 0;
        for (ii = 0; ii < cpl_bivector_get_size(result); ++ii) {
            if (cpl_vector_get(cpl_bivector_get_x(result), ii) == vv) {
                found = 1;
            }
            if (cpl_vector_get(cpl_bivector_get_y(result), ii) == vv) {
                found = 1;
            }
        }
        if (found == 0)
            return 0;
    }
    return 1;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_create_angpairs_test1(void) {
    cpl_vector* angin = NULL;
    int ii = 0;
    int nangs = 20;
    double minang = 2.0;
    cpl_bivector* result = NULL;

    angin = cpl_vector_new(nangs);
    cpl_test_nonnull( angin );
    for (ii = 0; ii < nangs; ++ii) {
        cpl_vector_set(angin, ii, minang * 0.5 * (ii + 1));
    }

    result = sph_andromeda_support_create_angpairs(angin, minang, NULL);

    cpl_test_nonnull( result );
    cpl_test(cutest_sph_andromeda_check_all_in_pairs(angin,result));
    cpl_test_eq((int)cpl_vector_get(cpl_bivector_get_x_const(result),2), 2);
    cpl_test_eq((int)cpl_vector_get(cpl_bivector_get_y_const(result),2), 4);
    cpl_test_eq(cpl_bivector_get_size(result), 20);
    cpl_bivector_delete(result);
    cpl_vector_delete(angin);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_create_angpairs_test2(void) {
    cpl_vector* angin = NULL;
    int ii = 0;
    int nangs = 20;
    double minang = 2.0;
    cpl_bivector* result = NULL;

    angin = cpl_vector_new(nangs);
    cpl_test_nonnull( angin );
    for (ii = 0; ii < nangs; ++ii) {
        cpl_vector_set(angin, ii, minang * 0.01 * (ii * ii + 1));
    }

    result = sph_andromeda_support_create_angpairs(angin, minang, NULL);

    cpl_test_nonnull( result );

    cpl_test(cutest_sph_andromeda_check_all_in_pairs(angin,result));
    cpl_bivector_delete(result);
    cpl_vector_delete(angin);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_create_angpairs_test3(void) {
    cpl_vector* angin = NULL;
    int ii = 0;
    int nangs = 20;
    double minang = 2.0;
    cpl_bivector* result = NULL;
    gsl_rng* pRNG = NULL;
    double dx = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    angin = cpl_vector_new(nangs);
    cpl_test_nonnull( angin );
    for (ii = 0; ii < nangs; ++ii) {
        dx = gsl_ran_flat(pRNG, 0.001 * minang, 2.0 * minang);
        cpl_vector_set(angin, ii, minang * 0.01 * (ii + 1) + dx);
    }

    result = sph_andromeda_support_create_angpairs(angin, minang, NULL);

    cpl_test_nonnull( result );

    cpl_test(cutest_sph_andromeda_check_all_in_pairs(angin,result));
    cpl_bivector_delete(result);
    cpl_vector_delete(angin);
    gsl_rng_free(pRNG);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_calc_psf_shiftcube(void) {
    cpl_image* psfin = NULL;
    cpl_image* psfcomp = NULL;
    const cpl_image* comp = NULL;
    int nx = 64;
    int ny = 64;
    sph_psfcube* psfcube = NULL;
    double dx = 0.3;
    double dy = 0.7;

    psfin = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    psfcomp = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(psfin, 32.0, 32.0, 1.0, 1.0, 1.0);
    cpl_image_fill_gaussian(psfcomp, 32.0 + dx, 32.0 + dy, 1.0, 1.0, 1.0);

    psfcube = sph_andromeda_support_calc_psf_shift_subpix(psfin, 10);
    cpl_test_nonnull( psfcube );

    comp = sph_psfcube_get_exact(psfcube, dx, dy);
    cpl_test_nonnull( comp );
    cpl_image_subtract(psfcomp, comp);
    cpl_test_abs( cpl_image_get_mean(psfcomp), 0.0, 0.000001);
    cpl_test_abs( cpl_image_get_max(psfcomp), 0.0, 0.01);
    cpl_test_abs( cpl_image_get_min(psfcomp), 0.0, 0.01);
    cpl_image_delete(psfin);
    cpl_image_delete(psfcomp);
    sph_psfcube_delete(psfcube);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_cal_freq_image(void) {
    int nx = 16;
    int ny = 16;
    cpl_image* im = NULL;
    int bp = 0;

    im = sph_andromeda_support_cal_freq_image(nx, ny, (nx - 1.0) / 2.0,
            (ny - 1.0) / 2.0);
    cpl_test_nonnull( im );

    cpl_image_save(im, "distim.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_test_abs(cpl_image_get(im,9,9,&bp), 0.707, 0.01);
    cpl_test_abs(cpl_image_get(im,9,16,&bp), 8.0, 0.8);
    cpl_image_delete(im);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_calc_mask_image(void) {
    int nx = 16;
    int ny = 16;
    cpl_image* im = NULL;
    cpl_image* mask = NULL;
    int bp = 0;

    im = sph_andromeda_support_cal_freq_image(nx, ny, (nx - 1.0) / 2.0,
            (ny - 1.0) / 2.0);
    cpl_test_nonnull( im );
    mask = sph_andromeda_support_calc_mask_image(im, 3.0, 6.0);
    cpl_test_nonnull( mask );
    cpl_image_save(mask, "distmask.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_test_abs(cpl_image_get(mask,6,8,&bp), 0, 0.00000001);
    cpl_test_abs(cpl_image_get(mask,3,8,&bp), 1.0, 0.0000001);
    cpl_image_delete(im);
    cpl_image_delete(mask);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_angular_difference(void) {
    cpl_image* imageA = NULL;
    int nx = 256;
    int ny = 256;
    cpl_imagelist* imlist = NULL;
    cpl_bivector* indices = NULL;
    cpl_vector* angles = NULL;
    int nangs = 10;
    cpl_imagelist* newlist = NULL;

    imlist = cutest_sph_andromeda_support_helper_create_imlist(nangs, nx, ny,
            &angles, NULL);
    cpl_test_nonnull( imlist );
    indices = sph_andromeda_support_create_angpairs(angles, 5.0, NULL);
    cpl_test_nonnull( indices );

    newlist = sph_andromeda_support_angular_difference(imlist, NULL,
            cpl_bivector_get_x_const(indices),
            cpl_bivector_get_y_const(indices), 1.0, 50.0, 0, NULL, NULL, NULL);

    cpl_imagelist_save(newlist, "newlist.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    imageA = cpl_imagelist_get(newlist, 1);
    cpl_test_abs(cpl_image_get_mean(imageA), 0.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(imageA),
            -1.0 * cpl_image_get_min(imageA), 0.01);
    cpl_bivector_delete(indices);
    cpl_imagelist_delete(imlist);
    cpl_imagelist_delete(newlist);
    cpl_vector_delete(angles);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_andromeda_support_angular_difference_test2(void) {
    cpl_image* imageA = NULL;
    int nx = 256;
    int ny = 256;
    cpl_imagelist* imlist = NULL;
    cpl_bivector* indices = NULL;
    cpl_vector* angles = NULL;
    cpl_vector* gamma = NULL;
    int nangs = 10;
    int ii = 0;
    cpl_imagelist* newlist = NULL;

    imlist = cutest_sph_andromeda_support_helper_create_imlist(nangs, nx, ny,
            &angles, NULL);
    cpl_test_nonnull( imlist );

    indices = sph_andromeda_support_create_angpairs(angles, 5.0, NULL);
    cpl_test_nonnull( indices );

    newlist = sph_andromeda_support_angular_difference(imlist, NULL,
            cpl_bivector_get_x_const(indices),
            cpl_bivector_get_y_const(indices), 1.0, 50.0, 0, &gamma, NULL,
            NULL);
    cpl_test_assert(gamma);

    cpl_imagelist_save(newlist, "newlist_gam.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    imageA = cpl_imagelist_get(newlist, 1);
    cpl_test_abs(cpl_image_get_mean(imageA), 0.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(imageA),
            -1.0 * cpl_image_get_min(imageA), 0.01);
    for (ii = 0; ii < 10; ++ii) {
        cpl_test_abs(cpl_vector_get(gamma,ii), 1.0, 0.0001);
    }
    cpl_vector_delete(gamma);
    cpl_bivector_delete(indices);
    cpl_imagelist_delete(imlist);
    cpl_imagelist_delete(newlist);
    cpl_vector_delete(angles);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_angular_difference_test3(void) {
    cpl_image* imageA = NULL;
    int nx = 256;
    int ny = 256;
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* varinlist = NULL;
    cpl_bivector* indices = NULL;
    cpl_vector* angles = NULL;
    cpl_vector* gamma = NULL;
    int nangs = 10;
    int ii = 0;
    cpl_imagelist* newlist = NULL;
    cpl_imagelist* varlist = NULL;
    cpl_imagelist* wlist = NULL;

    imlist = cutest_sph_andromeda_support_helper_create_imlist(nangs, nx, ny,
            &angles, &varinlist);
    cpl_test_nonnull( imlist );

    indices = sph_andromeda_support_create_angpairs(angles, 5.0, NULL);
    cpl_test_nonnull( indices );

    newlist = sph_andromeda_support_angular_difference(imlist, varinlist,
            cpl_bivector_get_x_const(indices),
            cpl_bivector_get_y_const(indices), 1.0, 50.0, 0, &gamma, &varlist,
            &wlist);

    cpl_test_assert(varlist);
    cpl_test_assert(gamma);
    cpl_test_assert(wlist);
    cpl_imagelist_save(varlist, "varlist.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_imagelist_save(wlist, "wlist.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    imageA = cpl_imagelist_get(newlist, 1);
    cpl_test_abs(cpl_image_get_mean(imageA), 0.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(imageA),
            -1.0 * cpl_image_get_min(imageA), 0.01);
    for (ii = 0; ii < 10; ++ii) {
        cpl_test_abs(cpl_vector_get(gamma,ii), 1.0, 0.0001);
    }
    imageA = cpl_imagelist_get(varlist, 1);
    cpl_test_abs(cpl_image_get_mean(imageA), 0.2, 0.00001);
    cpl_test_abs(cpl_image_get_max(imageA), cpl_image_get_min(imageA),
            0.00001);
    imageA = cpl_imagelist_get(wlist, 1);
    cpl_test_abs(cpl_image_get_mean(imageA), 5.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(imageA), cpl_image_get_min(imageA),
            0.00001);
    cpl_vector_delete(gamma);
    cpl_bivector_delete(indices);
    cpl_imagelist_delete(imlist);
    cpl_imagelist_delete(newlist);
    cpl_imagelist_delete(varlist);
    cpl_imagelist_delete(wlist);
    cpl_vector_delete(angles);
    cpl_imagelist_delete(varinlist);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_get_region_test1(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_andromeda_core* core = NULL;
    FILE              * stream = cpl_msg_get_level() > CPL_MSG_DEBUG
        ? fopen("/dev/null", "a") : stdout;

    int xx = 128 + 50;
    int yy = 128;

    cpl_test_nonnull( stream );

    core = cutest_sph_andromeda_support_helper_setupcore();
    cpl_test_nonnull( core );

    rerr = sph_andromeda_support_get_region(xx, yy, 0, core);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    sph_andromeda_core_dump(stream, core);
    cpl_test_eq(core->minx, 111);
    cpl_test_eq(core->maxx, 193);
    cpl_test_eq(core->miny, 112);
    cpl_test_eq(core->maxy, 193);
    cpl_test_eq(core->posleft, 162);
    cpl_test_eq(core->posbot, 112);
    cpl_test_eq(core->negleft, 111);
    cpl_test_eq(core->negbot, 162);

    sph_andromeda_core_delete(core);

    if (stream != stdout) cpl_test_zero( fclose(stream) );
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_support_place_psfs(void) {
    sph_andromeda_core* core = NULL;
    cpl_image* im = NULL;
    cpl_image* compim = NULL;
    cpl_image* compim2 = NULL;
    int xx = 128 + 50;
    int yy = 128;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_point_pattern* pp = NULL;
    FILE              * stream = cpl_msg_get_level() > CPL_MSG_DEBUG
        ? fopen("/dev/null", "a") : stdout;


    cpl_test_nonnull( stream );

    core = cutest_sph_andromeda_support_helper_setupcore();
    cpl_test_nonnull( core );

    rerr = sph_andromeda_support_get_region(xx, yy, 0, core);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    sph_andromeda_core_dump(stream, core);
    im = sph_andromeda_support_place_psfs(core, -1.0);
    cpl_test_nonnull( im );

    //cpl_image_subtract(compim,im);

    pp = sph_point_pattern_new_from_image(im, NULL, 1.0, 5.0, 3.0, 3.0, 0,
                                          NULL);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), 2);
    compim2 = cpl_image_duplicate(im);
    compim = cpl_image_duplicate(im);
    cpl_image_fill_gaussian(compim, 67.5, 16.5, 1.0, 1.0, 1.0);
    cpl_image_fill_gaussian(compim2, 16.5, 66.5, 1.0, 1.0, 1.0);
    cpl_image_add(compim, compim2);
    cpl_image_delete(compim2);
    cpl_image_subtract(compim, im);
    cpl_test_abs(cpl_image_get_max(compim), 0.0, 0.001);
    cpl_test_abs(cpl_image_get_min(compim), 0.0, 0.001);
    cpl_image_delete(compim);
    cpl_image_delete(im);
    sph_point_pattern_delete(pp);

    sph_andromeda_core_delete(core);
    if (stream != stdout) cpl_test_zero( fclose(stream) );
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_test_init();

    pSuite = sph_add_suite("Testing cutest_sph_andromeda_support",
            cutest_init_testsuite, cutest_clean_testsuite);
    cpl_test_nonnull(pSuite);

    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_create_angpairs_test1",
                    cutest_sph_andromeda_support_create_angpairs_test1)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_create_angpairs_test2",
                    cutest_sph_andromeda_support_create_angpairs_test2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_create_angpairs_test3",
                    cutest_sph_andromeda_support_create_angpairs_test3)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_calc_psf_shiftcube",
                    cutest_sph_andromeda_support_calc_psf_shiftcube)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_cal_freq_image",
                    cutest_sph_andromeda_support_cal_freq_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_calc_mask_image",
                    cutest_sph_andromeda_support_calc_mask_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_angular_difference",
                    cutest_sph_andromeda_support_angular_difference)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_angular_difference_test2",
                    cutest_sph_andromeda_support_angular_difference_test2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_angular_difference_test3",
                    cutest_sph_andromeda_support_angular_difference_test3)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "cutest_sph_andromeda_support_get_region_test1",
                    cutest_sph_andromeda_support_get_region_test1)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_andromeda_support_place_psfs",
                    cutest_sph_andromeda_support_place_psfs)) {
        return sph_test_get_error();
    }

    return sph_test_end();    
}

/**@}*/
