/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_apertures.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_get_width_x(void) {
    sph_master_frame* mframe = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    double widthy = 3.5;
    double widthx = 2.2;
    double width = 0.0;
    double sx, sy;
    cpl_apertures* aps = NULL;
    mframe = sph_master_frame_new(100, 100);

    for (xx = 25; xx <= 75; ++xx) {
        for (yy = 25; yy <= 75; ++yy) {
            sx = widthx * 0.5 / sqrt(log(2.0));
            sy = widthy * 0.5 / sqrt(log(2.0));
            val = exp(-pow((xx - 50) / sx, 2.0))
                    * exp(-pow((yy - 50) / sy, 2.0));
            cpl_image_set(mframe->image, xx + 1, yy + 1, val);
        }
    }
    sph_master_frame_save(mframe, "sph_apertures_mframe.fits", NULL);

    aps = cpl_apertures_extract_sigma(mframe->image, 3.0);
    cpl_test_eq(cpl_apertures_get_size(aps), 1);
    width = sph_apertures_get_width_x(aps, 1, mframe);
    cpl_test_lt(fabs(width - widthx), 0.5);
    cpl_test_lt(fabs(width - widthx), 0.2);

    cpl_apertures_delete(aps);
    sph_master_frame_delete(mframe);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_get_width_y(void) {
    sph_master_frame* mframe = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    double widthy = 3.5;
    double widthx = 7.2;
    double width = 0.0;
    double sx, sy;
    cpl_apertures* aps = NULL;
    mframe = sph_master_frame_new(100, 100);

    for (xx = 25; xx <= 75; ++xx) {
        for (yy = 25; yy <= 75; ++yy) {
            sx = widthx * 0.5 / sqrt(log(2.0));
            sy = widthy * 0.5 / sqrt(log(2.0));
            val = exp(-pow((xx - 50) / sx, 2.0))
                    * exp(-pow((yy - 50) / sy, 2.0));
            cpl_image_set(mframe->image, xx + 1, yy + 1, val);
        }
    }
    sph_master_frame_save(mframe, "sph_apertures_mframe.fits", NULL);
    aps = cpl_apertures_extract_sigma(mframe->image, 3.0);
    cpl_test_eq(cpl_apertures_get_size(aps), 1);
    width = sph_apertures_get_width_y(aps, 1, mframe);
    cpl_test_lt(fabs(width - widthy), 0.5);
    cpl_test_lt(fabs(width - widthy), 0.2);
    cpl_apertures_delete(aps);
    sph_master_frame_delete(mframe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_get_width_y_pdt(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_descriptor* pdescr = NULL;
    sph_master_frame* mframe = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    double widthy = 3.5;
    double widthx = 7.2;
    double width = 0.0;
    double sx, sy;
    cpl_apertures* aps = NULL;
    mframe = sph_master_frame_new(100, 100);
    pdt = sph_pixel_description_table_new(100, 100, 0.0, 0.0);

    for (xx = 25; xx <= 75; ++xx) {
        for (yy = 25; yy <= 75; ++yy) {
            sx = widthx * 0.5 / sqrt(log(2.0));
            sy = widthy * 0.5 / sqrt(log(2.0));
            val = exp(-pow((xx - 50) / sx, 2.0))
                    * exp(-pow((yy - 50) / sy, 2.0));
            cpl_image_set(mframe->image, xx + 1, yy + 1, val);
            pdescr = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            pdescr->wavelength = yy / 70.0;
            pdescr->dlambda = 1.0 / 70.0;
        }
    }
    sph_pixel_description_table_save(pdt, "sph_apertures_pdt.fits", NULL);
    sph_master_frame_save(mframe, "sph_apertures_mframe.fits", NULL);
    aps = cpl_apertures_extract_sigma(mframe->image, 3.0);
    cpl_test_eq(cpl_apertures_get_size(aps), 1);
    width = sph_apertures_get_width_y_pdt(aps, 1, pdt, mframe);
    cpl_test_lt(fabs(width * 70 - widthy), 0.5);
    cpl_test_lt(fabs(width * 70 - widthy), 0.2);
    cpl_apertures_delete(aps);
    sph_master_frame_delete(mframe);
    sph_pixel_description_table_delete(pdt);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_apertures", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_apertures_get_width_x",
                    cutest_test_get_width_x)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_apertures_get_width_y",
                    cutest_test_get_width_y)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_apertures_get_width_y_pdt",
                    cutest_test_get_width_y_pdt)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
