/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_create_super_flat.h"
#include "sph_master_frame.h"
#include "sph_common_keywords.h"
#include "sph_test_image_tools.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat(void) {
    sph_master_frame* ldff1 = NULL;
    sph_master_frame* ldff2 = NULL;
    sph_master_frame* ldff3 = NULL;
    sph_master_frame* ldff4 = NULL;
    sph_master_frame* ldffbb = NULL;
    sph_master_frame* sdffbb = NULL;
    sph_master_frame* result = NULL;
    sph_master_frame* ideal = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    cpl_image* wim = NULL;
    cpl_propertylist* pl = NULL;

    model = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( model );
    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    cpl_test_nonnull( pdt );
    ldff1 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff1->image);
    ldff1->image = NULL;
    ldff2 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff2->image);
    ldff2->image = NULL;
    ldff3 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff3->image);
    ldff3->image = NULL;
    ldff4 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff4->image);
    ldff4->image = NULL;
    ldffbb = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldffbb->image);
    ldffbb->image = NULL;
    sdffbb = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(sdffbb->image);
    sdffbb->image = NULL;
    ldff1->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, model->minlambda);
    ldff2->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    ldff3->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    ldff4->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, model->maxlambda);
    ldffbb->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, 1.0);
    sdffbb->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, 1.0);

    cpl_propertylist_update_double(ldff1->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA, model->minlambda);
    cpl_propertylist_update_double(ldff2->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    cpl_propertylist_update_double(ldff3->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    cpl_propertylist_update_double(ldff4->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA, model->maxlambda);
    result = sph_create_super_flat(pdt, ldff1, ldff2, ldff3, ldff4);
    sph_master_frame_mask_tolerance(result, 0.5, 100000);
    sph_master_frame_quality_check(result);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;

    ideal = sph_master_frame_new_from_cpl_image(wim =
            sph_pixel_description_table_get_wavimage(pdt));
    sph_master_frame_subtract_master_frame(result, ideal);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat_difference.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;

    cpl_test_abs( sph_master_frame_get_absflux(result,NULL), 0.0,
            0.1);
    cpl_test_nonnull( result );
    cpl_test_error(CPL_ERROR_NONE);
    sph_master_frame_delete(ldff1);
    ldff1 = NULL;
    sph_master_frame_delete(ldff2);
    ldff2 = NULL;
    sph_master_frame_delete(ldff3);
    ldff3 = NULL;
    sph_master_frame_delete(ldff4);
    ldff4 = NULL;
    sph_master_frame_delete(ldffbb);
    ldffbb = NULL;
    sph_master_frame_delete(sdffbb);
    sdffbb = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(ideal);
    ideal = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    cpl_image_delete(wim);
    wim = NULL;
    cpl_error_reset();
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat_only_three(void) {
    sph_master_frame* ldff1 = NULL;
    sph_master_frame* ldff2 = NULL;
    cpl_image* wim = NULL;
    sph_master_frame* ldff3 = NULL;
    sph_master_frame* ldffbb = NULL;
    sph_master_frame* sdffbb = NULL;
    sph_master_frame* result = NULL;
    sph_master_frame* ideal = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;

    model = sph_ifs_lenslet_model_new(); //sph_test_create_small_lenslet_model();
    cpl_test_nonnull( model );
    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "sph_create_super_flat_pdt.fits", model);
    ldff1 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff1->image);
    ldff1->image = NULL;
    ldff2 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff2->image);
    ldff2->image = NULL;
    ldff3 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff3->image);
    ldff3->image = NULL;
    ldffbb = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldffbb->image);
    ldffbb->image = NULL;
    sdffbb = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(sdffbb->image);
    sdffbb->image = NULL;
    ldff1->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, model->minlambda);
    ldff2->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    ldff3->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    ldffbb->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, 1.0);
    sdffbb->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, 1.0);

    cpl_propertylist_update_double(ldff1->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA, model->minlambda);
    cpl_propertylist_update_double(ldff2->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    cpl_propertylist_update_double(ldff3->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    result = sph_create_super_flat(pdt, ldff1, ldff2, ldff3, NULL);
    sph_master_frame_mask_tolerance(result, 0.5, 100000);
    sph_master_frame_quality_check(result);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;

    ideal = sph_master_frame_new_from_cpl_image(wim =
            sph_pixel_description_table_get_wavimage(pdt));
    sph_master_frame_subtract_master_frame(result, ideal);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat_difference.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;

    cpl_test_abs( sph_master_frame_get_absflux(result,NULL), 0.0,
            0.1);
    cpl_test_nonnull( result );
    sph_master_frame_delete(ldff1);
    ldff1 = NULL;
    sph_master_frame_delete(ldff2);
    ldff2 = NULL;
    sph_master_frame_delete(ldff3);
    ldff3 = NULL;
    sph_master_frame_delete(ldffbb);
    ldffbb = NULL;
    sph_master_frame_delete(sdffbb);
    sdffbb = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(ideal);
    ideal = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    cpl_image_delete(wim);
    wim = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat_no_sdffb(void) {
    sph_master_frame* ldff1 = NULL;
    sph_master_frame* ldff2 = NULL;
    sph_master_frame* ldff3 = NULL;
    sph_master_frame* ldffbb = NULL;
    sph_master_frame* result = NULL;
    sph_master_frame* ideal = NULL;
    cpl_image* wim = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;

    model = sph_ifs_lenslet_model_new(); //sph_test_create_small_lenslet_model();
    cpl_test_nonnull( model );
    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "sph_create_super_flat_pdt.fits", model);
    ldff1 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff1->image);
    ldff1->image = NULL;
    ldff2 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff2->image);
    ldff2->image = NULL;
    ldff3 = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldff3->image);
    ldff3->image = NULL;
    ldffbb = sph_master_frame_new(model->detsize_pixels, model->detsize_pixels);
    cpl_image_delete(ldffbb->image);
    ldffbb->image = NULL;
    ldff1->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, model->minlambda);
    ldff2->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    ldff3->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    ldffbb->image = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels, model->detsize_pixels, 1.0);

    cpl_propertylist_update_double(ldff1->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA, model->minlambda);
    cpl_propertylist_update_double(ldff2->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.3333);
    cpl_propertylist_update_double(ldff3->properties,
            SPH_COMMON_KEYWORD_CALIB_LAMBDA,
            model->minlambda + (model->maxlambda - model->minlambda) * 0.6666);
    result = sph_create_super_flat(pdt, ldff1, ldff2, ldff3, NULL);
    sph_master_frame_mask_tolerance(result, 0.5, 100000);
    sph_master_frame_quality_check(result);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;
    ideal = sph_master_frame_new_from_cpl_image(wim =
            sph_pixel_description_table_get_wavimage(pdt));
    sph_master_frame_subtract_master_frame(result, ideal);
    sph_master_frame_save(result, "sph_create_super_flat_super_flat_difference.fits", pl =
            sph_ifs_lenslet_model_get_as_propertylist(model));
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_test_abs( sph_master_frame_get_absflux(result,NULL), 0.0,
            0.1);
    cpl_test_nonnull( result );
    sph_master_frame_delete(ldff1);
    ldff1 = NULL;
    sph_master_frame_delete(ldff2);
    ldff2 = NULL;
    sph_master_frame_delete(ldff3);
    ldff3 = NULL;
    sph_master_frame_delete(ldffbb);
    ldffbb = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(ideal);
    ideal = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    cpl_image_delete(wim);
    wim = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("create_super_flat", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing with wavelength flats",
                    cutest_test_create_flat)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing with only three wav. flats",
                    cutest_test_create_flat_only_three)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing without short FF",
                    cutest_test_create_flat_no_sdffb)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
