/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_dataset_peak.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
sph_dataset*
cutest_util_dataset_new_load(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        cpl_vector_set(vals, ii, (double) ii);
    }

    cpl_test_eq(cpl_vector_get( lams, 0 ), 1.0);
    cpl_test_eq(cpl_vector_get( lams, 99 ), 100.0);
    return dat;
}

static
sph_dataset*
cutest_util_dataset_new_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 50.0;
        val = exp(-val * val / 25.0);
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}

static
sph_dataset*
cutest_util_dataset_new_double_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    double valB = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 25.0;
        val = 5.0 * exp(-val * val / 400.0);
        valB = cpl_vector_get(lams, ii) - 75.0;
        valB = exp(-valB * valB / 400.0);
        val = val + valB;
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}

static
int cutest_init_dataset_peak_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_dataset_peak_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_peak_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_peak_new_root(void) {
    sph_dataset_peak* peak = NULL;
    sph_dataset* dat = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );

    peak = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( peak );

    cpl_test_error(CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );

    peak = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( peak );

    cpl_test_error(CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_peak_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_peak_new_sub(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset_peak* peak = NULL;
    sph_dataset* dat = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    peak = sph_dataset_peak_new_sub(root, 1.0);
    cpl_test_nonnull( peak );

    cpl_test_error(CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    peak = sph_dataset_peak_new_sub(root, 1.0);
    cpl_test_nonnull( peak );

    cpl_test_error(CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_peak_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_peak_find_subpeaks(void) {
    sph_dataset_peak* root = NULL;
    sph_dataset* dat = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_double_gauss();
    cpl_test_nonnull( dat );

    root = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( root );

    root->frac_threshold = 0.0;
    root->frac_acc = 0.01;

    rerr = sph_dataset_peak_find_subpeaks(root);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( root->firstchild );
    cpl_test_nonnull( root->firstchild->next );
    cpl_test_null( root->firstchild->next->next);
    cpl_test_abs( root->firstchild->maxpos, 25.0, 0.05);
    cpl_test_abs( root->firstchild->next->maxpos, 75.0, 1.15);
    if (cpl_test_get_failed() > 0) {
        printf("First peak: %lf, Second peak: %lf\n", root->firstchild->maxpos,
                root->firstchild->next->maxpos);
    }
    rerr = sph_dataset_peak_delete(root);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_peak_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_peak_delete(void) {
    sph_dataset* dat = NULL;
    sph_dataset_peak* peak = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    dat = cutest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );

    peak = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( peak );

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );

    peak = sph_dataset_peak_new_root(dat);
    cpl_test_nonnull( peak );

    rerr = sph_dataset_peak_delete(peak);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("dataset_peak_test",
            cutest_init_dataset_peak_testsuite,
            cutest_clean_dataset_peak_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_dataset_peak_new_root",
                    cutest_dataset_peak_new_root)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_peak_new_sub",
                    cutest_dataset_peak_new_sub)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_find_subpeaks",
                    cutest_dataset_peak_find_subpeaks)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_peak_delete",
                    cutest_dataset_peak_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
