/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_distortion_model.h"
#include "sph_distortion_map.h"
#include "sph_utils.h"
#include "sph_fitting.h"
#include "stdio.h"
#include "sph_test_image_tools.h"
#include "sph_time.h"

static
int cutest_init_testsuite(void);
static
int cutest_clean_testsuite(void);
static
void cutest_sph_distortion_model_apply(void);
static
void cutest_sph_distortion_model_apply_fixture(void);
static
void cutest_sph_distortion_model_scale(void);
static sph_distortion_model*
cutest_sph_distortion_model_setup_simple__(void);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Polynomial distortion model testsuite",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing apply",
                    cutest_sph_distortion_model_apply)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing scaling",
                    cutest_sph_distortion_model_scale)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    //sph_error_dump( SPH_ERROR_ERROR );
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_distortion_model_apply(void) {
    cutest_sph_distortion_model_apply_fixture();
}

static
void cutest_sph_distortion_model_apply_fixture(void) {
    int ngauss_x = 10;
    int ngauss_y = 10;
    int nx = 256;
    int ny = 256;
    cpl_image* undistorted_image = NULL;
    cpl_image* dedistorted_image = NULL;
    cpl_image* distorted_image = NULL;
    cpl_image* dumimage = NULL;
    int xx = 0;
    int yy = 0;
    int n = 0;
    double xpos[ngauss_x * ngauss_y];
    double ypos[ngauss_x * ngauss_y];
    double dx = 0.0;
    double dy = 0.0;
    double cx = 0.0;
    double cy = 0.0;
    double c1 = 10.0;
    double c2 = 0.1;
    sph_distortion_model* distmodel = NULL;

    distmodel = cutest_sph_distortion_model_setup_simple__();

    dx = nx / (ngauss_x + 1);
    dy = ny / (ngauss_y + 1);
    cx = nx / 2.0;
    cy = ny / 2.0;

    distmodel->offx = cx;
    distmodel->offy = cy;
    for (xx = 0; xx < ngauss_x; ++xx) {
        for (yy = 0; yy < ngauss_y; ++yy) {
            xpos[yy * ngauss_x + xx] = dx * (xx + 1);
            ypos[yy * ngauss_x + xx] = dy * (yy + 1);
        }
    }

    undistorted_image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

    for (n = 0; n < ngauss_x * ngauss_y; ++n) {
        dumimage = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image_fill_gaussian(dumimage, xpos[n], ypos[n], 1000.0, 3.0, 3.0);
        cpl_image_add(undistorted_image, dumimage);
        cpl_image_delete(dumimage);
        dumimage = NULL;
    }

    for (xx = 0; xx < ngauss_x; ++xx) {
        for (yy = 0; yy < ngauss_y; ++yy) {
            dx = c1 + c2 * (xpos[yy * ngauss_x + xx] - cx);
            dy = c1 + c2 * (xpos[yy * ngauss_x + xx] - cx);

            xpos[yy * ngauss_x + xx] -= dx;
            ypos[yy * ngauss_x + xx] -= dy;
        }
    }

    distorted_image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

    for (n = 0; n < ngauss_x * ngauss_y; ++n) {
        dumimage = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image_fill_gaussian(dumimage, xpos[n], ypos[n], 1000.0, 3.0, 3.0);
        cpl_image_add(distorted_image, dumimage);
        cpl_image_delete(dumimage);
        dumimage = NULL;
    }

    cpl_image_save(undistorted_image, "input_undistort.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
    cpl_image_delete(undistorted_image);
    cpl_image_save(distorted_image, "input_distort.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);

    dedistorted_image = sph_distortion_model_apply(distmodel,
                                                   distorted_image,
                                                   CPL_KERNEL_DEFAULT,
                                                   CPL_KERNEL_DEF_WIDTH,
                                                   NULL, NULL, NULL);
    cpl_image_delete(distorted_image);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_image_save(dedistorted_image, "output_dedistort.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
    cpl_image_delete(dedistorted_image);

    sph_distortion_model_delete(distmodel);

}

/*----------------------------------------------------------------------------*/
/**
 @brief
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_distortion_model_scale(void) {

    double posxy[2];
    double distxy[2];
    sph_distortion_model* distmodel =
        cutest_sph_distortion_model_setup_simple__();
    cpl_error_code code;

    cpl_test_nonnull(distmodel);

    posxy[0] = 1.0;
    posxy[1] = 1.0;
    sph_distortion_model_get_dist(distmodel, posxy, distxy);

    cpl_test_abs(distxy[0], 10.0 + 0.1, 0.0001);

    /* Changes units from 1.0 to 0.5 */
    code = sph_distortion_model_scale_units(distmodel, 0.5);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    posxy[0] = 0.0;
    posxy[1] = 0.0;
    sph_distortion_model_get_dist(distmodel, posxy, distxy);

    cpl_test_abs(distxy[0], 5.0, 0.0001);

    posxy[0] = 0.5;
    posxy[1] = 0.5;
    sph_distortion_model_get_dist(distmodel, posxy, distxy);

    cpl_test_abs(distxy[0], 0.5 * (10.0 + 0.1), 0.0001);

    sph_distortion_model_delete(distmodel);
}

static sph_distortion_model*
cutest_sph_distortion_model_setup_simple__(void) {
    sph_distortion_model* distmodel = NULL;
    double c1 = 10.0;
    double c2 = 0.1;
    cpl_size pows[2];

    distmodel = sph_distortion_model_new(NULL, NULL);
    pows[0] = pows[1] = 0;
    cpl_polynomial_set_coeff(distmodel->polyx, pows, c1);
    cpl_polynomial_set_coeff(distmodel->polyy, pows, c1);
    pows[0] = 1;
    pows[1] = 0;
    cpl_polynomial_set_coeff(distmodel->polyx, pows, c2);
    pows[0] = 1;
    pows[1] = 0;
    cpl_polynomial_set_coeff(distmodel->polyy, pows, c2);
    return distmodel;
}
/**@}*/
