/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_fft.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"


#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_fft.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"


#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_sph_fft_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_sph_fft_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_shift_image(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 10.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 148.0, 148.0, 10.0, 10.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 20.0, 20.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, im);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(im);
    cpl_image_delete(tim2);
    tim2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
static
void cutest_sph_fft_shift_image_partpixel(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 10.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 148.25, 148.25, 10.0, 10.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 20.25, 20.25);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, im);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

static
void cutest_sph_fft_shift_image_partpixel_fftw(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 10.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 148.25, 148.25, 10.0, 10.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 20.25, 20.25);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, im);
    cpl_image_save(tim2, "workimage_shift.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_shift_image_zero_shift(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 10.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 0.0, 0.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, tim);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_shift_image_zero_shift_radix2(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(128, 128, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 64.0, 64.0, 1.0, 1.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 0.0, 0.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, tim);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_shift_image_zero_shift_fftw(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 10.0);

    tim2 = sph_fft_shift_image(sphfft, tim, 0.0, 0.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, tim);
    cpl_image_save(tim2, "workimage_shift_zero.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_absflux(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.005);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.01);

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_freq_filter(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 100000.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 128.0, 128.0, 10.0, 100000.0);
    sph_test_image_tools_add_noise(tim, 10.0, NULL);
    tim2 = sph_fft_freq_filter(sphfft, tim, SPH_FFT_FILTER_METHOD_TH,
                               0.98, 0.0, 0.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, im);
    cpl_image_save(tim2, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_median(tim2);
    cpl_test_lt(val, 0.15);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.15);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.1 * cpl_image_get_max(im));

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_freq_filter_fftw(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 100000.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 128.0, 128.0, 10.0, 100000.0);
    sph_test_image_tools_add_noise(tim, 10.0, NULL);
    cpl_image_set(tim, 100, 100, 10000.0);
    cpl_image_set(im, 100, 100, 10000.0);
    tim2 = sph_fft_freq_filter(sphfft, tim, SPH_FFT_FILTER_METHOD_TH, 0.98, 0.0,
            0.0);
    cpl_test_nonnull( tim2 );
    cpl_image_save(tim, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(tim2, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim2, im);
    cpl_image_save(tim2, "workimage_filter.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_delete(tim);
    tim = NULL;
    val = cpl_image_get_median(tim2);
    cpl_test_lt(val, 0.15);
    val = cpl_image_get_mean(tim2);
    cpl_test_lt(val, 0.15);
    val = cpl_image_get_max(tim2);
    cpl_test_lt(val, 0.1 * cpl_image_get_max(im));

    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static cpl_image* cutest_sph_fft_freq_filter_fftw_compare(
        sph_fft_filter_method method, double rms, double bpix_val,
        double radius, double param1, double param2) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    tim = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    cpl_test_nonnull( tim );
    sph_test_image_tools_add_gauss(tim, 128.0, 128.0, 10.0, 100000.0);
    sph_test_image_tools_add_noise(tim, rms, NULL);
    cpl_image_set(tim, 100, 100, bpix_val * rms);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_add_gauss(im, 128.5, 128.5, 10.0, 100000.0);
    sph_test_image_tools_add_noise(im, rms, NULL);
    cpl_image_set(im, 100, 100, bpix_val * rms);
    tim2 = sph_fft_freq_filter(sphfft, tim, method, radius,
            param1, param2);
    cpl_test_nonnull( tim2 );
    cpl_image_delete(tim);
    tim = NULL;
    tim = sph_fft_shift_image(sphfft, tim2, 0.5, 0.5);
    cpl_image_subtract(tim, im);
    cpl_image_delete(im);
    cpl_image_delete(tim2);
    sph_fft_delete(sphfft);
    cpl_test_nonnull( tim );
    return tim;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_freq_filter_compare_methods(void) {
    cpl_imagelist* fermi_list = NULL;
    cpl_imagelist* th_list = NULL;
    cpl_imagelist* butter_list = NULL;
    int ii = 0;

    fermi_list = cpl_imagelist_new();
    th_list = cpl_imagelist_new();
    butter_list = cpl_imagelist_new();

    for (ii = 0; ii < 20; ++ii) {
        cpl_imagelist_set(
                th_list,
                cutest_sph_fft_freq_filter_fftw_compare(
                        SPH_FFT_FILTER_METHOD_TH, 10.0, 10.0 * ii, 0.5, 0.0,
                        0.0), ii);
        cpl_imagelist_set(
                fermi_list,
                cutest_sph_fft_freq_filter_fftw_compare(
                        SPH_FFT_FILTER_METHOD_FERMI, 10.0, 10 * ii, 0.1, 0.2,
                        0.0), ii);
        cpl_imagelist_set(
                butter_list,
                cutest_sph_fft_freq_filter_fftw_compare(
                        SPH_FFT_FILTER_METHOD_BUTTER, 10.0, 10 * ii, 0.1, 0.05,
                        0.3), ii);
    }
    cpl_imagelist_save(fermi_list, "fermi_list.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_imagelist_save(butter_list, "butter_list.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_imagelist_save(th_list, "th_list.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_imagelist_delete(fermi_list);
    fermi_list = NULL;
    cpl_imagelist_delete(butter_list);
    butter_list = NULL;
    cpl_imagelist_delete(th_list);
    th_list = NULL;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_fft_shift_test", cutest_init_sph_fft_testsuite,
            cutest_clean_sph_fft_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift", cutest_sph_fft_shift_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_freq_filter_fftw",
                    cutest_sph_fft_freq_filter_fftw)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_freq_filter_compare",
                    cutest_sph_fft_freq_filter_compare_methods)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_freq_filter",
                    cutest_sph_fft_freq_filter)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift_partpixel",
                    cutest_sph_fft_shift_image_partpixel)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift_partpixel_fftw",
                    cutest_sph_fft_shift_image_partpixel_fftw)) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift_zero",
                    cutest_sph_fft_shift_image_zero_shift)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift_zero_radix2",
                    cutest_sph_fft_shift_image_zero_shift_radix2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_shift_zero_fftw",
                    cutest_sph_fft_shift_image_zero_shift_fftw)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
