#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>

#include <string.h>
#include <stdio.h>
#ifdef HAVE_UNISTD_H
/* Needed for access() */
#include <unistd.h>
#endif

#include "cutest_sph_filter_generation.h"

#define PRO_RECi_PIPE_ID         "ESO PRO REC1 PIPE ID"
#define PRO_RECi_PIPE_ID_COMMENT "Pipeline (unique) identifier"

#define PRO_RECi_ID              "ESO PRO REC1 ID"
#define PRO_RECi_ID_COMMENT      "Pipeline recipe (unique)"


void sph_filter_plist_append_double_value_and_comment(cpl_propertylist * plist, const char * key,
		const double value, const char * comment){

    cpl_error_code err = cpl_propertylist_update_double(plist, key, value);
    cpl_test_eq_error(err, CPL_ERROR_NONE);

    if(!comment) return;

	err = cpl_propertylist_set_comment(plist, key, comment);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
}

void sph_filter_plist_append_string_value_and_comment(cpl_propertylist * plist, const char * key,
		const char * value, const char * comment){

    cpl_error_code err = cpl_propertylist_update_string(plist, key, value);
    cpl_test_eq_error(err, CPL_ERROR_NONE);

    if(!comment) return;

	err = cpl_propertylist_set_comment(plist, key, comment);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief Append Neutral Density filter magnitudes to the specified FITS file
 @param path              Initial path to filter frame
 @param base              Name of FITS file
 @param pipeid            Name of FITS producing "recipe"
 @param nd_filters_names  Names of the ND filters
 @param nd_filters_num    Number of ND filters
 @param cllab             Name of the filters
 @param cl_size           Number of filters
 @param nd             	  Matrix[nd_filters][filter] providing the attenuation
 	 	 	 	 	 	  for all combinations of filters and nd_filters
 @param doit              Perform the write operation
 @param catg			  Expected category of the table
 @param arm			  	  The arm we are generating the data for (e.g. IRDIS)
 @return   void

 */
/*----------------------------------------------------------------------------*/
void sph_filter_doit(const char  * path,
					 const char  * base,
					 const char  * pipeid,
					 const char ** nd_filters_names,
					 const size_t nd_filters_num,
					 const char **cllab,
					 const size_t cl_size,
					 const double ** nd,
					 cpl_boolean doit,
					 const char * catg,
					 const sph_filter_generation_arm arm)
{
    char * file = cpl_sprintf("%s%s", path, base);
    const int no_file =

#ifdef HAVE_ACCESS
        access(file, F_OK | R_OK | W_OK)
#else
        1
#endif
        ;

    cpl_test_nonnull(path);
    cpl_test_nonnull(base);

    if (no_file) {
        cpl_msg_info(cpl_func, "Not readable %s", file);
        cpl_free(file);
        return;
    }

	cpl_propertylist* pprim;
	cpl_propertylist* plist;
	cpl_table*        tfilter;
	const char*       strval;
	cpl_error_code    code;


	cpl_test_lt(0, nd_filters_num);

	/* Drop comments to avoid accumulating FITS ones */
	pprim = cpl_propertylist_load_regexp(file, 0, "^COMMENT", CPL_TRUE);
	cpl_test_error(CPL_ERROR_NONE);
	cpl_test_nonnull(pprim);

	strval = cpl_propertylist_get_string(pprim, "INSTRUME");
	cpl_test_error(CPL_ERROR_NONE);
	cpl_test_nonnull(strval);
	cpl_test_eq_string(strval, "SPHERE");

	strval = cpl_propertylist_get_string(pprim, "ESO PRO CATG");
	cpl_test_error(CPL_ERROR_NONE);
	cpl_test_nonnull(strval);
	cpl_test_eq_string(strval, catg);

	plist = cpl_propertylist_load(file, 1);
	cpl_test_error(CPL_ERROR_NONE);
	cpl_test_nonnull(plist);

	tfilter = cpl_table_load(file, 1, 0);

	cpl_propertylist_erase(pprim, PRO_RECi_PIPE_ID);
	cpl_propertylist_erase(pprim, PRO_RECi_ID);
	cpl_propertylist_erase(pprim, "ESO PRO SCIENCE");
	cpl_propertylist_erase(pprim, "ESO PRO TECH");
	cpl_propertylist_erase(pprim, "ESO PRO TYPE");
	cpl_propertylist_erase(pprim, "ESO PRO DID");
	cpl_propertylist_erase(pprim, "INTRUME");

	sph_filter_plist_append_string_value_and_comment(pprim, "INSTRUME", "SPHERE", "Instrument used");

	sph_filter_plist_append_double_value_and_comment(pprim, "MJD-OBS", 57113.0,
			"This calib cannot be used before this date");

	sph_filter_plist_append_string_value_and_comment(pprim, "ESO PRO CATG",
			catg, NULL);

	const char * arm_name =
			arm == SPH_FILTER_GENERATION_ARM_IRDIS ? "IRDIS" : "ZIMPOL";

	sph_filter_plist_append_string_value_and_comment(pprim, "ESO SEQ ARM",
			arm_name, "Name of the sub-system");

	code = cpl_propertylist_update_string(plist, "EXTNAME", "Wavelengths");
	cpl_test_eq_error(code, CPL_ERROR_NONE);

	if (doit) {
		code = cpl_table_save(tfilter, pprim, plist, file, CPL_IO_CREATE);
		cpl_test_eq_error(code, CPL_ERROR_NONE);
	} else {
		cpl_msg_info(cpl_func, "Run with extra CLI-argument to (over)write "
                             "%s (pipeid=%s)", file, pipeid ? pipeid : "<NULL>");
	}

	cpl_propertylist_empty(plist);
	cpl_test_error(CPL_ERROR_NONE);

	cpl_table_delete(tfilter);
	cpl_propertylist_delete(pprim);

	for (size_t i = 0; i < nd_filters_num; i++) {
		cpl_table* ndtab = cpl_table_new(1);
		cpl_size count;

		cpl_test_nonnull(ndtab);

		code = cpl_propertylist_update_string(plist, "EXTNAME", nd_filters_names[i]);
		cpl_test_eq_error(code, CPL_ERROR_NONE);

		for (size_t j = 0; j < cl_size; j++) {
			code = cpl_table_new_column(ndtab, cllab[j], CPL_TYPE_DOUBLE);
			cpl_test_eq_error(code, CPL_ERROR_NONE);

			code = cpl_table_set_column_unit(ndtab, cllab[j], "magnitude");
			cpl_test_eq_error(code, CPL_ERROR_NONE);

			cpl_test_lt(nd[i][j], 0.0);

			code = cpl_table_set_double(ndtab, cllab[j], 0, nd[i][j]);

			cpl_test_eq_error(code, CPL_ERROR_NONE);

		}

		count = cpl_table_get_nrow(ndtab);
		cpl_test_error(CPL_ERROR_NONE);
		cpl_test_eq(count, 1);

		count = cpl_table_get_ncol(ndtab);
		cpl_test_error(CPL_ERROR_NONE);
		cpl_test_eq(count, cl_size);

		if (doit) {
			code = cpl_table_save(ndtab, NULL, plist, file, CPL_IO_EXTEND);
			cpl_test_error(CPL_ERROR_NONE);
		}

		cpl_table_delete(ndtab);
	}
	cpl_propertylist_delete(plist);
    cpl_free(file);
}
