/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_filter_table.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filter_table_new(void) {
    sph_filter_table* filter_table = NULL;

    filter_table = sph_filter_table_new();
    cpl_test_nonnull( filter_table );
    cpl_test_zero(sph_filter_table_get_size(filter_table));
    sph_filter_table_delete(filter_table);
    filter_table = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filter_table_add_row(void) {
    sph_filter_table* filter_table = NULL;

    filter_table = sph_filter_table_new();
    cpl_test_nonnull( filter_table );
    sph_filter_table_add_row(filter_table, "FILTER1", 2.0, 3.0);
    cpl_test_eq(sph_filter_table_get_size(filter_table), 1);
    cpl_test_abs(cpl_vector_get(filter_table->left_lambda,0), 2.0,
            0.00001);
    cpl_test_abs(cpl_vector_get(filter_table->right_lambda,0), 3.0,
            0.00001);
    cpl_test_eq_string(filter_table->filterNames[0], "FILTER1");
    sph_filter_table_delete(filter_table);
    filter_table = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filter_table_get(void) {
    sph_filter_table* filter_table = NULL;
    cpl_vector* xc = NULL;
    cpl_vector* yc = NULL;
    double ll, lr;
    cpl_error_code code;

    filter_table = sph_filter_table_new();
    cpl_test_nonnull( filter_table );

    code = sph_filter_table_get_left_right_lambda(NULL, "F", &ll, &lr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = sph_filter_table_get_left_right_lambda(filter_table, NULL, &ll, &lr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = sph_filter_table_get_left_right_lambda(filter_table, "F", NULL, &lr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = sph_filter_table_get_left_right_lambda(filter_table, "F", &ll, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = sph_filter_table_get_left_right_lambda(filter_table, "F", &ll, &lr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    sph_filter_table_add_row(filter_table, "FILTER1", 2.0, 3.0);

    code = sph_filter_table_get_left_right_lambda(filter_table, "F", &ll, &lr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_rel(ll, -1.0, 0.00001);
    cpl_test_rel(lr, -1.0, 0.00001);

    code = sph_filter_table_get_left_right_lambda(filter_table, "FILTER1", &ll,
            &lr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_rel(ll, 2.0, 0.00001);
    cpl_test_rel(lr, 3.0, 0.00001);

    xc = sph_filter_table_get_left_lambdas(filter_table);
    cpl_test_nonnull( xc );
    yc = sph_filter_table_get_right_lambdas(filter_table);
    cpl_test_nonnull( yc );
    cpl_test_eq(cpl_vector_get_size(xc), 1);
    cpl_test_eq(cpl_vector_get_size(yc), 1);
    cpl_test_abs(cpl_vector_get(xc,0), 2.0, 0.00001);
    cpl_test_abs(cpl_vector_get(yc,0), 3.0, 0.00001);

    cpl_vector_delete(xc);
    xc = NULL;
    cpl_vector_delete(yc);
    yc = NULL;
    sph_filter_table_delete(filter_table);
    filter_table = NULL;
    return;
}

#ifdef SPH_TABLE_ASCII
/*----------------------------------------------------------------------------*/
/**
   @brief        A test function of the testsuite
*/
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filter_table_load_ascii(void) {
    sph_filter_table*        filter_table = NULL;
    FILE*               f = NULL;

    f = fopen("filter_table.txt","w");
    fprintf(f,"%s\n","# Hello this is a comment");
    fprintf(f,"%s\n","# Hello this is a comment too");
    fprintf(f,"  FILTER1     2.0      11.0\n");
    fprintf(f,"  FILTER2     22.0      111.0\n");
    fprintf(f,"  FILTER3     222.0      1111.0\n");
    fclose(f);
    filter_table = sph_filter_table_load_ascii("filter_table.txt");
    cpl_test_nonnull( filter_table );
    cpl_test_eq(sph_filter_table_get_size(filter_table), 3);
    cpl_test_abs(cpl_vector_get(filter_table->left_lambda,0),2.0,0.00001);
    cpl_test_abs(cpl_vector_get(filter_table->right_lambda,0),11.0,0.00001);
    cpl_test_eq_string(filter_table->filterNames[0],"FILTER1");
    cpl_test_abs(cpl_vector_get(filter_table->left_lambda,1),22.0,0.00001);
    cpl_test_abs(cpl_vector_get(filter_table->right_lambda,1),111.0,0.00001);
    cpl_test_eq_string(filter_table->filterNames[1],"FILTER2");
    cpl_test_abs(cpl_vector_get(filter_table->left_lambda,2),222.0,0.00001);
    cpl_test_abs(cpl_vector_get(filter_table->right_lambda,2),1111.0,0.00001);
    cpl_test_eq_string(filter_table->filterNames[2],"FILTER3");

    cpl_test_zero(unlink("filter_table.txt"));

    sph_filter_table_delete(filter_table); filter_table = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
   @brief        A test function of the testsuite
*/
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filter_table_save_ascii(void) {
    sph_filter_table*        filter_table = NULL;
    filter_table = sph_filter_table_new();
    cpl_error_code code;

    cpl_test_nonnull( filter_table );
    sph_filter_table_add_row(filter_table,"FILTER1",1.0,2.0);
    sph_filter_table_add_row(filter_table,"FILTER2",22.0,33.0);
    sph_filter_table_add_row(filter_table,"FITLER3",222.0,333.0);
    code = sph_filter_table_save_ascii(filter_table,"filter_table.txt");
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    sph_filter_table_delete(filter_table); filter_table = NULL;

    filter_table = sph_filter_table_load_ascii("filter_table.txt");
    cpl_test_nonnull( filter_table );
    cpl_test_eq(sph_filter_table_get_size(filter_table), 3);

    cpl_test_zero(unlink("filter_table.txt"));
    sph_filter_table_delete(filter_table); filter_table = NULL;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A testsuite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing constructor",
                    cutest_sph_filter_table_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Adding row",
                    cutest_sph_filter_table_add_row)) {
        return sph_test_get_error();
    }

#ifdef SPH_TABLE_ASCII
     if ( NULL == sph_test_do(pSuite, "Loading from ASCII",
     cutest_sph_filter_table_load_ascii ) )
     {
     return sph_test_get_error();
     }
#endif
    if (NULL
            == sph_test_do(pSuite, "get vectors", cutest_sph_filter_table_get)) {
        return sph_test_get_error();
    }

#ifdef SPH_TABLE_ASCII
     if ( NULL == sph_test_do(pSuite, "Saving to ASCII",
     cutest_sph_filter_table_save_ascii ) )
     {
     return sph_test_get_error();
     }
#endif

    return sph_test_end();
}

/**@}*/
