/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_fits.c"
#include "sph_fits.h"
#include "sph_time.h"
#include "sph_cube.h"
#include "sph_utils.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_INFO);
    sph_end_test();
    return 0;
}
static
double cutest_sph_fits_test_performance_fast_append(int nx, int ny, int maxn,
        int next, const char* filen) {
    cpl_image* tmpim = NULL;
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    cpl_vector* times = NULL;
    int ee = 0;
    const char* fname = "test_cube_sph_fits_fast_append.fits";
    FILE* fp = NULL;
    double result_time = 0.0;
    times = cpl_vector_new( (cpl_size) maxn);
    cpl_vector_fill(times, 0.0);
    cpl_test_nonnull( times );

    imlist = cpl_imagelist_new();
    cpl_test_nonnull( imlist );

    tmpim = sph_test_image_tools_create_flat_image_double(nx, ny, 0);
    cpl_test_nonnull( tmpim );

    cpl_imagelist_set(imlist, tmpim, 0);
    cpl_imagelist_save(imlist, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    for (ii = 1; ii < next; ++ii) {
        cpl_imagelist_save(imlist, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }
    cpl_imagelist_delete(imlist);
    imlist = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    tmpim = sph_test_image_tools_create_flat_image_double(nx, ny, 0);
    sph_time_start_timer(1);
    for (ee = 0; ee < next; ++ee) {
        for (ii = 0; ii < maxn; ++ii) {
            sph_fits_append_image(fname, tmpim, ee);
        }
    }
    result_time = sph_time_stop_timer(1);

    cpl_image_delete(tmpim);
    tmpim = NULL;
    fp = fopen(filen, "w");
    cpl_test_nonnull( fp );
    cpl_vector_dump(times, fp);
    fclose(fp);
    fp = NULL;
    cpl_vector_delete(times);
    times = NULL;
    return result_time;
}

static
double cutest_sph_fits_test_performance(int nx, int ny, int maxn, int next,
        const char* filen) {
    cpl_image* tmpim = NULL;
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    cpl_vector* times = NULL;
    int ee = 0;
    const char* fname = "test_cube_sph_fits_test_performance.fits";
    FILE* fp = NULL;
    double result_time = 0.0;

    cpl_test_error(CPL_ERROR_NONE);
    unlink(fname);
    times = cpl_vector_new( (cpl_size) maxn);
    cpl_vector_fill(times, 0.0);
    cpl_test_nonnull( times );

    imlist = cpl_imagelist_new();
    cpl_test_nonnull( imlist );

    tmpim = sph_test_image_tools_create_flat_image_double(nx, ny, 0);
    cpl_test_nonnull( tmpim );

    cpl_imagelist_set(imlist, tmpim, 0);
    cpl_imagelist_save(imlist, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    for (ii = 1; ii < next; ++ii) {
        cpl_imagelist_save(imlist, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }
    cpl_imagelist_delete(imlist);
    imlist = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    tmpim = sph_test_image_tools_create_flat_image_double(nx, ny, 0);
    unlink(fname);
    sph_time_start_timer(0);
    for (ii = 0; ii < maxn; ++ii) {
        sph_time_start_timer(1);
        for (ee = 0; ee < next; ++ee) {
            sph_cube_append_image(fname, tmpim, NULL, ee);
        }
        cpl_vector_set(times, ii, sph_time_stop_timer(1));
    }
    sph_cube_finalise_file(fname);
    result_time = sph_time_stop_timer(0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(tmpim);
    tmpim = NULL;
    fp = fopen(filen, "w");
    cpl_test_nonnull( fp );
    cpl_vector_dump(times, fp);
    fclose(fp);
    fp = NULL;
    cpl_vector_delete(times);
    times = NULL;
    unlink(fname);
    return result_time;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fits_test_performance_append_image(void) {
    int np = 3;
    int nx = 256 * 4;
    int ny = 256 * 4;
    double rt1, rt2, rt3, rt4;
    rt1 = cutest_sph_fits_test_performance(nx, ny, np, 1, "times_1.txt");
    rt2 = cutest_sph_fits_test_performance(nx, ny, np, 2, "times_2.txt");
    rt3 = cutest_sph_fits_test_performance(nx, ny, np, 3, "times_3.txt");
    rt4 = cutest_sph_fits_test_performance(nx, ny, np, 4, "times_4.txt");
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Times taken using normal mode: "
    "%f, %f, %f, %f", rt1, rt2, rt3, rt4);
    rt1 = cutest_sph_fits_test_performance_fast_append(nx, ny, np, 1,
            "times_1_f.txt");
    rt2 = cutest_sph_fits_test_performance_fast_append(nx, ny, np, 2,
            "times_2_f.txt");
    rt3 = cutest_sph_fits_test_performance_fast_append(nx, ny, np, 3,
            "times_3_f.txt");
    rt4 = cutest_sph_fits_test_performance_fast_append(nx, ny, np, 4,
            "times_4_f.txt");
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Times taken using fast mode: %f, %f, %f, %f", rt1, rt2, rt3, rt4);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A testsuite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing append image performance",
                    cutest_sph_fits_test_performance_append_image)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
