/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_frame_validator.h"

#include <math.h>
#include <math.h>
#include "sph_test.h"
#include "sph_cube.h"
#include "sph_master_frame.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_frame_validator_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_frame_validator_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_identical_true(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_frame* frame_shouldbe = cpl_frame_new();
    sph_cube* cube_sb = NULL;
    sph_cube* cube_is = NULL;
    sph_master_frame* master_frame = NULL;
    cpl_image* image = NULL;
    int result = 0;

    unlink("sph_frame_validator_validate_test_cube_is.fits");
    unlink("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_frame_set_tag(frame_shouldbe, "TESTFRAME");
    cube_is = sph_cube_new("sph_frame_validator_validate_test_cube_is.fits");
    cpl_test_nonnull( cube_is );
    cube_sb = sph_cube_new("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_test_nonnull( cube_sb );
    image = sph_test_image_tools_create_flat_image_int(248, 248, 1.0);
    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_is, master_frame, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_is, master_frame, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_is, master_frame, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    cpl_frame_set_filename(frame_is, "sph_frame_validator_validate_test_cube_is.fits");
    cpl_frame_set_filename(frame_shouldbe, "sph_frame_validator_validate_test_cube_sb.fits");
    sph_cube_finalise_file(cube_sb->filename);
    sph_cube_finalise_file(cube_is->filename);

    result = sph_frame_validator_validate_identical(frame_is, frame_shouldbe);

    sph_cube_delete(cube_sb);
    sph_cube_delete(cube_is);
    cpl_frame_delete(frame_is);
    cpl_frame_delete(frame_shouldbe);
    cpl_image_delete(image);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_VALIDATED);
}
/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_identical_different_next(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_frame* frame_shouldbe = cpl_frame_new();
    sph_cube* cube_sb = NULL;
    sph_cube* cube_is = NULL;
    sph_master_frame* master_frame = NULL;
    cpl_image* image = NULL;
    int result = 0;

    unlink("sph_frame_validator_validate_test_cube_is.fits");
    unlink("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_frame_set_tag(frame_shouldbe, "TESTFRAME");
    cube_is = sph_cube_new("sph_frame_validator_validate_test_cube_is.fits");
    cpl_test_nonnull( cube_is );
    cube_sb = sph_cube_new("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_test_nonnull( cube_sb );
    image = sph_test_image_tools_create_flat_image_int(248, 248, 1.0);
    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_is, master_frame, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    sph_cube_append_master(cube_is, master_frame, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;

    cpl_frame_set_filename(frame_is, "sph_frame_validator_validate_test_cube_is.fits");
    cpl_frame_set_filename(frame_shouldbe, "sph_frame_validator_validate_test_cube_sb.fits");
    sph_cube_finalise_file(cube_sb->filename);
    sph_cube_finalise_file(cube_is->filename);

    result = sph_frame_validator_validate_identical(frame_is, frame_shouldbe);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_NOT_VALIDATED);
    sph_cube_delete(cube_sb);
    sph_cube_delete(cube_is);
    cpl_frame_delete(frame_is);
    cpl_frame_delete(frame_shouldbe);
    cpl_image_delete(image);
}
/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_identical_different_size(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_frame* frame_shouldbe = cpl_frame_new();
    sph_cube* cube_sb = NULL;
    sph_cube* cube_is = NULL;
    sph_master_frame* master_frame = NULL;
    sph_master_frame* master_frame_is = NULL;
    cpl_image* image = NULL;
    cpl_image* image_is = NULL;
    int result = 0;

    unlink("sph_frame_validator_validate_test_cube_is.fits");
    unlink("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_frame_set_tag(frame_shouldbe, "TESTFRAME");
    cube_is = sph_cube_new("sph_frame_validator_validate_test_cube_is.fits");
    cpl_test_nonnull( cube_is );
    cube_sb = sph_cube_new("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_test_nonnull( cube_sb );
    image = sph_test_image_tools_create_flat_image_int(248, 248, 1.0);
    image_is = sph_test_image_tools_create_flat_image_int(242, 248, 1.0);
    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    cpl_frame_set_filename(frame_is, "sph_frame_validator_validate_test_cube_is.fits");
    cpl_frame_set_filename(frame_shouldbe, "sph_frame_validator_validate_test_cube_sb.fits");
    sph_cube_finalise_file(cube_sb->filename);
    sph_cube_finalise_file(cube_is->filename);

    result = sph_frame_validator_validate_identical(frame_is, frame_shouldbe);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_NOT_VALIDATED);
    sph_cube_delete(cube_sb);
    sph_cube_delete(cube_is);
    cpl_frame_delete(frame_is);
    cpl_frame_delete(frame_shouldbe);
    cpl_image_delete(image);
    cpl_image_delete(image_is);
}
/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_identical_different_pixel_val(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_frame* frame_shouldbe = cpl_frame_new();
    sph_cube* cube_sb = NULL;
    sph_cube* cube_is = NULL;
    sph_master_frame* master_frame = NULL;
    sph_master_frame* master_frame_is = NULL;
    cpl_image* image = NULL;
    cpl_image* image_is = NULL;
    int result = 0;

    unlink("sph_frame_validator_validate_test_cube_is.fits");
    unlink("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_frame_set_tag(frame_shouldbe, "TESTFRAME");
    cube_is = sph_cube_new("sph_frame_validator_validate_test_cube_is.fits");
    cpl_test_nonnull( cube_is );
    cube_sb = sph_cube_new("sph_frame_validator_validate_test_cube_sb.fits");
    cpl_test_nonnull( cube_sb );
    image = sph_test_image_tools_create_flat_image_int(248, 248, 1.0);
    image_is = sph_test_image_tools_create_flat_image_int(248, 248, 1.0);
    cpl_image_set(image_is, 10, 10, 2.0);
    cpl_image_set(image_is, 12, 12, -2.0);
    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    master_frame = sph_master_frame_new_from_cpl_image(image);
    master_frame_is = sph_master_frame_new_from_cpl_image(image_is);
    sph_cube_append_master(cube_is, master_frame_is, 2.0);
    sph_cube_append_master(cube_sb, master_frame, 3.0);
    sph_master_frame_delete(master_frame);
    master_frame = NULL;
    sph_master_frame_delete(master_frame_is);
    master_frame_is = NULL;

    cpl_frame_set_filename(frame_is, "sph_frame_validator_validate_test_cube_is.fits");
    cpl_frame_set_filename(frame_shouldbe, "sph_frame_validator_validate_test_cube_sb.fits");

    sph_cube_finalise_file(cube_sb->filename);
    sph_cube_finalise_file(cube_is->filename);

    result = sph_frame_validator_validate_identical(frame_is, frame_shouldbe);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_NOT_VALIDATED);
    sph_cube_delete(cube_sb);
    sph_cube_delete(cube_is);
    cpl_frame_delete(frame_is);
    cpl_frame_delete(frame_shouldbe);
    cpl_image_delete(image);
    cpl_image_delete(image_is);
}

/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_contains_keywords_soft_true(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_propertylist* pl_is = cpl_propertylist_new();
    cpl_propertylist* pl_sb = cpl_propertylist_new();
    sph_master_frame* mframe = sph_master_frame_new(120, 120);
    int result = 0;

    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_propertylist_append_bool(pl_is, "ESO ABOOL", CPL_TRUE);
    cpl_propertylist_append_char(pl_is, "ESO ACHAR", 'C');
    cpl_propertylist_append_double(pl_is, "ESO ADOUBLE", 10.0);
    cpl_propertylist_append_float(pl_is, "ESO AFLOAT", -20.0);
    cpl_propertylist_append_int(pl_is, "ESO AINT", 30);
    cpl_propertylist_append_string(pl_is, "ESO ASTRING", "TESTSTRING");
    cpl_propertylist_append_long(pl_is, "ESO ALONG", 123456789012345);
    sph_master_frame_save(mframe, "sph_frame_validator_proplist_test_mframe.fits", pl_is);
    cpl_frame_set_filename(frame_is, "sph_frame_validator_proplist_test_mframe.fits");

    cpl_propertylist_append_bool(pl_sb, "ESO ABOOL", CPL_TRUE);
    cpl_propertylist_append_char(pl_sb, "ESO ACHAR", 'C');
    cpl_propertylist_append_double(pl_sb, "ESO ADOUBLE", 10.0);
    cpl_propertylist_append_float(pl_sb, "ESO AFLOAT", -20.0);
    cpl_propertylist_append_int(pl_sb, "ESO AINT", 30);
    cpl_propertylist_append_string(pl_sb, "ESO ASTRING", "TESTSTRING");
    cpl_propertylist_append_long(pl_sb, "ESO ALONG", 123456789012345);

    result = sph_frame_validator_validate_contains_keywords_soft(frame_is, 0,
            pl_sb, 1);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_VALIDATED);
    cpl_frame_delete(frame_is);
    cpl_propertylist_delete(pl_is);
    cpl_propertylist_delete(pl_sb);
    sph_master_frame_delete(mframe);
}
/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_validate_contains_keywords_soft_true_no_vals(void) {
    cpl_frame* frame_is = cpl_frame_new();
    cpl_propertylist* pl_is = cpl_propertylist_new();
    cpl_propertylist* pl_sb = cpl_propertylist_new();
    sph_master_frame* mframe = sph_master_frame_new(120, 120);
    int result = 0;

    cpl_frame_set_tag(frame_is, "TESTFRAME");
    cpl_propertylist_append_bool(pl_is, "ESO ABOOL", CPL_TRUE);
    cpl_propertylist_append_char(pl_is, "ESO ACHAR", 'C');
    cpl_propertylist_append_double(pl_is, "ESO ADOUBLE", 10.0);
    cpl_propertylist_append_float(pl_is, "ESO AFLOAT", -20.0);
    cpl_propertylist_append_int(pl_is, "ESO AINT", 30);
    cpl_propertylist_append_string(pl_is, "ESO ASTRING", "TESTSTRING");
    cpl_propertylist_append_long(pl_is, "ESO ALONG", 123456789012345);
    sph_master_frame_save(mframe, "sph_frame_validator_proplist_test_mframe.fits", pl_is);
    cpl_frame_set_filename(frame_is, "sph_frame_validator_proplist_test_mframe.fits");

    cpl_propertylist_append_bool(pl_sb, "ESO ABOOL", CPL_FALSE);
    cpl_propertylist_append_char(pl_sb, "ESO ACHAR", 'B');
    cpl_propertylist_append_double(pl_sb, "ESO ADOUBLE", 110.0);
    cpl_propertylist_append_float(pl_sb, "ESO AFLOAT", -120.0);
    cpl_propertylist_append_int(pl_sb, "ESO AINT", 302);
    cpl_propertylist_append_string(pl_sb, "ESO ASTRING", "TESTRING");
    cpl_propertylist_append_long(pl_sb, "ESO ALONG", 1212456789012345);

    result = sph_frame_validator_validate_contains_keywords_soft(frame_is, 0,
            pl_sb, 0);
    cpl_test_eq(result, SPH_FRAME_VALIDATOR_VALIDATED);
    cpl_frame_delete(frame_is);
    cpl_propertylist_delete(pl_is);
    cpl_propertylist_delete(pl_sb);
    sph_master_frame_delete(mframe);
}
/*----------------------------------------------------------------------------*/
//
/*----------------------------------------------------------------------------*/
static
void cutest_sph_frame_validator_warn_keyword_tolerance(void) {
    cpl_frameset* inframeset = NULL;
    const char* keyname = "ESO ADOUB";
    double tol = 0.01;
    cpl_image* im = NULL;
    cpl_propertylist* pl = NULL;
    cpl_frame* aframe = NULL;
    int ii = 0;
    cpl_frameset* badframes = NULL;
    cpl_frameset* badinframes = NULL;

    inframeset = cpl_frameset_new();
    badinframes = cpl_frameset_new();
    for (ii = 0; ii < 10; ++ii) {
        aframe = sph_filemanager_create_temp_frame("TEST_FRAME_VALID", "ATAG",
                CPL_FRAME_GROUP_RAW);
        im = sph_test_image_tools_create_flat_image_int(256, 256, 10);
        pl = cpl_propertylist_new();
        cpl_propertylist_append_double(pl, keyname, 10.0);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_TYPE_INT, pl,
                CPL_IO_CREATE);
        cpl_frameset_insert(badinframes, cpl_frame_duplicate(aframe));
        cpl_frameset_insert(inframeset, aframe);
        cpl_propertylist_delete(pl);
        pl = NULL;
        cpl_image_delete(im);
        im = NULL;
    }

    for (ii = 0; ii < 10; ++ii) {
        aframe = sph_filemanager_create_temp_frame("TEST_FRAME_VALID", "ATAG",
                CPL_FRAME_GROUP_RAW);
        im = sph_test_image_tools_create_flat_image_int(256, 256, 10);
        pl = cpl_propertylist_new();
        cpl_propertylist_append_double(pl, keyname, 12.0);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_TYPE_INT, pl,
                CPL_IO_CREATE);
        cpl_frameset_insert(badinframes, aframe);
        cpl_propertylist_delete(pl);
        pl = NULL;
        cpl_image_delete(im);
        im = NULL;
    }

    cpl_test_eq(
            sph_frame_validator_warn_keyword_tolerance( inframeset,0,keyname,tol,NULL),
            SPH_FRAME_VALIDATOR_VALIDATED);
    badframes = cpl_frameset_new();
    cpl_test_eq(
            sph_frame_validator_warn_keyword_tolerance( inframeset,0,keyname,tol,badframes),
            SPH_FRAME_VALIDATOR_VALIDATED);
    cpl_test_zero(cpl_frameset_get_size(badframes));
    cpl_test_eq(
            sph_frame_validator_warn_keyword_tolerance( badinframes,0,keyname,tol,badframes),
            SPH_FRAME_VALIDATOR_NOT_VALIDATED);
    cpl_test_eq(cpl_frameset_get_size(badframes), 20);
    cpl_frameset_delete(badframes);
    badframes = NULL;
    cpl_frameset_delete(badinframes);
    badinframes = NULL;
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief  Main executable
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_frame_validator_test",
            cutest_init_frame_validator_testsuite,
            cutest_clean_frame_validator_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite,
                    "sph_frame_validator_validate_identical_true",
                    cutest_sph_frame_validator_validate_identical_true)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_frame_validator_validate_identical_different_next",
                    cutest_sph_frame_validator_validate_identical_different_next)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_frame_validator_validate_identical_different_size",
                    cutest_sph_frame_validator_validate_identical_different_size)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_frame_validator_validate_identical_different_pixel_val",
                    cutest_sph_frame_validator_validate_identical_different_pixel_val)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_frame_validator_validate_contains_keywords_soft_true",
                    cutest_sph_frame_validator_validate_contains_keywords_soft_true)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_frame_validator_validate_contains_keywords_soft_true_no_vals",
                    cutest_sph_frame_validator_validate_contains_keywords_soft_true_no_vals)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_frame_validator_validate_warn_keyword_tolerance",
                    cutest_sph_frame_validator_warn_keyword_tolerance)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/

