/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_interpolation.h"
#include "sph_utils.h"

#include <cpl.h>
#include <math.h>

static
int
cutest_sph_interpolation_init_testsuite(void);

static
int
cutest_sph_interpolation_clean_testsuite(void);

static
void
cutest_sph_interpolation_find_closest_points(void);

static
void cutest_sph_interpolation_find_closest_points_too_few(void);

static
void cutest_sph_interpolation_prepare_index_assocs(void);

static
void cutest_sph_interpolation_find_closest_points_many_reps(void);

static
void cutest_sph_interpolation_interpolate(void);

static cpl_vector*
cutest_sph_interpolation_find_closest_brute_force__(double xpos, double ypos,
        cpl_array* xpositions, cpl_array* ypositions, int n);
static sph_interpolation*
cutest_sph_interpolation_arrays_fixture__(int np, int nx, int ny);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_interpolation_find_closest_points(void) {
    cpl_vector* result = NULL;
    cpl_vector* expected_result = NULL;
    int np = 20 * 20;
    int ii = 0;
    int nx = 256;
    int ny = 256;
    double xpos = 55.5;
    double ypos = 251.5;
    sph_interpolation* interp = NULL;

    interp = cutest_sph_interpolation_arrays_fixture__(np, nx, ny);

    result = sph_interpolation_find_closest_points(interp, xpos, ypos);
    cpl_test_nonnull( result );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_vector_get_size(result), 3);

    cpl_vector_delete(result);
    result = sph_interpolation_find_closest_points(interp, xpos, ypos);
    cpl_test_nonnull( result );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_vector_get_size(result), 3);

    expected_result = cutest_sph_interpolation_find_closest_brute_force__(xpos,
            ypos, interp->xpositions, interp->ypositions, 3);
    cpl_test_nonnull( expected_result );

    for (ii = 0; ii < cpl_vector_get_size(result); ++ii) {
        cpl_test_abs( cpl_vector_get(result,ii),
                cpl_vector_get(expected_result,ii), 0.0001);
    }
    cpl_vector_delete(result);
    cpl_vector_delete(expected_result);
    sph_interpolation_delete(interp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_interpolation_find_closest_points_many_reps(void) {
    cpl_vector* result = NULL;
    cpl_vector* expected_result = NULL;
    int np = 20 * 20;
    int ii = 0;
    int nx = 256;
    int ny = 256;
    double xpos = 55.5;
    double ypos = 251.5;
    sph_interpolation* interp = NULL;
    gsl_rng* pRNG = NULL;
    int nreps = 1000;
    int rr = 0;

    interp = cutest_sph_interpolation_arrays_fixture__(np, nx, ny);

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    gsl_rng_set(pRNG, 2);
    cpl_test_nonnull( pRNG );
    for (rr = 0; rr < nreps; ++rr) {
        xpos = gsl_ran_flat(pRNG, 0.0, nx);
        ypos = gsl_ran_flat(pRNG, 0.0, ny);
        result = sph_interpolation_find_closest_points(interp, xpos, ypos);
        cpl_test_nonnull( result );
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(cpl_vector_get_size(result), 3);

        expected_result = cutest_sph_interpolation_find_closest_brute_force__(
                xpos, ypos, interp->xpositions, interp->ypositions, 3);
        cpl_test_nonnull( expected_result );

        for (ii = 0; ii < cpl_vector_get_size(result); ++ii) {
            cpl_test_abs( cpl_vector_get(result,ii),
                    cpl_vector_get(expected_result,ii), 0.0001);
        }
        cpl_vector_delete(result);
        cpl_vector_delete(expected_result);
    }
    sph_interpolation_delete(interp);
    gsl_rng_free(pRNG);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_interpolation_find_closest_points_too_few(void) {
    cpl_vector* result = NULL;
    cpl_vector* expected_result = NULL;
    int np = 2;
    int nx = 256;
    int ny = 256;
    double xpos = 10.0;
    double ypos = 10.0;
    sph_interpolation* interp = NULL;
    interp = cutest_sph_interpolation_arrays_fixture__(np, nx, ny);

    result = sph_interpolation_find_closest_points(interp, xpos, ypos);
    cpl_test_null( result);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_vector_delete(result);
    cpl_vector_delete(expected_result);
    sph_interpolation_delete(interp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_interpolation_prepare_index_assocs(void) {
    int np = 20 * 20;
    int nx = 256;
    int ny = 256;
    int pp = 0;
    sph_interpolation* interp = NULL;

    interp = cutest_sph_interpolation_arrays_fixture__(np, nx, ny);

    sph_interpolation_prepare_index_assocs(interp, nx, ny);

    cpl_test_nonnull( interp->index_imagelist );
    cpl_test_eq(cpl_imagelist_get_size(interp->index_imagelist), 3);

    for (pp = 0; pp < 3; ++pp) {
        cpl_test_eq(
                cpl_mask_count( cpl_image_get_bpm( cpl_imagelist_get(interp->index_imagelist,pp))),
                0);
    }

    sph_interpolation_delete(interp);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_interpolation_interpolate(void) {
    int np = 20 * 20;
    int nx = 256;
    int ny = 256;
    sph_interpolation* interp = NULL;
    cpl_image* result = NULL;
    cpl_array* values = NULL;

    interp = cutest_sph_interpolation_arrays_fixture__(np, nx, ny);

    sph_interpolation_prepare_index_assocs(interp, nx, ny);

    cpl_test_nonnull( interp->distances_imagelist );

    values = cpl_array_duplicate(interp->xpositions);
    cpl_array_multiply_scalar(values, 0.0);
    cpl_array_add_scalar(values, 1.0);
    result = sph_interpolation_interpolate(interp, values);
    cpl_test_nonnull( result );

    cpl_image_save(result, "result.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_subtract_scalar(result, 1.0);
    cpl_test_abs( cpl_image_get_absflux(result), 0.0, 0.00001);
    sph_interpolation_delete(interp);
    cpl_image_delete(result);
    cpl_array_delete(values);
    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/
static cpl_vector*
cutest_sph_interpolation_find_closest_brute_force__(double xpos, double ypos,
        cpl_array* xpositions, cpl_array* ypositions, int n) {
    cpl_vector* result = NULL;
    cpl_vector* used_indices = NULL;
    int vv = 0;
    int ii = 0;
    int minind = 0;
    double r = 0.0;
    double dx = 0.0;
    double dy = 0.0;
    double minr = 0.0;
    int bp = 0;
    result = cpl_vector_new(n);
    used_indices = cpl_vector_new(cpl_array_get_size(xpositions));
    cpl_vector_fill(used_indices, 0);
    cpl_vector_fill(result, -1);
    for (vv = 0; vv < n; ++vv) {
        minr = 1.0e10;
        minind = -1;
        for (ii = 0; ii < cpl_array_get_size(xpositions); ++ii) {
            if (cpl_vector_get(used_indices, ii) == 0) {
                dx = cpl_array_get(xpositions, ii, &bp) - xpos;
                dy = cpl_array_get(ypositions, ii, &bp) - ypos;
                r = dx * dx + dy * dy;
                if (r <= minr) {
                    minind = ii;
                    minr = r;
                }
            }
        }
        cpl_vector_set(result, vv, minind);
        cpl_vector_set(used_indices, minind, 1);
    }
    cpl_vector_delete(used_indices);
    return result;
}

static sph_interpolation*
cutest_sph_interpolation_arrays_fixture__(int np, int nx, int ny) {
    int ii = 0;
    gsl_rng* pRNG = NULL;
    sph_interpolation* interp = NULL;
    cpl_array* xpositions = NULL;
    cpl_array* ypositions = NULL;
    xpositions = cpl_array_new(np, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( xpositions );
    ypositions = cpl_array_new(np, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( ypositions );

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    gsl_rng_set(pRNG, 1);
    cpl_test_nonnull( pRNG );
    for (ii = 0; ii < np; ++ii) {
        cpl_array_set(xpositions, ii, gsl_ran_flat(pRNG, 0.0, nx));
        cpl_array_set(ypositions, ii, gsl_ran_flat(pRNG, 0.0, ny));
    }
    gsl_rng_free(pRNG);
    interp = sph_interpolation_new(xpositions, ypositions, 3);
    return interp;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_interpolation_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_interpolation_clean_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A cutest_sph_interpolation testsuite",
            cutest_sph_interpolation_init_testsuite,
            cutest_sph_interpolation_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing find closest points",
                    cutest_sph_interpolation_find_closest_points)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing find closest points too few",
                    cutest_sph_interpolation_find_closest_points_too_few)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "Testing prepare index assocs (no distances)",
                    cutest_sph_interpolation_prepare_index_assocs)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing find closest many reps",
                    cutest_sph_interpolation_find_closest_points_many_reps)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing interpolate",
                    cutest_sph_interpolation_interpolate)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
