/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_ldt.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include "cutest_sph_ldt_fixtures.h"

#include <cpl.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*-----------------------------------------------------------------------------
 Private
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_ldt_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_ldt_testsuite(void) {
    sph_error_dump(SPH_ERROR_INFO);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_new(void) {
    sph_ldt* ldt = NULL;
    int rerr = CPL_ERROR_NONE;
    sph_ifs_lenslet_model* model = NULL;

    /* Setup and run ...*/

    model = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( model );

    ldt = sph_ldt_new(model);
    cpl_test_nonnull( ldt );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt );
    cpl_test_eq(ldt->nlens, 217);
    sph_ldt_delete(ldt);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_divide(void) {
    sph_ldt* ldtA = NULL;
    sph_ldt* ldtB = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/

    ldtA = cutest_sph_ldt_fixtures_setup_ball_ldt();
    ldtB = cutest_sph_ldt_fixtures_setup_ball_ldt();
    cpl_test_nonnull( ldtA );
    cpl_test_nonnull( ldtB );

    /*Verify */
    rerr = sph_ldt_divide(ldtA, ldtB);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( ldtA );
    cpl_test_eq(ldtA->nlens, 217);
    sph_ldt_save(ldtA, "ldt_divided.fits", CPL_IO_CREATE);
    sph_ldt_delete(ldtA);
    sph_ldt_delete(ldtB);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_new_rotate(void) {
    sph_ldt* ldtA = NULL;
    sph_ldt* ldtB = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/

    ldtA = cutest_sph_ldt_fixtures_setup_ball_ldt();
    cpl_test_nonnull( ldtA );

    /*Verify */
    ldtB = sph_ldt_new_rotate_gimros(ldtA, 45.0, 0.1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( ldtB );
    cpl_test_eq(ldtB->nlens, 217);
    sph_ldt_save(ldtB, "ldt_rotated.fits", CPL_IO_CREATE);
    sph_ldt_delete(ldtA);
    sph_ldt_delete(ldtB);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_save(void) {
    sph_ldt* ldt = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    ldt = cutest_sph_ldt_fixtures_setup_ball_ldt();

    cpl_test_nonnull( ldt );

    /* run ...*/

    rerr = sph_ldt_save(ldt, "testldt.fits", CPL_IO_CREATE);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt->arr );
    cpl_test_eq(ldt->nlens, 217);
    sph_ldt_delete(ldt);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_save_big_cube(void) {
    sph_ldt* ldt = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    ldt = cutest_sph_ldt_fixtures_setup_big_ldt_ids();

    cpl_test_nonnull( ldt );

    /* run ...*/

    rerr = sph_ldt_save_cube(ldt, "testldt_cube.fits", NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, 0.00746,0.0,NULL);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt->arr );
    cpl_test_eq(ldt->nlens, 63511);
    sph_ldt_delete(ldt);
    ldt = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_save_big(void) {
    sph_ldt* ldt = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    ldt = cutest_sph_ldt_fixtures_setup_big_ldt();

    cpl_test_nonnull( ldt );

    /* run ...*/

    rerr = sph_ldt_save(ldt, "testldt.fits", CPL_IO_CREATE);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt->arr );
    cpl_test_eq(ldt->nlens, 63511);
    sph_ldt_delete(ldt);
    ldt = NULL;
    ldt = sph_ldt_load("testldt.fits");
    cpl_test_nonnull( ldt );
    cpl_test_nonnull( ldt->arr );
    cpl_test_eq(ldt->nlens, 63511);
    sph_ldt_delete(ldt);
    ldt = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_new_load(void) {
    sph_ldt* ldtA = NULL;
    sph_ldt* ldtB = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_frame* outframe = NULL;

    /* Setup and run ...*/

    ldtA = cutest_sph_ldt_fixtures_setup_ball_ldt();
    cpl_test_nonnull( ldtA );
    ldtB = cutest_sph_ldt_fixtures_setup_ball_ldt();
    cpl_test_nonnull( ldtB );
    cpl_test_eq(ldtB->nlens, 217);
    outframe = sph_filemanager_create_temp_frame("ldt_save_test.fits", "LDT",
            CPL_FRAME_GROUP_CALIB);
    cpl_test_nonnull( outframe );
    rerr = sph_ldt_save(ldtB, cpl_frame_get_filename(outframe), CPL_IO_CREATE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_ldt_delete(ldtB);
    ldtB = NULL;

    /* Loading B LDT again ... */
    ldtB = sph_ldt_load(cpl_frame_get_filename(outframe));
    /*Verify */
    cpl_test_nonnull( ldtB );
    cpl_test_eq(ldtB->nlens, 217);

    cpl_frame_delete(outframe);
    sph_ldt_delete(ldtA);
    sph_ldt_delete(ldtB);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_create_image function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_create_image(void) {
    sph_ldt* ldt = NULL;
    sph_pixel_description_table* pdt = NULL;
    cpl_image* image = NULL;
    sph_master_frame* mframe = NULL;
    int xx = 0;
    int yy = 0;
    int rerr = CPL_ERROR_NONE;
    cpl_frameset* frames = NULL;
    cpl_image* idim = NULL;
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;
    const char* filename;

    /* Setup */
    frames = cpl_frameset_new();
    sph_test_create_ifs_master_pdt_frame(frames);
    cpl_test_error(CPL_ERROR_NONE);
    filename = cpl_frame_get_filename(cpl_frameset_get_first_const(frames));
    idim = cpl_image_load(filename, CPL_TYPE_INT, 0, 2);
    cpl_test_nonnull( idim );
    pdt = sph_pixel_description_table_load(filename);
    cpl_test_nonnull( pdt );
    pl = cpl_propertylist_load(filename, 0);
    cpl_test_nonnull( pl );
    model = sph_ifs_lenslet_model_new_from_propertylist(pl);
    cpl_test_nonnull( model );
    cpl_propertylist_delete(pl);
    pl = NULL;

    image = cpl_image_new(256, 256, CPL_TYPE_FLOAT);

    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 0.0);
            if (cpl_image_get(idim, xx + 1, yy + 1, &rerr) > 0) {
                cpl_image_set(image, xx + 1, yy + 1, 12.3 * (xx + yy));
            }
        }
    }

    /* run ...*/

    ldt = sph_ldt_new_from_pdt_image(pdt, model, image);

    cpl_test_nonnull( ldt );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt->arr );
    cpl_image_delete(image);
    image = NULL;
    mframe = sph_ldt_create_mframe(ldt, 256, 256, 2, 0.0, NULL);
    image = mframe->image;
    cpl_test_nonnull( image );
    rerr = cpl_image_save(image, "ldt_test_image_plane_2.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_DEFAULT);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_master_frame_delete(mframe);
    mframe = NULL;

    sph_ldt_delete(ldt);
    cpl_image_delete(idim);
    cpl_frameset_delete(frames);
    sph_pixel_description_table_delete(pdt);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_new_from_pdt_image(void) {
    sph_ldt* ldt = NULL;
    sph_pixel_description_table* pdt = NULL;
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;
    int rerr = CPL_ERROR_NONE;
    cpl_frameset* frames = NULL;
    cpl_image* idim = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_lenslet_descriptor* ldesc = NULL;
    int id = 0;
    cpl_propertylist* pl = NULL;
    /* Setup */
    frames = cpl_frameset_new();
    sph_test_create_ifs_master_pdt_frame(frames);
    cpl_test_error(CPL_ERROR_NONE);
    pdt = sph_pixel_description_table_load(
            cpl_frame_get_filename(cpl_frameset_get_first(frames)));
    cpl_test_nonnull( pdt );
    model = sph_ifs_lenslet_model_new_from_propertylist(
            pl = cpl_propertylist_load(
                    cpl_frame_get_filename(cpl_frameset_get_first(frames)), 0));
    cpl_test_nonnull( model );
    cpl_propertylist_delete(pl);
    pl = NULL;

    idim = cpl_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(frames)),
            CPL_TYPE_INT, 0, 2);
    cpl_test_nonnull( idim );

    image = cpl_image_new(256, 256, CPL_TYPE_FLOAT);

    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, 0.0);
            if (cpl_image_get(idim, xx + 1, yy + 1, &rerr) > 0) {
                cpl_image_set(image, xx + 1, yy + 1,
                        xx * yy + 12.3 * (xx + yy));
            }
        }
    }

    /* run ...*/

    ldt = sph_ldt_new_from_pdt_image(pdt, model, image);

    cpl_test_nonnull( ldt );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( ldt->arr );
    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            if ((id = cpl_image_get(idim, xx + 1, yy + 1, &rerr)) > 0) {
                ldesc = ldt->arr[id - 1];
                cpl_test_nonnull(ldesc);
            }
        }
    }
    sph_ldt_delete(ldt);
    cpl_image_delete(idim);
    cpl_image_delete(image);
    cpl_frameset_delete(frames);
    sph_pixel_description_table_delete(pdt);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_new_from_pdt_master_frame(void) {
    sph_ldt* ldt = NULL;
    sph_pixel_description_table* pdt = NULL;
    int xx = 0;
    int yy = 0;
    int rerr = CPL_ERROR_NONE;
    cpl_frameset* frames = NULL;
    cpl_image* idim = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_lenslet_descriptor* ldesc = NULL;
    sph_master_frame* mf = NULL;
    int id = 0;
    int bad = 0;
    cpl_propertylist* pl = NULL;
    /* Setup */
    frames = cpl_frameset_new();

    sph_test_create_ifs_master_pdt_frame(frames);
    cpl_test_error(CPL_ERROR_NONE);

    pdt = sph_pixel_description_table_load(
            cpl_frame_get_filename(cpl_frameset_get_first(frames)));

    cpl_test_nonnull( pdt );

    model = sph_ifs_lenslet_model_new_from_propertylist(
            pl = cpl_propertylist_load(
                    cpl_frame_get_filename(cpl_frameset_get_first(frames)), 0));

    cpl_test_nonnull( model );
    cpl_propertylist_delete(pl);
    pl = NULL;

    idim = cpl_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(frames)),
            CPL_TYPE_INT, 0, 2);

    cpl_test_nonnull( idim );

    mf = sph_master_frame_new(256, 256);

    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            cpl_image_set(mf->image, xx + 1, yy + 1, 0.0);
            if (cpl_image_get(idim, xx + 1, yy + 1, &rerr) > 0) {
                cpl_image_set(mf->image, xx + 1, yy + 1,
                        xx * yy + 12.3 * (xx + yy));
            }
        }
    }

    /* run ...*/

    ldt = sph_ldt_new_from_pdt_master_frame(pdt, model, mf);

    cpl_test_nonnull( ldt );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( ldt->arr );
    bad = 0;
    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            if ((id = cpl_image_get(idim, xx + 1, yy + 1, &rerr)) > 0) {
                ldesc = ldt->arr[id - 1];
                if (!ldesc)
                    bad++;
            }
        }
    }

    /*Verify */
    cpl_test_lt(bad, 10);
    sph_ldt_delete(ldt);
    cpl_image_delete(idim);
    sph_master_frame_delete(mf);
    cpl_frameset_delete(frames);
    sph_pixel_description_table_delete(pdt);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_create_detframe(void) {
    sph_ldt* ldt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_master_frame* mframe = NULL;
    sph_master_frame* mf0 = NULL;
    sph_pixel_description_table* pdt = NULL;
    cpl_image* im = NULL;

    model = sph_test_create_small_lenslet_model();

    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);

    cpl_test_nonnull( pdt );
    im = sph_test_image_tools_create_flat_image_double(model->detsize_pixels,
            model->detsize_pixels, 0.0);
    cpl_image_set(im, 132, 126, 100.0);
    cpl_image_set(im, 131, 126, 100.0);
    cpl_image_set(im, 130, 126, 100.0);
    cpl_image_set(im, 129, 126, 100.0);
    //sph_test_image_tools_add_gauss(im,128.0,128.0,5.0,100.0);
    mf0 = sph_master_frame_new_from_cpl_image(im);

    ldt = sph_ldt_new_from_pdt_master_frame(pdt, model, mf0);
    cpl_test_nonnull( ldt );
    mframe = sph_ldt_create_detframe(ldt, pdt);
    cpl_test_nonnull( mframe );
    sph_master_frame_save(mframe, "det_ldt.fits", NULL);
    sph_master_frame_subtract_master_frame(mframe, mf0);
    sph_master_frame_save(mframe, "det_ldt_sub.fits", NULL);
    sph_master_frame_save(mf0, "det_ldt_zero.fits", NULL);
    sph_master_frame_delete(mframe);
    sph_ldt_delete(ldt);
    cpl_image_delete(im);
    sph_master_frame_delete(mf0);
    sph_pixel_description_table_delete(pdt);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ldt_delete(void) {
    sph_ldt* ldt = NULL;
    sph_ifs_lenslet_model* model = NULL;

    model = sph_test_create_small_lenslet_model();

    ldt = sph_ldt_new(model);
    cpl_test_nonnull( ldt );

    sph_ldt_delete(ldt);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    (void)sph_test_init();

    pSuite = sph_add_suite("ldt_test", cutest_init_ldt_testsuite,
            cutest_clean_ldt_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_ldt_new", cutest_ldt_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ldt_create_detframe",
                    cutest_ldt_create_detframe)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ldt_new_rotate", cutest_ldt_new_rotate)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_ldt_divide", cutest_ldt_divide)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_ldt_new_from_pdt_master_frame",
                    cutest_ldt_new_from_pdt_master_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ldt_new_from_pdt_image",
                    cutest_ldt_new_from_pdt_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ldt_create_image",
                    cutest_ldt_create_image)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_ldt_save", cutest_ldt_save)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_ldt_new_load", cutest_ldt_new_load)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_ldt_save_big", cutest_ldt_save_big)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ldt_save_big_cube",
                    cutest_ldt_save_big_cube)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_ldt_delete", cutest_ldt_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
