/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>


#include <cpl.h>

#include "sph_ldt.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_INFO);
    return sph_end_test();
}
static
void cutest_standard_happy_case(void) {
    sph_master_frame* mframe = NULL;
    int nx = 1024;
    int ny = 1024;
    sph_pixel_description_table* pdt = NULL;
    sph_ldt* ldt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    int ll = 0;
    cpl_vector* tick_off_list = NULL;
    cpl_vector* vals = NULL;
    cpl_vector* valsd = NULL;

    mframe = sph_master_frame_new_empty();
    model = sph_ifs_lenslet_model_new();
    model->detsize_pixels = nx;
    model->lenslets_per_side = 30;

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( mframe );

    mframe->image = sph_test_image_tools_create_flat_image_double(nx,ny,1.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->image );

    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(nx,ny,0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->badpixelmap );

    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(nx,ny,0.1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->rmsmap );

    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(nx,ny,2.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->ncombmap );

    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    sph_pixel_description_table_save(pdt, "sph_ldt_new_from_pdt_master_frame_pdt_debug.fits", model);
    sph_pixel_description_table_setup_regions(pdt);
    tick_off_list = cpl_vector_new(pdt->nregions);
    cpl_vector_fill(tick_off_list, 0.0);
    for (ll = 0; ll < pdt->nregions; ++ll) {
        cpl_vector_set(tick_off_list, pdt->regions[ll]->specid - 1, 1);
    }
    cpl_test_abs( cpl_vector_get_min(tick_off_list), 1,
                            0.0000000001);
    cpl_test_nonnull( pdt );
    cpl_test_error(CPL_ERROR_NONE);

    ldt = sph_ldt_new_from_pdt_master_frame(pdt, model, mframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( ldt );
    cpl_vector_delete(tick_off_list);
    tick_off_list = cpl_vector_new(sph_ifs_lenslet_model_get_nlens(model));
    cpl_vector_fill(tick_off_list, 0.0);
    for (ll = 0; ll < ldt->nlens; ++ll) {
        cpl_vector_set(tick_off_list, pdt->regions[ll]->lensid - 1, 1);
        valsd = cpl_vector_wrap(cpl_vector_get_size(ldt->arr[ll]->lambdas),
                ldt->arr[ll]->values);
        vals = cpl_vector_duplicate(valsd);
        cpl_vector_unwrap(valsd);
        valsd = NULL;

        if (sph_lenslet_descriptor_get_median(ldt->arr[ll], NULL) < 0.001) {
            SPH_ERROR_RAISE_ERR(
                    SPH_ERROR_GENERAL,
                    "The lenslet at %d- %d,%d had a median of %f", ll, ldt->arr[ll]->u, ldt->arr[ll]->v, sph_lenslet_descriptor_get_median(ldt->arr[ll],NULL));
        } else
            cpl_vector_subtract_scalar(vals,
                    sph_lenslet_descriptor_get_median(ldt->arr[ll], NULL));

        cpl_test_abs( cpl_vector_get_mean(vals), 0.0, 0.0001);
        cpl_test_abs( cpl_vector_get_min(vals), 0.0, 0.0001);
        cpl_test_abs( cpl_vector_get_max(vals), 0.0, 0.0001);
        cpl_vector_delete(vals);
        vals = NULL;
    }
    cpl_test_abs( cpl_vector_get_min(tick_off_list), 1,
                            0.0000000001);
    cpl_vector_delete(tick_off_list);
    cpl_test_error(CPL_ERROR_NONE);

    sph_ldt_save_cube(ldt, "sph_ldt_new_from_pdt_master_frame_ldt2.fits", NULL, NULL, NULL, NULL, NULL, NULL, NULL,
            0.00746,0.0,NULL);

    sph_master_frame_delete(mframe);
    mframe = NULL;
    mframe = sph_ldt_collapse(ldt, 0, 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe );
    sph_master_frame_save(mframe, "sph_ldt_new_from_pdt_master_frame_mframe_nonoise.fits", NULL);
    SPH_RAISE_CPL_RESET;cpl_test_error(CPL_ERROR_NONE);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    sph_ldt_delete(ldt);
    ldt = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    cpl_test_error(CPL_ERROR_NONE);
}
static
void cutest_standard_happy_case_but_noisy(void) {
    sph_master_frame* mframe = NULL;
    int nx = 1024;
    int ny = 1024;
    int ll = 0;
    int plane = 0;
    sph_pixel_description_table* pdt = NULL;
    sph_ldt* ldt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    gsl_rng* pRNG = NULL;
    cpl_array* arr = NULL;
    cpl_array* rmsarr = NULL;
    cpl_array* means = NULL;
    cpl_array* rmss = NULL;
    sph_polygon* spec_region_poly = NULL;
    double spec_region_area = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_error(CPL_ERROR_NONE);
    mframe = sph_master_frame_new_empty();
    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ifs_lenslet_model_new();
    cpl_test_error(CPL_ERROR_NONE);
    model->detsize_pixels = nx;
    model->lenslets_per_side = 30;

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe );

    mframe->image = sph_test_image_tools_create_flat_image_double(nx,ny,51.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->image );

    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(nx,ny,0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->badpixelmap );

    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(nx,ny,0.1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->rmsmap );

    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(nx,ny,2.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( mframe->ncombmap );

    sph_test_image_tools_apply_poisson_noise(mframe->image, pRNG);

    cpl_test_error(CPL_ERROR_NONE);
    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "sph_ldt_new_from_pdt_master_frame_pdt11.fits", model);
    cpl_test_error(CPL_ERROR_NONE);

    ldt = sph_ldt_new_from_pdt_master_frame(pdt, model, mframe);

    cpl_test_nonnull( ldt );

    spec_region_poly = sph_ifs_lenslet_model_get_spec_region(model, 0, 0);
    cpl_test_nonnull( spec_region_poly );
    spec_region_area = sph_polygon_area(spec_region_poly);
    cpl_test_lt(0.0, spec_region_area);
    means = cpl_array_new(ldt->nlens, CPL_TYPE_DOUBLE);
    rmss = cpl_array_new(ldt->nlens, CPL_TYPE_DOUBLE);
    for (ll = 0; ll < ldt->nlens; ++ll) {
        if (ldt->arr[ll]) {
            arr = cpl_array_new(cpl_vector_get_size(ldt->reflambdas),
                    CPL_TYPE_DOUBLE);
            rmsarr = cpl_array_new(cpl_vector_get_size(ldt->reflambdas),
                    CPL_TYPE_DOUBLE);
            for (plane = 0; plane < cpl_vector_get_size(ldt->reflambdas);
                    ++plane) {
                if (ldt->arr[ll]->bpix[plane] == 0) {
                    cpl_array_set(arr, plane, ldt->arr[ll]->values[plane]);
                    cpl_array_set(rmsarr, plane, ldt->arr[ll]->rms[plane]);
                } else {
                    cpl_array_set_invalid(arr, plane);
                    cpl_array_set_invalid(rmsarr, plane);
                }
            }
            cpl_array_set(means, ll, cpl_array_get_mean(arr));
            cpl_array_set(rmss, ll, cpl_array_get_mean(rmsarr));
            cpl_array_delete(arr);
            arr = NULL;
            cpl_array_delete(rmsarr);
            rmsarr = NULL;
        }
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "The lenslet mean mean values were:"
    " %4.2f+/-%4.2f", cpl_array_get_mean(means), cpl_array_get_stdev(means));
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "The lenslet mean RMS values were:"
    " %4.2f+/-%4.2f", cpl_array_get_mean(rmss), cpl_array_get_stdev(rmss));

    cpl_array_delete(means);
    means = NULL;
    cpl_array_delete(rmss);
    rmss = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    mframe = sph_ldt_collapse(ldt, 0, -1.0);
    cpl_test_nonnull( mframe );
    sph_master_frame_save(mframe, "standard_happy_case_but_noisy_mframe.fits", NULL);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ldt_delete(ldt);
    ldt = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    gsl_rng_free(pRNG);
    pRNG = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("ldt_new_from_pdt_master_frame",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "standard happy case",
                    cutest_standard_happy_case)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "standard happy case noisy",
                    cutest_standard_happy_case_but_noisy)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
