/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_master_frame.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

#include <cpl.h>
#include <hdrl.h>

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <gsl/gsl_rng.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_master_frame"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_master_frame Unit test for the sph_master_frame module            
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_sph_master_frame_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_sph_master_frame_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_master_frame_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_master_frame_inverse(void) {
    cpl_image* im = NULL;
    sph_master_frame* mframe = NULL;
    sph_master_frame* result = NULL;
    cpl_image* resim = NULL;

    im = sph_test_image_tools_create_flat_image_double(256, 256, 2.0);
    mframe = sph_master_frame_new_from_cpl_image(im);
    result = sph_master_frame_inverse(mframe);
    resim = sph_master_frame_extract_image(result, 1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( resim );
    cpl_test_abs(cpl_image_get_max(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_min(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_mean(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_median(resim), 0.5, 0.0001);
    cpl_image_delete(resim);
    resim = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    cpl_image_set(mframe->image, 100, 100, 0.0);
    result = sph_master_frame_inverse(mframe);
    resim = sph_master_frame_extract_image(result, 1);
    cpl_test_nonnull( resim );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(cpl_image_get_max(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_min(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_mean(resim), 0.5, 0.0001);
    cpl_test_abs(cpl_image_get_median(resim), 0.5, 0.0001);
    cpl_test_eq(cpl_mask_count(cpl_image_get_bpm_const(resim)), 1);
    cpl_image_delete(resim);
    resim = NULL;
    sph_master_frame_delete(result);
    result = NULL;

    cpl_image_delete(im);
    sph_master_frame_delete(mframe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_master_frame_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_master_frame_interpolate_bpix(void) {
    const int nx  = 512;
    const int ny  = 128;
    const int nxy = nx * ny;
    cpl_image* im = NULL;
    sph_master_frame* mframe = NULL;
    gsl_rng* pRNG = NULL;
    cpl_mask * full;
    cpl_error_code code;
    int status;
    const int mbads = 100;
    int nbads;

    cpl_test_error(CPL_ERROR_NONE);
    pRNG = gsl_rng_alloc(gsl_rng_taus);

    im = sph_test_image_tools_create_flat_image_double(nx, ny, 50.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(im);

    full = cpl_mask_new(nx, ny);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(im);

    code = cpl_mask_not(full);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    status = sph_test_image_tools_apply_poisson_noise(im, pRNG);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(status);

    status = sph_test_image_tools_add_hotpixels(im, mbads, 100000.0, pRNG);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(status);

    mframe = sph_master_frame_new_from_cpl_image(im);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(mframe);

    code = sph_master_frame_quality_check(mframe);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = sph_master_frame_mask_sigma(mframe, 5.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nbads = sph_master_frame_get_nbads(mframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nbads, mbads);

    code = sph_master_frame_interpolate_bpix(mframe);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nbads = sph_master_frame_get_nbads(mframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(nbads);

    /* Test with no bad pixels */
    code = sph_master_frame_interpolate_bpix(mframe);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nbads = sph_master_frame_get_nbads(mframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(nbads);

    /* Test with all bad pixels */
    code = sph_master_frame_set_bads_from_mask( mframe, full);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    nbads = sph_master_frame_get_nbads(mframe);
    cpl_test_eq(nbads, nxy);

    code = sph_master_frame_interpolate_bpix(mframe);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    nbads = sph_master_frame_get_nbads(mframe);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(nbads, nxy);

    sph_master_frame_delete(mframe);
    cpl_image_delete(im);
    cpl_mask_delete(full);
    gsl_rng_free(pRNG);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_master_frame_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_master_frame_smooth(void) {
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    sph_master_frame* mframe = NULL;
    double flux0 = 0.0;
    double flux = 0.0;
    cpl_error_code code;

    cpl_test_error(CPL_ERROR_NONE);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    sph_test_image_tools_add_gauss(im, 128.0, 128.0, 1.0, 1.0);
    sph_test_image_tools_add_gauss(im, 198.0, 198.0, 1.0, 1.0);
    flux0 = cpl_image_get_absflux(im);
    im2 = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    sph_test_image_tools_add_gauss(im2, 128.0, 128.0, 15.0 * CPL_MATH_SIG_FWHM, 1.0);
    sph_test_image_tools_add_gauss(im2, 198.0, 198.0, 15.0 * CPL_MATH_SIG_FWHM, 1.0);
    cpl_image_multiply_scalar(im2, flux0 / cpl_image_get_absflux(im2));
    mframe = sph_master_frame_new_from_cpl_image(im);
    cpl_test_zero(sph_master_frame_get_nbads(mframe));
    sph_master_frame_smooth(mframe, 15.0);
    cpl_test_zero(sph_master_frame_get_nbads(mframe));
    flux = cpl_image_get_absflux(mframe->image);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(flux0, flux, 0.000001);
    cpl_test_abs(flux0, cpl_image_get_flux(im2), 0.000001);
    code = sph_master_frame_save(mframe, SPH_BASE "_smoothed.fits", NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_smoothed.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(im2, mframe->image);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_smoothed.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(cpl_image_get_absflux(im2), 0.0, 0.1);
    cpl_test_abs(cpl_image_get_max(im2), 0.0, 0.0001);
    cpl_test_abs(cpl_image_get_min(im2), 0.0, 0.0001);
    cpl_test_zero(unlink(SPH_BASE "_smoothed.fits"));
    cpl_image_delete(im);
    sph_master_frame_delete(mframe);
    cpl_image_delete(im2);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the scalar operations in sph_master_frame.
 */
/*----------------------------------------------------------------------------*/

#define get_vect_size(vect) sizeof(vect) / sizeof(vect[0])

static void check_mul_div_scalar(void){

    const int x_sz = 2;
    const int y_sz = 3;

    sph_master_frame * frame = sph_master_frame_new(x_sz, y_sz);

    double vals[] = {0,    -2,   2,   4,  5,   8};
    double rms[]  = {0.1, 0.0, 0.2, 0.4, .3, 0.0};
    int badpxs[]  = {0,     0,   1,   2,  0,   0}; /*2 rejected pixels*/
    double cmbs[] = {2,     4,   5,   6,  7,   8};

    cpl_test_eq(cpl_image_get_size_x(frame->image), x_sz);
    cpl_test_eq(cpl_image_get_size_y(frame->image), y_sz);

    cpl_test_eq(get_vect_size(vals), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms), x_sz * y_sz);
    cpl_test_eq(get_vect_size(badpxs), x_sz * y_sz);
    cpl_test_eq(get_vect_size(cmbs), x_sz * y_sz);

    cpl_size idx = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){
            cpl_image_set(frame->image, x, y, vals[idx]);
            cpl_image_set(frame->rmsmap, x, y, rms[idx]);
            cpl_image_set(frame->badpixelmap, x, y, badpxs[idx]);
            cpl_image_set(frame->ncombmap, x, y, cmbs[idx]);
            idx++;
        }
    }

    sph_master_frame * mul_frame = sph_master_frame_duplicate(frame);
    const double mul_coeff = -2.0;
    sph_master_frame_multiply_double(mul_frame, mul_coeff);


    idx = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej = 0;

            cpl_test_eq(cpl_image_get(mul_frame->badpixelmap, x, y, &rej), badpxs[idx]);

            if(!badpxs[idx]){
                cpl_test_eq(cpl_image_get(mul_frame->ncombmap, x, y, &rej), cmbs[idx]);
                cpl_test_eq(cpl_image_get(mul_frame->image, x, y, &rej), mul_coeff * vals[idx]);
                cpl_test_rel(cpl_image_get(mul_frame->rmsmap, x, y, &rej),
                        (rms[idx] * fabs(mul_coeff)), 1e-10);
            }

            idx++;
        }
    }

    cpl_image * im_cpl = sph_master_frame_extract_image(frame, 1);
    hdrl_image * hdrl_im = hdrl_image_create(im_cpl, frame->rmsmap);
    hdrl_image * hdrl_im_mul = hdrl_image_duplicate(hdrl_im);
    hdrl_image_mul_scalar(hdrl_im_mul, (hdrl_value){-2.0, 0.0});
    cpl_image_delete(im_cpl);

    int n_els = 0;
    int n_els_non_zero = 0;

    int rej = 0;

    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej_hdrl = 0;
            hdrl_value pix = hdrl_image_get_pixel(hdrl_im_mul, x, y, &rej_hdrl);

            int rej_sph = cpl_image_get(mul_frame->badpixelmap, x, y, &rej) > 0.5 ? 1 : 0;

            cpl_test_eq(rej_hdrl, rej_sph);

            if(rej_hdrl == 1) continue;

            cpl_test_rel(pix.error, cpl_image_get(mul_frame->rmsmap, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            cpl_test_rel(pix.data, cpl_image_get(mul_frame->image, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            n_els++;

            if(cpl_image_get(mul_frame->rmsmap, x, y, &rej) != 0 &&
                    cpl_image_get(mul_frame->image, x, y, &rej) != 0){
                n_els_non_zero ++;
            }
        }
    }

    cpl_test(n_els > 0);
    cpl_test(n_els_non_zero > 0);

    hdrl_image_delete(hdrl_im_mul);
    sph_master_frame_delete(mul_frame);

    sph_master_frame * div_frame = sph_master_frame_duplicate(frame);
    const double div_coeff = -4.0;
    sph_master_frame_divide_double(div_frame, div_coeff);

    idx = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej = 0;

            cpl_test_eq(cpl_image_get(div_frame->badpixelmap, x, y, &rej), badpxs[idx]);

            if(!badpxs[idx]){
                cpl_test_eq(cpl_image_get(div_frame->ncombmap, x, y, &rej), cmbs[idx]);
                cpl_test_eq(cpl_image_get(div_frame->image, x, y, &rej),  vals[idx] / div_coeff);
                cpl_test_rel(cpl_image_get(div_frame->rmsmap, x, y, &rej),
                        (rms[idx] / fabs(div_coeff)), 1e-10);
            }

            idx++;
        }
    }

    hdrl_image * hdrl_im_div = hdrl_image_duplicate(hdrl_im);
    hdrl_image_div_scalar(hdrl_im_div, (hdrl_value){-4.0, 0.0});

    n_els = 0;
    n_els_non_zero = 0;
    rej = 0;

    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej_hdrl = 0;
            hdrl_value pix = hdrl_image_get_pixel(hdrl_im_div, x, y, &rej_hdrl);

            int rej_sph = cpl_image_get(div_frame->badpixelmap, x, y, &rej) > 0.5 ? 1 : 0;

            cpl_test_eq(rej_hdrl, rej_sph);

            if(rej_hdrl == 1) continue;

            cpl_test_rel(pix.error, cpl_image_get(div_frame->rmsmap, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            cpl_test_rel(pix.data, cpl_image_get(div_frame->image, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            n_els++;

            if(cpl_image_get(div_frame->rmsmap, x, y, &rej) != 0 &&
                    cpl_image_get(div_frame->image, x, y, &rej) != 0){
                n_els_non_zero ++;
            }
        }
    }

    cpl_test(n_els > 0);
    cpl_test(n_els_non_zero > 0);

    hdrl_image_delete(hdrl_im_div);

    sph_master_frame_divide_double(div_frame, 0.0);
    cpl_test_error(CPL_ERROR_DIVISION_BY_ZERO);

    sph_master_frame_delete(div_frame);

    sph_master_frame_delete(frame);
    hdrl_image_delete(hdrl_im);
}

static void check_inversion(void){

    const int x_sz = 2;
    const int y_sz = 3;

    sph_master_frame * frame = sph_master_frame_new(x_sz, y_sz);

    double vals[] = {0,    -2,   2,   4,  5,   8};
    double rms[]  = {0.1, 0.0, 0.2, 0.4, .3, 0.0};
    int badpxs[]  = {0,     0,   1,   2,  0,   0}; /*2 rejected pixels*/
    double cmbs[] = {2,     4,   5,   6,  7,   8};

    cpl_test_eq(cpl_image_get_size_x(frame->image), x_sz);
    cpl_test_eq(cpl_image_get_size_y(frame->image), y_sz);

    cpl_test_eq(get_vect_size(vals), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms), x_sz * y_sz);
    cpl_test_eq(get_vect_size(badpxs), x_sz * y_sz);
    cpl_test_eq(get_vect_size(cmbs), x_sz * y_sz);

    cpl_size idx = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){
            cpl_image_set(frame->image, x, y, vals[idx]);
            cpl_image_set(frame->rmsmap, x, y, rms[idx]);
            cpl_image_set(frame->badpixelmap, x, y, badpxs[idx]);
            cpl_image_set(frame->ncombmap, x, y, cmbs[idx]);
            idx++;
        }
    }

    sph_master_frame * inversion_frame = sph_master_frame_inverse(frame);

    int rej = 0;
    cpl_test(cpl_image_get(inversion_frame->badpixelmap, 1, 1, &rej) > 0.5);
    cpl_test_eq(cpl_image_get(inversion_frame->image, 1, 1, &rej), 0.0);


    idx = 1;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            if(x == 1 && y == 1) continue;
            int expected = badpxs[idx] ? 1 : 0;
            cpl_test_eq(cpl_image_get(inversion_frame->badpixelmap, x, y, &rej), expected);

            if(!badpxs[idx]){
                cpl_test_eq(cpl_image_get(inversion_frame->ncombmap, x, y, &rej), cmbs[idx]);
                cpl_test_rel(cpl_image_get(inversion_frame->image, x, y, &rej),
                             1.0 / vals[idx], DBL_EPSILON);
                cpl_test_rel(cpl_image_get(inversion_frame->rmsmap, x, y, &rej),
                        (rms[idx] / (vals[idx] * vals[idx])), 1e-10);
            }

            idx++;
        }
    }

    hdrl_image * numerator = hdrl_image_new(x_sz, y_sz);
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){
            hdrl_image_set_pixel(numerator, x, y, (hdrl_value){1.0, 0.0});
        }
    }

    cpl_image * cpl_im = sph_master_frame_extract_image(frame, 1);
    hdrl_image * denominator = hdrl_image_create(cpl_im, frame->rmsmap);
    cpl_image_delete(cpl_im);

    hdrl_image * hdrl_image_inverse = hdrl_image_div_image_create(numerator, denominator);

    int n_els = 0;
    int n_els_non_zero = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej_hdrl = 0;
            hdrl_value pix = hdrl_image_get_pixel(hdrl_image_inverse, x, y, &rej_hdrl);

            int rej_sph = cpl_image_get(inversion_frame->badpixelmap, x, y, &rej) > 0.5 ? 1 : 0;

            cpl_test_eq(rej_hdrl, rej_sph);

            if(rej_hdrl == 1) continue;

            cpl_test_rel(pix.error, cpl_image_get(inversion_frame->rmsmap, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            cpl_test_rel(pix.data, cpl_image_get(inversion_frame->image, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            n_els++;

            if(cpl_image_get(inversion_frame->rmsmap, x, y, &rej) != 0 &&
                    cpl_image_get(inversion_frame->image, x, y, &rej) != 0){
                n_els_non_zero ++;
            }
        }
    }

    cpl_test(n_els > 0);
    cpl_test(n_els_non_zero > 0);

    hdrl_image_delete(hdrl_image_inverse);
    hdrl_image_delete(numerator);
    hdrl_image_delete(denominator);

    sph_master_frame_delete(frame);
    sph_master_frame_delete(inversion_frame);
}

static void check_sqrt_log10(void){

    const int x_sz = 2;
    const int y_sz = 3;

    sph_master_frame * frame = sph_master_frame_new(x_sz, y_sz);

    double vals[] = {-2,    1,   2,   4,  5,   8};
    double rms[]  = {0.1, 0.0, 0.2, 0.4, .3, 0.0};
    int badpxs[]  = {0,     0,   1,   2,  0,   0}; /*2 rejected pixels*/
    double cmbs[] = {2,     4,   5,   6,  7,   8};
    cpl_size idx = 0;

    cpl_test_eq(cpl_image_get_size_x(frame->image), x_sz);
    cpl_test_eq(cpl_image_get_size_y(frame->image), y_sz);

    cpl_test_eq(get_vect_size(vals), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms), x_sz * y_sz);
    cpl_test_eq(get_vect_size(badpxs), x_sz * y_sz);
    cpl_test_eq(get_vect_size(cmbs), x_sz * y_sz);

    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){
            cpl_image_set(frame->image, x, y, vals[idx]);
            cpl_image_set(frame->rmsmap, x, y, rms[idx]);
            cpl_image_set(frame->badpixelmap, x, y, badpxs[idx]);
            cpl_image_set(frame->ncombmap, x, y, cmbs[idx]);
            idx++;
        }
    }
    /*---------------- Log10 TESTS ----------------*/
    {
        sph_master_frame * log10_frame = sph_master_frame_duplicate(frame);
        cpl_test_nonnull(log10_frame);

        sph_master_frame_take_log10(log10_frame);
        cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

        cpl_test_nonnull(frame->image);
        int rej = 0;
        cpl_test(cpl_image_get(frame->image, 1, 1, &rej) < 0.0);

        rej = 0;
        cpl_test_nonnull(log10_frame->badpixelmap);
        cpl_test(cpl_image_get(log10_frame->badpixelmap, 1, 1, &rej) > 0.5);
        cpl_test_eq(cpl_image_get(log10_frame->image, 1, 1, &rej), 0.0);

        idx = 1;
        for(cpl_size x = 1; x <= x_sz; ++x){
            for(cpl_size y = 1; y <= y_sz; ++y){
                if(x == 1 && y == 1) continue;

                cpl_test_eq(cpl_image_get(log10_frame->badpixelmap, x, y, &rej), badpxs[idx]);
                cpl_test_eq(cpl_image_get(log10_frame->ncombmap, x, y, &rej), cmbs[idx]);

                if(!badpxs[idx]){
                    cpl_test_rel(cpl_image_get(log10_frame->image, x, y, &rej),
                                 log10(vals[idx]), DBL_EPSILON);
                    cpl_test_rel(cpl_image_get(log10_frame->rmsmap, x, y, &rej),
                                 (rms[idx] / (vals[idx] * log(10.0))), 1e-10);
                }

                idx++;
            }
        }
        sph_master_frame_delete(log10_frame);
    }

    /*---------------- SQRT TESTS ----------------*/
    {
        sph_master_frame * sqrt_frame = sph_master_frame_duplicate(frame);
        cpl_test_nonnull(sqrt_frame);

        sph_master_frame_sqrt(sqrt_frame);
        cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

        cpl_test_nonnull(frame->image);
        int rej = 0;
        cpl_test(cpl_image_get(frame->image, 1, 1, &rej) < 0.0);

        rej = 0;
        cpl_test_nonnull(sqrt_frame->badpixelmap);
        cpl_test(cpl_image_get(sqrt_frame->badpixelmap, 1, 1, &rej) > 0.5);
        cpl_test_eq(cpl_image_get(sqrt_frame->image, 1, 1, &rej), 0.0);

        idx = 1;
        for(cpl_size x = 1; x <= x_sz; ++x){
            for(cpl_size y = 1; y <= y_sz; ++y){
                //first element is negative, should become bad pixel
                if(x == 1 && y == 1) continue;

                cpl_test_eq(cpl_image_get(sqrt_frame->badpixelmap, x, y, &rej), badpxs[idx]);
                cpl_test_eq(cpl_image_get(sqrt_frame->ncombmap, x, y, &rej), cmbs[idx]);

                if(!badpxs[idx]){
                    cpl_test_eq(cpl_image_get(sqrt_frame->image, x, y, &rej), sqrt(vals[idx]));
                    cpl_test_rel(cpl_image_get(sqrt_frame->rmsmap, x, y, &rej),
                            (rms[idx] / (sqrt(vals[idx]) * 2.0)), 1e-10);
                }

                idx++;
            }
        }

        cpl_image * cpl_im = sph_master_frame_extract_image(frame, 1);
        hdrl_image * h_im = hdrl_image_create(cpl_im, frame->rmsmap);
        cpl_image_delete(cpl_im);

        hdrl_image_pow_scalar(h_im, (hdrl_value){.5, 0.0});
        int n_els = 0;
        for(cpl_size x = 1; x <= x_sz; ++x){
            for(cpl_size y = 1; y <= y_sz; ++y){

                int rej_hdrl = 0;
                hdrl_value pix = hdrl_image_get_pixel(h_im, x, y, &rej_hdrl);

                int rej_sph = cpl_image_get(sqrt_frame->badpixelmap, x, y, &rej) > 0.5 ? 1 : 0;

                cpl_test_eq(rej_hdrl, rej_sph);

                if(rej_hdrl == 1) continue;

                cpl_test_rel(pix.error, cpl_image_get(sqrt_frame->rmsmap, x, y, &rej), 1e-10);
                cpl_test_zero(rej);
                cpl_test_rel(pix.data, cpl_image_get(sqrt_frame->image, x, y, &rej), 1e-10);
                cpl_test_zero(rej);
                n_els++;
            }
        }

        cpl_test(n_els > 0);

        hdrl_image_delete(h_im);
        sph_master_frame_delete(sqrt_frame);
    }

    sph_master_frame_delete(frame);
}

static void cutest_sph_master_frame_scalar_operation(void){

    check_sqrt_log10();
    check_inversion();
    check_mul_div_scalar();
}

static
sph_master_frame * gen_master_frame(const double * vals, const double * rms, const int * badpxs,
        const double * cmbs, const cpl_size x_sz, const cpl_size y_sz){

    sph_master_frame * frame = sph_master_frame_new(x_sz, y_sz);

    cpl_size idx = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){
            cpl_image_set(frame->image, x, y, vals[idx]);
            cpl_image_set(frame->rmsmap, x, y, rms[idx]);
            cpl_image_set(frame->badpixelmap, x, y, badpxs[idx]);
            cpl_image_set(frame->ncombmap, x, y, cmbs[idx]);
            idx++;
        }
    }
    return frame;
}

static cpl_boolean sph_master_frame_are_equal(const sph_master_frame * a, const sph_master_frame * b){
    int rej = 0;
    for(cpl_size x = 1; x <= cpl_image_get_size_x(a->image); ++x){
        for(cpl_size y = 1; y <= cpl_image_get_size_y(a->image); ++y){
            if(cpl_image_get(a->image, x, y, &rej) != cpl_image_get(b->image, x, y, &rej)) return CPL_FALSE;
            if(cpl_image_get(a->rmsmap, x, y, &rej) != cpl_image_get(b->rmsmap, x, y, &rej)) return CPL_FALSE;
            if(cpl_image_get(a->badpixelmap, x, y, &rej) != cpl_image_get(b->badpixelmap, x, y, &rej)) return CPL_FALSE;
            if(cpl_image_get(a->ncombmap, x, y, &rej) != cpl_image_get(b->ncombmap, x, y, &rej)) return CPL_FALSE;
        }
    }
    return CPL_TRUE;
}

typedef double (*P_Operator)(double a, double b);

static cpl_boolean is_bad(const double a){

    if(isnan(a)) return CPL_TRUE;
    if(isinf(a)) return CPL_TRUE;

    return CPL_FALSE;
}

static cpl_boolean sph_master_frame_check_mul_div(const sph_master_frame * res,
        const sph_master_frame * a, const sph_master_frame * b, P_Operator op){

    int rej = 0;

    for(cpl_size x = 1; x <= cpl_image_get_size_x(a->image); ++x){
        for(cpl_size y = 1; y <= cpl_image_get_size_y(a->image); ++y){

            const double b_a = cpl_image_get(a->badpixelmap, x, y, &rej);
            const double b_b = cpl_image_get(b->badpixelmap, x, y, &rej);
            const int expected_b = (b_a + b_b) > 0.5;
            const int obtained_b = cpl_image_get(res->badpixelmap, x, y, &rej) > 0.5;

            if(expected_b != obtained_b){
                    //if a new rejected appears it means that one of the images contain 0
                    cpl_boolean is_error_invalid = cpl_image_get(a->image, x, y, &rej) == 0.0 ||
                            cpl_image_get(b->image, x, y, &rej) == 0.0;
                    if(!is_error_invalid)
                        return CPL_FALSE;
            }

            const double c_a = cpl_image_get(a->ncombmap, x, y, &rej);
            const double c_b = cpl_image_get(b->ncombmap, x, y, &rej);
            const double expected_c = c_a + c_b;

            if(expected_c != cpl_image_get(res->ncombmap, x, y, &rej)){
                    return CPL_FALSE;
            }

            const double v_a = cpl_image_get(a->image, x, y, &rej);
            const double v_b = cpl_image_get(b->image, x, y, &rej);
            const double expected = op(v_a, v_b);

            if(is_bad(expected) && cpl_image_get(res->badpixelmap, x, y, &rej) > 0.5) continue;

            if(expected != cpl_image_get(res->image, x, y, &rej)) return CPL_FALSE;

            const double r_a = cpl_image_get(a->rmsmap, x, y, &rej);
            const double r_b = cpl_image_get(b->rmsmap, x, y, &rej);
            const double expected_r = hypot(r_a / v_a, r_b / v_b);

            if(is_bad(expected_r) && cpl_image_get(res->badpixelmap, x, y, &rej) > 0.5) continue;

            if(fabs(expected_r - cpl_image_get(res->rmsmap, x, y, &rej) / expected_r) < 1e-10) {
                return CPL_FALSE;
            }
        }
    }
    return CPL_TRUE;

}


static cpl_boolean sph_master_frame_check_sum_sub(const sph_master_frame * res,
        const sph_master_frame * a, const sph_master_frame * b, P_Operator op){

    int rej = 0;
    for(cpl_size x = 1; x <= cpl_image_get_size_x(a->image); ++x){
        for(cpl_size y = 1; y <= cpl_image_get_size_y(a->image); ++y){

            const double v_a = cpl_image_get(a->image, x, y, &rej);
            const double v_b = cpl_image_get(b->image, x, y, &rej);
            const double expected = op(v_a, v_b);

            if(expected != cpl_image_get(res->image, x, y, &rej)) return CPL_FALSE;

            const double r_a = cpl_image_get(a->rmsmap, x, y, &rej);
            const double r_b = cpl_image_get(b->rmsmap, x, y, &rej);
            const double expected_r = sqrt(r_a * r_a + r_b * r_b);

            if(fabs(expected_r - cpl_image_get(res->rmsmap, x, y, &rej) / expected_r) < 1e-10) {
                return CPL_FALSE;
            }

            const double b_a = cpl_image_get(a->badpixelmap, x, y, &rej);
            const double b_b = cpl_image_get(b->badpixelmap, x, y, &rej);
            const int expected_b = (b_a + b_b) > 0.5;
            const int obtained_b = cpl_image_get(res->badpixelmap, x, y, &rej) > 0.5;

            if(expected_b != obtained_b){
                    return CPL_FALSE;
            }

            const double c_a = cpl_image_get(a->ncombmap, x, y, &rej);
            const double c_b = cpl_image_get(b->ncombmap, x, y, &rej);
            const double expected_c = c_a + c_b;

            if(expected_c != cpl_image_get(res->ncombmap, x, y, &rej)){
                    return CPL_FALSE;
            }

        }
    }
    return CPL_TRUE;
}

static hdrl_image *
sph_master_frame_to_hdrl_image(const sph_master_frame * f){
    cpl_image * cpl_im = sph_master_frame_extract_image(f, 1);
    hdrl_image * to_ret = hdrl_image_create(cpl_im, f->rmsmap);
    cpl_image_delete(cpl_im);
    return to_ret;
}

typedef cpl_error_code (*hdrl_op)(hdrl_image * im1, const hdrl_image * im2);

static hdrl_image *
convert_and_operate_on_hdrl(const sph_master_frame * f1, const sph_master_frame * f2, hdrl_op op,
        const cpl_boolean reject_if_im_is_zero){
    hdrl_image * im1 = sph_master_frame_to_hdrl_image(f1);
    hdrl_image * im2 = sph_master_frame_to_hdrl_image(f2);
    hdrl_image * to_ret = hdrl_image_duplicate(im1);
    op(to_ret, im2);

    /*if img1(x,y) == 0 or img2(x,y) == 0 the error is not valid but the corresponding pixel
     * value might be valid (see PIPE-7697). HDRL should reject the pixels but the current version does
     * not, hence we have to hack our way around this limitation. This for loop should be removed once PIPE-7697 is
     * addressed.
    */
    if(reject_if_im_is_zero){
        int rej = 0;
        for(cpl_size x = 1; x <= hdrl_image_get_size_x(im1); ++x){
            for(cpl_size y = 1; y <= hdrl_image_get_size_y(im1); ++y){
                if(hdrl_image_get_pixel(im1, x, y, &rej).data == 0 || hdrl_image_get_pixel(im2, x, y, &rej).data == 0){
                    hdrl_image_reject(to_ret, x, y);
                }
            }
        }
    }

    hdrl_image_delete(im2);
    hdrl_image_delete(im1);
    return to_ret;
}

static void
check_equality_master_hdrl_img(const sph_master_frame * frame, const hdrl_image * h){

    const int x_sz = cpl_image_get_size_x(frame->image);
    const int y_sz = cpl_image_get_size_y(frame->image);

    int n_els = 0;
    int n_els_non_zero = 0;

    int rej = 0;
    for(cpl_size x = 1; x <= x_sz; ++x){
        for(cpl_size y = 1; y <= y_sz; ++y){

            int rej_hdrl = 0;
            hdrl_value pix = hdrl_image_get_pixel(h, x, y, &rej_hdrl);

            int rej_sph = cpl_image_get(frame->badpixelmap, x, y, &rej) > 0.5 ? 1 : 0;

            cpl_test_eq(rej_hdrl, rej_sph);

            if(rej_hdrl == 1 || rej_sph == 1) continue;

            cpl_test_rel(pix.data, cpl_image_get(frame->image, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            cpl_test_rel(pix.error, cpl_image_get(frame->rmsmap, x, y, &rej), 1e-10);
            cpl_test_zero(rej);
            n_els++;

            if(cpl_image_get(frame->rmsmap, x, y, &rej) != 0 &&
                    cpl_image_get(frame->image, x, y, &rej) != 0){
                n_els_non_zero ++;
            }
        }
    }

    cpl_test(n_els > 0);
    cpl_test(n_els_non_zero > 0);

}

static double op_minus(const double a, const double b){return a - b;}
static double op_plus(const double a, const double b){return a + b;}

static void check_vectorial_sum_sub(void){

    const int x_sz = 2;
    const int y_sz = 3;

    double vals1[] = {-2,    0,   2,   4,  5,   8};
    double vals2[] = {3,    -4,   1,   5,  7,   0};

    double rms1[]  = {0.1, 0.0, 0.2, 0.4, .3, 0.0};
    double rms2[]  = {0.3, 0.4, 0.1, 0.3, .4, 0.8};

    int badpxs1[]  = {0,     0,   1,   2,  0,   0}; /*2 rejected pixels*/
    int badpxs2[]  = {1,     0,   0,   1,  0,   0}; /*2 rejected pixels*/

    double cmbs1[] = {0,     4,   3,   1,  0,   2};
    double cmbs2[] = {2,     2,   5,   6,  7,   8};

    sph_master_frame * frame1 = gen_master_frame(vals1, rms1, badpxs1, cmbs1, x_sz, y_sz);
    sph_master_frame * frame2 = gen_master_frame(vals2, rms2, badpxs2, cmbs2, x_sz, y_sz);

    sph_master_frame * frame1_ori = sph_master_frame_duplicate(frame1);
    sph_master_frame * frame2_ori = sph_master_frame_duplicate(frame2);

    sph_master_frame * sub_create = sph_master_frame_subtract_master_frame_create(frame1, frame2);

    hdrl_image * sub_hdrl = convert_and_operate_on_hdrl(frame1, frame2, hdrl_image_sub_image, CPL_FALSE);
    hdrl_image * add_hdrl = convert_and_operate_on_hdrl(frame1, frame2, hdrl_image_add_image, CPL_FALSE);

    cpl_test_eq(get_vect_size(vals1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms1), x_sz * y_sz);

    cpl_test_eq(get_vect_size(vals2), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms2), x_sz * y_sz);

    cpl_test_eq(get_vect_size(badpxs1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(badpxs2), x_sz * y_sz);

    cpl_test_eq(get_vect_size(cmbs1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(cmbs2), x_sz * y_sz);

    cpl_test(sph_master_frame_are_equal(frame1, frame1_ori));
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_sum_sub(sub_create, frame1, frame2, &op_minus));
    check_equality_master_hdrl_img(sub_create, sub_hdrl);
    sph_master_frame_delete(sub_create);

    sph_master_frame * sum_create = sph_master_frame_add_master_frame_create(frame1, frame2);
    cpl_test(sph_master_frame_are_equal(frame1, frame1_ori));
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_sum_sub(sum_create, frame1, frame2, &op_plus));
    check_equality_master_hdrl_img(sum_create, add_hdrl);
    sph_master_frame_delete(sum_create);

    sph_master_frame_subtract_master_frame(frame1, frame2);
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_sum_sub(frame1, frame1_ori, frame2, &op_minus));
    check_equality_master_hdrl_img(frame1, sub_hdrl);

    sph_master_frame_delete(frame1);
    frame1 = sph_master_frame_duplicate(frame1_ori);
    sph_master_frame_add_master_frame(frame1, frame2);
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_sum_sub(frame1, frame1_ori, frame2, &op_plus));
    check_equality_master_hdrl_img(frame1, add_hdrl);

    sph_master_frame_delete(frame1);
    sph_master_frame_delete(frame2);
    sph_master_frame_delete(frame1_ori);
    sph_master_frame_delete(frame2_ori);
    hdrl_image_delete(sub_hdrl);
    hdrl_image_delete(add_hdrl);
}

static double op_mul(const double a, const double b){return a * b;}
static double op_div(const double a, const double b){return a / b;}

static void
check_vectorial_mul_div(void){


    const int x_sz = 2;
    const int y_sz = 3;


    double vals1[] = {-2,    1,   0,   4,  5,   8};
    double vals2[] = {3,    -4,   1,   5,  7,  -2};

    double rms1[]  = {0.1, 0.0, 0.2, 0.4, .3, 0.0};
    double rms2[]  = {0.3, 0.4, 0.1, 0.3, .4, 0.8};

    int badpxs1[]  = {0,     1,   0,   2,  0,   0}; /*2 rejected pixels*/
    int badpxs2[]  = {1,     0,   0,   1,  0,   0}; /*2 rejected pixels*/

    double cmbs1[] = {0,     4,   3,   1,  0,   2};
    double cmbs2[] = {2,     2,   5,   6,  7,   8};

    sph_master_frame * frame1 = gen_master_frame(vals1, rms1, badpxs1, cmbs1, x_sz, y_sz);
    sph_master_frame * frame2 = gen_master_frame(vals2, rms2, badpxs2, cmbs2, x_sz, y_sz);

    sph_master_frame * frame1_ori = sph_master_frame_duplicate(frame1);
    sph_master_frame * frame2_ori = sph_master_frame_duplicate(frame2);

    hdrl_image * mul_hdrl = convert_and_operate_on_hdrl(frame1, frame2, hdrl_image_mul_image, CPL_TRUE);
    hdrl_image * div_hdrl = convert_and_operate_on_hdrl(frame1, frame2, hdrl_image_div_image, CPL_TRUE);


    cpl_test_eq(get_vect_size(vals1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms1), x_sz * y_sz);

    cpl_test_eq(get_vect_size(vals2), x_sz * y_sz);
    cpl_test_eq(get_vect_size(rms2), x_sz * y_sz);

    cpl_test_eq(get_vect_size(badpxs1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(badpxs2), x_sz * y_sz);

    cpl_test_eq(get_vect_size(cmbs1), x_sz * y_sz);
    cpl_test_eq(get_vect_size(cmbs2), x_sz * y_sz);

    sph_master_frame_multiply_master_frame(frame1, frame2);
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_mul_div(frame1, frame1_ori, frame2, &op_mul));
    check_equality_master_hdrl_img(frame1, mul_hdrl);

    sph_master_frame_delete(frame1);
    frame1 = sph_master_frame_duplicate(frame1_ori);

    sph_master_frame_divide_master_frame(frame1, frame2);
    cpl_test(sph_master_frame_are_equal(frame2, frame2_ori));
    cpl_test(sph_master_frame_check_mul_div(frame1, frame1_ori, frame2, &op_div));
    check_equality_master_hdrl_img(frame1, div_hdrl);

    sph_master_frame_delete(frame1);
    sph_master_frame_delete(frame2);
    sph_master_frame_delete(frame1_ori);
    sph_master_frame_delete(frame2_ori);
    hdrl_image_delete(mul_hdrl);
    hdrl_image_delete(div_hdrl);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the vectorial operations in sph_master_frame.
 */
/*----------------------------------------------------------------------------*/
static void cutest_sph_master_frame_vectorial_operation(void){

    check_vectorial_sum_sub();
    check_vectorial_mul_div();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_master_frame_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_master_frame_fit_gauss_hist(void) {
    cpl_image* im = NULL;
    gsl_rng* pRNG = NULL;
    sph_master_frame* mframe = NULL;
    cpl_vector* binpos = NULL;
    cpl_vector* binvals = NULL;
    double mean = 0.0;
    double rms = 0.0;
    double red = 0.0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    sph_test_image_tools_add_noise(im, 100.0, pRNG);

    binvals = cpl_vector_new(100);
    cpl_vector_fill(binvals, 0.0);
    binpos = cpl_vector_new(100);

    mframe = sph_master_frame_new_from_cpl_image(im);
    sph_master_frame_fit_gauss_hist(mframe, binpos, binvals, -200.0, 200.0,
            &mean, &rms, &red);

    gsl_rng_free(pRNG);

    cpl_test_error(CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Mean: %f +/- %f with certainty: %f", mean, rms, red);
    cpl_test_abs( mean, 0.0, 1.0);
    cpl_test_abs( rms, 100.0, 10.0);
    cpl_test_abs( red, 1.0, 1.0);
    cpl_vector_delete(binpos);
    cpl_vector_delete(binvals);
    cpl_image_delete(im);
    sph_master_frame_delete(mframe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_master_frame",
            cutest_init_sph_master_frame_testsuite,
            cutest_clean_sph_master_frame_testsuite);
    cpl_test_nonnull(pSuite);

    if (NULL
            == sph_test_do(pSuite, "sph_master_frame_fit_gauss_hist",
                    cutest_sph_master_frame_fit_gauss_hist)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_master_frame_interpolate_bpix",
                    cutest_sph_master_frame_interpolate_bpix)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_master_frame_smooth",
                    cutest_sph_master_frame_smooth)) {
        return sph_test_get_error();
    }

    if (NULL
                == sph_test_do(pSuite, "sph_master_frame_scalar_op",
                        cutest_sph_master_frame_scalar_operation)) {
            return sph_test_get_error();
    }

    if (NULL
                == sph_test_do(pSuite, "sph_master_frame_vectorial_op",
                        cutest_sph_master_frame_vectorial_operation)) {
            return sph_test_get_error();
    }

    if (NULL
                == sph_test_do(pSuite, "sph_master_frame_inverse",
                        cutest_sph_master_frame_inverse)) {
            return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
