/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/


#include "sph_quad_image.h"
#include "sph_common_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_filemanager.h"
#include "sph_zpl_exposure.h"
#include "sph_test_zpl_simulator.h"
#include "sph_test_image_tools.h"

#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include <cpl.h>

#include <math.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_quad_image Unit test for the sph_quad_image module
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_sph_quad_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();

}

static
int cutest_clean_sph_quad_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    return sph_end_test();
}

static
void cutest_util_sph_quad_image_create_zplpreproc_cube_minus(
        cpl_frame** zexpcube, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 51;
    int zero_even = 151;
    int pi_odd = 151;
    int pi_even = 51;
    int dV = -1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    *zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_minus.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_minus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
    }
    cpl_propertylist_delete(pl);
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
}

static
void cutest_util_sph_quad_image_create_zplpreproc_cube_plus(
        cpl_frame** zexpcube, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    int dV = 1;
    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    *zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_plus.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {

        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);
        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
    }
    cpl_propertylist_delete(pl);
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, *zexpcube);
    cpl_frameset_delete(zexpframes);
}

static
void cutest_util_sph_quad_image_create_zplpreproc_frames(
        cpl_frameset** zexpframes, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    //cpl_frameset*        zexpframes    = NULL;
    //cpl_propertylist*    pl            = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    int dV = 1;

    int nz; //nframes

    nz = nframes;
//    pl = cpl_propertylist_new();
//    cpl_propertylist_append_int( pl, SPH_ZPL_KEYWORD_CAMERA_NAME, SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    *zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(*zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
    }

}

static
void cutest_util_sph_quad_image_create_zplpreproc_frames_diff(
        cpl_frameset** zexpframes, int nframes) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    //cpl_frameset*        zexpframes    = NULL;
    cpl_propertylist* pl = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;

    const int nx = 100;
    const int ny = 100;
    int zero_odd = 100;
    int zero_even = 200;
    int pi_odd = 300;
    int pi_even = 400;
    int dV = 1;

    int nz; //nframes

    nz = nframes;
    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    *zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(*zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
        zplexp = NULL;
    }
    cpl_propertylist_delete(pl);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_quad_image_new_empty function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_new_empty(void) {
    sph_quad_image* qdimage = NULL;

    qdimage = sph_quad_image_new_empty();
    cpl_test_nonnull( qdimage );

    sph_quad_image_delete(qdimage);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_quad_image_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_new(void) {
    sph_quad_image* qdimage = NULL;
    const int nx = 100;
    const int ny = 200;

    qdimage = sph_quad_image_new(nx, ny);
    cpl_test_nonnull( qdimage );

    cpl_test_eq( cpl_image_get_size_x( qdimage->zero_image->iframe->image ),
            nx);
    cpl_test_eq(
            cpl_image_get_size_y( qdimage->zero_image->pframe->badpixelmap ),
            ny);

    sph_quad_image_delete(qdimage);
    cpl_test_error(CPL_ERROR_NONE);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_create_quad_image_from_zplexp_frameset
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_create_quad_image_from_zplexp_frameset(void) {
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* quadframes = NULL;
    sph_quad_image* qdimage = NULL;
    const int nframes = 3;

    cutest_util_sph_quad_image_create_zplpreproc_frames(&zexpframes, nframes);
    quadframes = sph_quad_image_create_quad_image_frameset_from_zplexp_frameset(
            zexpframes);
    cpl_test_nonnull( quadframes );

    qdimage = sph_quad_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(quadframes)), 0);
    cpl_test_nonnull( qdimage );

    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->iframe->image ), 169.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->pframe->image ), 29.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->iframe->image ), 29.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->pframe->image ), 169.0,
            1e-20);

    sph_quad_image_delete(qdimage);
    cpl_frameset_delete(zexpframes);
    cpl_frameset_delete(quadframes);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_create_quad_image_from_zplexp_frameset
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_create_quad_image_from_zplexp_cubes(void) {
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_frameset* quadframes = NULL;
    sph_quad_image* qdimage = NULL;
    const int nframes = 3;

    zexpcubes = cpl_frameset_new();

    cutest_util_sph_quad_image_create_zplpreproc_cube_plus(&zexpcube_plus,
            nframes);
    cpl_test_eq_string(cpl_frame_get_filename(zexpcube_plus),
            "test_zplexp_cube_plus.fits");
    cpl_frameset_insert(zexpcubes, zexpcube_plus);

    cutest_util_sph_quad_image_create_zplpreproc_cube_minus(&zexpcube_minus,
            nframes);
    cpl_test_eq_string(cpl_frame_get_filename(zexpcube_minus),
            "test_zplexp_cube_minus.fits");
    cpl_frameset_insert(zexpcubes, zexpcube_minus);

    quadframes = sph_quad_image_create_quad_image_frameset_from_zplexp_cubes(
            zexpcubes);
    cpl_test_nonnull( quadframes );
    cpl_test_eq(cpl_frameset_get_size( quadframes ), 6);

    qdimage = sph_quad_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(quadframes)), 0);
    cpl_test_nonnull( qdimage );

    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->iframe->image ), 169.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->pframe->image ), 29.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->iframe->image ), 29.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->pframe->image ), 169.0,
            1e-20);
    sph_quad_image_delete(qdimage);
    qdimage = NULL;

    qdimage = sph_quad_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_position_const(quadframes, 3)), 0);
    cpl_test_nonnull( qdimage );

    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->iframe->image ), 51.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->pframe->image ), 151.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->iframe->image ), 151.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->pframe->image ), 51.0,
            1e-20);
    sph_quad_image_delete(qdimage);
    qdimage = NULL;

    cpl_frameset_delete(zexpcubes);
    cpl_frameset_delete(quadframes);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_quality_check
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_create_quad_image_quality_check(void) {
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* quadframes = NULL;
    sph_quad_image* qdimage = NULL;

    cpl_error_code rerr;
    const int nframes = 1;

    cutest_util_sph_quad_image_create_zplpreproc_frames_diff(&zexpframes,
            nframes);
    quadframes = sph_quad_image_create_quad_image_frameset_from_zplexp_frameset(
            zexpframes);
    cpl_test_nonnull( quadframes );

    qdimage = sph_quad_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(quadframes)), 0);
    cpl_test_nonnull( qdimage );

    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->iframe->image ), 100.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->zero_image->pframe->image ), 200.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->iframe->image ), 300.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( qdimage->pi_image->pframe->image ), 400.0,
            1e-20);

    //verify sph_quad_image_quality_check: quad image without noise
    rerr = sph_quad_image_quality_check(qdimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_quad_image_save(qdimage, "test_quad_image_QC.fits.tmp", NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_quad_image_delete(qdimage);
    cpl_frameset_delete(zexpframes);
    cpl_frameset_delete(quadframes);

    qdimage = sph_quad_image_load("test_quad_image_QC.fits.tmp", 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( qdimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            100.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qdimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            200.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qdimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            300.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qdimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            400.0, 1e-20);
    sph_quad_image_delete(qdimage);

    //verify sph_quad_image_quality_check: quad image with noise
    //(see Unit test for the sph_quad_image_new_from_sph_double_images)

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_new_from_double_images
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_new_from_double_images(void) {
    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    double dV = 10.0;
    double dS = 1.0;

    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di0 );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value + dV, pframe_value + dV, sigma_iframe + dS,
            sigma_pframe + dS);
    cpl_test_nonnull( di0 );

    qi = sph_quad_image_new_from_double_images(di0, dipi);
    cpl_test_nonnull( qi );

    //verify sph_quad_image_quality_check: quad image with noise
    rerr = sph_quad_image_quality_check(qi);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            iframe_value, 5 * sigma_iframe/sqrt(nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            pframe_value, 5 * sigma_pframe/sqrt(nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            iframe_value+dV, 5 * (sigma_iframe+dS)/sqrt(nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            pframe_value+dV, 5 * (sigma_pframe+dS)/sqrt(nx*ny));

    sph_quad_image_save(qi, "test_sph_quad_image.fits.tmp", NULL);
    sph_quad_image_delete(qi);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_mask_sigma
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_mask_sigma(void) {
    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 2.0;
    double dV = 10.0;
    double dS = 0.0;

    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di0 );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value + dV, pframe_value + dV, sigma_iframe + dS,
            sigma_pframe + dS);
    cpl_test_nonnull( dipi );

    qi = sph_quad_image_new_from_double_images(di0, dipi);
    cpl_test_nonnull( qi );

    //verify sph_quad_image_quality_check: quad image with noise
    rerr = sph_quad_image_quality_check(qi);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    rerr = sph_quad_image_mask_sigma(qi, sigma_iframe);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    //printf("mask_sigma: %4.4f \n", cpl_image_get_absflux(qi->zero_image->iframe->badpixelmap)/(nx*ny));
    // P(-sigma,+sigma) =~ 68%
    cpl_test_abs(
            cpl_image_get_absflux(qi->zero_image->iframe->badpixelmap)/(nx*ny),
            0.32, 1.5e-2);

    //printf("mask_sigma: %4.4f \n", cpl_image_get_absflux(qi->zero_image->pframe->badpixelmap)/(nx*ny));
    cpl_test_abs(
            cpl_image_get_absflux(qi->zero_image->pframe->badpixelmap)/(nx*ny),
            0.32, 1.5e-2);

    //printf("mask_sigma: %4.4f \n", cpl_image_get_absflux(qi->pi_image->iframe->badpixelmap)/(nx*ny));
    cpl_test_abs(
            cpl_image_get_absflux(qi->pi_image->iframe->badpixelmap)/(nx*ny),
            0.32, 1.5e-2);

    //printf("mask_sigma: %4.4f \n", cpl_image_get_absflux(qi->pi_image->pframe->badpixelmap)/(nx*ny));
    cpl_test_abs(
            cpl_image_get_absflux(qi->pi_image->pframe->badpixelmap)/(nx*ny),
            0.32, 1.5e-2);

    sph_quad_image_save(qi, "test_qi_sph_quad_image_mask_sigma.fits.tmp", NULL);
    sph_quad_image_delete(qi);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_divide_quad_image
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_divide_quad_image(void) {
    sph_quad_image* qi = NULL;
    sph_quad_image* qi2 = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    sph_double_image* di2_0 = NULL;
    sph_double_image* di2_pi = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 100;
    double iframe_value = 200.0;
    double pframe_value = 100.0;
    double sigma_iframe = 2.0;
    double sigma_pframe = 2.0;
    double dV = 10.0;
    double dS = 1.0;
    double dval;

    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di0 );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value * dV, pframe_value * dV, sigma_iframe * dS,
            sigma_pframe * dS);
    cpl_test_nonnull( dipi );

    qi = sph_quad_image_new_from_double_images(di0, dipi);
    cpl_test_nonnull( qi );

    //di2:zero double image with noise

    di2_0 = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value * 2 * dV, pframe_value * 2 * dV, sigma_iframe * dS,
            sigma_pframe * dS);
    cpl_test_nonnull( di2_0 );

    //di2:pi double image with noise
    di2_pi = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value * 3 * dV, pframe_value * 3 * dV, sigma_iframe * dS,
            sigma_pframe * dS);
    cpl_test_nonnull( di2_pi );

    qi2 = sph_quad_image_new_from_double_images(di2_0, di2_pi);
    cpl_test_nonnull( qi2 );

    rerr = sph_quad_image_divide_quad_image(qi2, qi);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    //verify sph_quad_image_quality_check: quad image with noise
    rerr = sph_quad_image_quality_check(qi2);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /* Add 10 % tolerance, due to the randomness of the noise */
    dval = cpl_propertylist_get_double( qi2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD );
    cpl_test_abs(dval, 2.0*dV, sigma_iframe*dS/sqrt(nx*ny) * 1.1);

    dval = cpl_propertylist_get_double( qi2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN );
    cpl_test_abs(dval, 2.0*dV, sigma_pframe*dS/sqrt(nx*ny) * 1.1);

    dval = cpl_propertylist_get_double( qi2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD );
    cpl_test_abs(dval, 3.0, sigma_iframe*dS/sqrt(nx*ny) * 1.1);

    dval = cpl_propertylist_get_double( qi2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN );
    cpl_test_abs(dval, 3.0, sigma_pframe*dS/sqrt(nx*ny) * 1.1);

    sph_quad_image_save(qi2, "test_sph_quad_image_divide_quad_image.fits.tmp",
            NULL);
    sph_quad_image_delete(qi);
    sph_quad_image_delete(qi2);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_create_quad_image_from_zplexp_frameset
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_new_from_zpl_exposures(void) {
    sph_quad_image* qimage = NULL;
    sph_zpl_exposure* zplexp = NULL;
    int nx = 100;
    int ny = 100;

    zplexp = sph_test_zpl_simulator_create_exposure(nx, ny);
    qimage = sph_quad_image_new_from_zpl_exposures(zplexp);
    sph_quad_image_quality_check(qimage);

    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            100.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            200.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            300.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            400.0, 1e-20);

    sph_zpl_exposure_delete(zplexp);
    sph_quad_image_delete(qimage);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_image_save_check_extnames
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_image_save_check_extnames(void) {
    sph_quad_image* qimage = NULL;
    sph_zpl_exposure* zplexp = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 100;

    zplexp = sph_test_zpl_simulator_create_exposure(nx, ny);
    qimage = sph_quad_image_new_from_zpl_exposures(zplexp);
    sph_quad_image_quality_check(qimage);

    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            100.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            200.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            300.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            400.0, 1e-20);

    rerr = sph_quad_image_save(qimage, "test_quad_image_check_extnames.fits",
            NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    //check extnames of the double image extensions saved in double image
    cpl_test_eq(
            0,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_IMAGE_EXTNAME));
    cpl_test_eq(
            1,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_BADPIXMAP_EXTNAME));
    cpl_test_eq(
            2,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_NCOMBMAP_EXTNAME));
    cpl_test_eq(
            3,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_RMSMAP_EXTNAME));
    cpl_test_eq(
            4,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_IMAGE_EXTNAME));
    cpl_test_eq(
            5,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_BADPIXMAP_EXTNAME));
    cpl_test_eq(
            6,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_NCOMBMAP_EXTNAME));
    cpl_test_eq(
            7,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_RMSMAP_EXTNAME));

    cpl_test_eq(
            8,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_IMAGE_EXTNAME));
    cpl_test_eq(
            9,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_BADPIXMAP_EXTNAME));
    cpl_test_eq(
            10,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_NCOMBMAP_EXTNAME));
    cpl_test_eq(
            11,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_RMSMAP_EXTNAME));
    cpl_test_eq(
            12,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_IMAGE_EXTNAME));
    cpl_test_eq(
            13,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_BADPIXMAP_EXTNAME));
    cpl_test_eq(
            14,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_NCOMBMAP_EXTNAME));
    cpl_test_eq(
            15,
            cpl_fits_find_extension("test_quad_image_check_extnames.fits", SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_RMSMAP_EXTNAME));

    sph_zpl_exposure_delete(zplexp);
    sph_quad_image_delete(qimage);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_quad_divide_master_frame
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_quad_divide_master_frame(void) {
    sph_quad_image* qi = NULL;
    sph_double_image* di0 = NULL;
    sph_double_image* dipi = NULL;
    sph_master_frame* mframe = NULL;
    cpl_image* im = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    int nx = 100;
    int ny = 100;
    double iframe_value = 200.0;
    double pframe_value = 100.0;
    double dV = 10.0;

    //zero double image with noise
    di0 = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value,
            pframe_value);
    cpl_test_nonnull( di0 );

    //pi double image with noise
    dipi = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value * dV,
            pframe_value * dV);
    cpl_test_nonnull( dipi );

    qi = sph_quad_image_new_from_double_images(di0, dipi);
    cpl_test_nonnull( qi );

    im = sph_test_image_tools_create_flat_image_double(100, 100, 10.0);
    mframe = sph_master_frame_new_from_cpl_image(im);

    rerr = sph_quad_image_divide_master_frame(qi, mframe);

    //verify sph_quad_image_quality_check: quad image with noise
    rerr = sph_quad_image_quality_check(qi);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ),
            iframe_value/10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ),
            pframe_value/10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),
            iframe_value*dV/10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( qi->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),
            pframe_value*dV/10.0, 1e-20);

    sph_quad_image_save(qi, "test_sph_quad_image_divide_master_frame.fits.tmp",
            NULL);
    sph_quad_image_delete(qi);
    cpl_image_delete(im);
    im = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_quad_image",
            cutest_init_sph_quad_image_testsuite,
            cutest_clean_sph_quad_image_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_new_empty",
                    cutest_sph_quad_image_new_empty)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_new",
                    cutest_sph_quad_image_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_new_from_zpl_exposures",
                    cutest_sph_quad_image_new_from_zpl_exposures)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_quad_image_create_quad_image_from_zplexp_frameset",
                    cutest_sph_quad_image_create_quad_image_from_zplexp_frameset)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite,
                    "sph_quad_image_create_quad_image_from_zplexp_cubes",
                    cutest_sph_quad_image_create_quad_image_from_zplexp_cubes)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite,
                    "sph_quad_image_create_quad_image_quality_check",
                    cutest_sph_quad_image_create_quad_image_quality_check)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_new_from_double_image",
                    cutest_sph_quad_image_new_from_double_images)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_mask_sigma",
                    cutest_sph_quad_image_mask_sigma)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_divide_quad_image",
                    cutest_sph_quad_divide_quad_image)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_quad_image_divide_master_frame",
                    cutest_sph_quad_divide_master_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_quad_image_save_check_extnames",
                    cutest_sph_quad_image_save_check_extnames)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
