/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_smart_imagelist.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

#include <cpl.h>
#include <gsl/gsl_rng.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>


/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_smart_imagelist"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_smart_imagelist Unit test for the sph_smart_imagelist
 *                  module            
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_sph_smart_imagelist_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_sph_smart_imagelist_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    return sph_end_test();
}

#ifdef SPH_TEST_SMART_IMAGELIST
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_new(void) {
    sph_test_later( "Not yet implemented!");
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_create function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_create(void) {
    sph_test_later( "Not yet implemented!");
    return;
}
#endif
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_create_cube function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_create_cube(void) {
    sph_master_frame* mframe = NULL;
    sph_cube* cube = NULL;
    int nframes = 2;
    int ii = 0;
    sph_smart_imagelist* slist = NULL;

    cube = sph_cube_new(SPH_BASE ".fits");
    cpl_test_nonnull( cube );
    slist = sph_smart_imagelist_new();
    cpl_test_nonnull( slist );
    slist->maxmem = 50000000; /* Set max to 50MB */
    for (ii = 0; ii < nframes; ++ii) {
        mframe = sph_master_frame_new_empty();
        cpl_test_nonnull( mframe );
        mframe->image = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->image );
        mframe->badpixelmap = cpl_image_new(2048, 2048, CPL_TYPE_INT);
        cpl_test_nonnull( mframe->badpixelmap );
        mframe->rmsmap = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->rmsmap );
        mframe->ncombmap = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->ncombmap );
        cpl_image_fill_gaussian(mframe->image, 1024, 1024, 2.0 * ii, 100.0 * ii,
                100.0 * ii);
        sph_cube_append_master(cube, mframe, ii);
        sph_master_frame_delete(mframe);
    }
    sph_cube_finalise_file(cube->filename);
    slist = sph_smart_imagelist_create_cube(slist, cube, SPH_CUBE_IMAGE_EXT);
    cpl_test_nonnull( slist );
    cpl_test_error(CPL_ERROR_NONE);
    sph_smart_imagelist_delete(slist);
    sph_cube_delete(cube);
    return;
}

#ifdef SPH_TEST_SMART_IMAGELIST
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_get_set function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_get_set(void) {
    sph_test_later( "Not yet implemented!");
    return;
}
#endif
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_get_set_cube function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_get_set_cube(void) {
    sph_master_frame* mframe = NULL;
    sph_cube* cube = NULL;
    sph_cube* cubeout = NULL;
    int nframes = 2;
    int ii = 0;
    cpl_imagelist* imlist = NULL;
    sph_smart_imagelist* slist = NULL;
    int cc = 0;

    cube = sph_cube_new(SPH_BASE ".fits");
    cpl_test_nonnull( cube );
    slist = sph_smart_imagelist_new();
    cpl_test_nonnull( slist );
    slist->maxmem = 50000000; /* Set max to 50MB */

    for (ii = 0; ii < nframes; ++ii) {
        mframe = sph_master_frame_new_empty();
        cpl_test_nonnull( mframe );
        mframe->image = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->image );
        mframe->badpixelmap = cpl_image_new(2048, 2048, CPL_TYPE_INT);
        cpl_test_nonnull( mframe->badpixelmap );
        mframe->rmsmap = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->rmsmap );
        mframe->ncombmap = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_test_nonnull( mframe->ncombmap );
        cpl_image_fill_gaussian(mframe->image, 1024, 1024, 2.0 * ii, 100.0 * ii,
                100.0 * ii);
        sph_cube_append_master(cube, mframe, ii);
        sph_master_frame_delete(mframe);
    }
    sph_cube_finalise_file(cube->filename);

    slist = sph_smart_imagelist_create_cube(slist, cube, SPH_CUBE_IMAGE_EXT);
    cpl_test_nonnull( slist );
    imlist = sph_smart_imagelist_get_set_cube(slist, 0);
    while (imlist) {
        cubeout = sph_cube_new(SPH_BASE "_cubout.fits");
        cpl_test_nonnull( cubeout );
        for (ii = 0; ii < cpl_imagelist_get_size(imlist); ++ii) {
            mframe = sph_master_frame_new_empty();
            cpl_test_nonnull( mframe );
            mframe->image = cpl_image_duplicate(
                    cpl_imagelist_get_const(imlist, ii));
            cpl_test_nonnull( mframe->image );
            mframe->badpixelmap = cpl_image_new(
                    cpl_image_get_size_x(mframe->image),
                    cpl_image_get_size_y(mframe->image), CPL_TYPE_INT);
            cpl_test_nonnull( mframe->badpixelmap );
            mframe->rmsmap = cpl_image_new(cpl_image_get_size_x(mframe->image),
                    cpl_image_get_size_y(mframe->image), CPL_TYPE_DOUBLE);
            cpl_test_nonnull( mframe->rmsmap );
            mframe->ncombmap = cpl_image_new(
                    cpl_image_get_size_x(mframe->image),
                    cpl_image_get_size_y(mframe->image), CPL_TYPE_DOUBLE);
            cpl_test_nonnull( mframe->ncombmap );
            sph_cube_append_master(cubeout, mframe, ii);
            sph_master_frame_delete(mframe);

            cpl_imagelist_delete(imlist);
            imlist = NULL;
            cc++;
            if (cc < slist->nregions) {
                imlist = sph_smart_imagelist_get_set_cube(slist, cc);
            } else {
                imlist = NULL;
            }
        }
        sph_cube_finalise_file(cubeout->filename);
        sph_cube_delete(cubeout);
    }
    if (!imlist) {
        cpl_error_reset();
    }cpl_test_lt(1, cc);
    cpl_test_error(CPL_ERROR_NONE);
    sph_smart_imagelist_delete(slist);
    sph_cube_delete(cube);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_reassemble_master_frame function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_get_raw_median_properties(void) {
    cpl_frameset* frameset = NULL;
    cpl_frame* aframe = NULL;
    sph_smart_imagelist* slist = NULL;
    cpl_image* im = NULL;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    cpl_propertylist* plist = NULL;

    frameset = cpl_frameset_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);

    for (ii = 0; ii < 11; ++ii) {
        im = sph_test_image_tools_create_flat_image_double(256, 256,
                ii * 1000.0 + 1000.0);
        cpl_test_nonnull( im );
        sph_test_image_tools_apply_poisson_noise(im, pRNG);
        aframe = sph_filemanager_create_temp_frame("test_smart_imlist",
                "TESTDATA", CPL_FRAME_GROUP_RAW);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_TYPE_FLOAT, NULL,
                CPL_IO_CREATE);
        cpl_frameset_insert(frameset, aframe);
        cpl_image_delete(im);
        im = NULL;
    }

    slist = sph_smart_imagelist_create(frameset, 0);
    cpl_test_nonnull( slist );
    plist = cpl_propertylist_new();
    sph_smart_imagelist_get_raw_median_properties(slist, plist);
    cpl_test_eq( cpl_propertylist_get_size(plist), 5);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG),
            6000.0, 100.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MED),
            6000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN),
            1000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX),
            11000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE),
            3316.0, 10.0);

    im = sph_test_image_tools_create_flat_image_double(256, 256, 111.0);
    aframe = sph_filemanager_create_temp_frame("test_smart_imlist_result",
            "TESTDATA", CPL_FRAME_GROUP_RAW);
    cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_TYPE_FLOAT, plist,
            CPL_IO_CREATE);
    sph_smart_imagelist_delete(slist);
    cpl_frame_delete(aframe);
    cpl_frameset_delete(frameset);
    cpl_image_delete(im);
    im = NULL;
    cpl_propertylist_delete(plist);
    gsl_rng_free(pRNG);
    //sph_filemanager_clean();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_reassemble_master_frame function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_get_raw_median_properties_cube(void) {
    cpl_imagelist* imlist = NULL;
    cpl_frameset* frameset = NULL;
    cpl_frame* aframe = NULL;
    sph_smart_imagelist* slist = NULL;
    cpl_image* im = NULL;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    cpl_propertylist* plist = NULL;

    frameset = cpl_frameset_new();
    imlist = cpl_imagelist_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);

    for (ii = 0; ii < 11; ++ii) {
        im = sph_test_image_tools_create_flat_image_double(256, 256,
                ii * 1000.0 + 1000.0);
        cpl_test_nonnull( im );
        sph_test_image_tools_apply_poisson_noise(im, pRNG);
        cpl_imagelist_set(imlist, im, ii);
    }

    aframe = sph_filemanager_create_temp_frame("test_smart_imlist_cube",
            "TESTDATA", CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe), CPL_TYPE_FLOAT,
            NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, aframe);
    slist = sph_smart_imagelist_create(frameset, 0);
    cpl_test_nonnull( slist );

    plist = cpl_propertylist_new();
    sph_smart_imagelist_get_raw_median_properties(slist, plist);
    cpl_test_eq( cpl_propertylist_get_size(plist), 5);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG),
            6000.0, 100.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MED),
            6000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN),
            1000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX),
            11000.0, 10.0);
    cpl_test_abs(
            cpl_propertylist_get_double(plist,SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE),
            3316.0, 10.0);

    cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe), CPL_TYPE_FLOAT,
            plist, CPL_IO_CREATE);
    sph_smart_imagelist_delete(slist);
    cpl_imagelist_delete(imlist);
    cpl_frameset_delete(frameset);
    cpl_propertylist_delete(plist);
    gsl_rng_free(pRNG);
    //sph_filemanager_clean();
}

#ifdef SPH_TEST_SMART_IMAGELIST
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_reassemble_master_frame function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_reassemble_master_frame(void) {
    sph_test_later( "Not yet implemented!");
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_smart_imagelist_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_smart_imagelist_delete(void) {
    sph_test_later( "Not yet implemented!");
    return;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_smart_imagelist",
            cutest_init_sph_smart_imagelist_testsuite,
            cutest_clean_sph_smart_imagelist_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_smart_imagelist_create_cube",
                    cutest_sph_smart_imagelist_create_cube)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_smart_imagelist_get_raw_median_properties",
                    cutest_sph_smart_imagelist_get_raw_median_properties)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_smart_imagelist_get_raw_median_properties_cube",
                    cutest_sph_smart_imagelist_get_raw_median_properties_cube)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_smart_imagelist_get_set_cube",
                    cutest_sph_smart_imagelist_get_set_cube)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
