/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_spectrum.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test.h"
#include "sph_utils.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
sph_spectrum*
cutest_util_spectrum_new_load(void) {
    sph_spectrum* spec = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    int ii = 0;

    cpl_error_reset();

    spec = sph_spectrum_new(100, 1.0, 100.0);
    cpl_test_nonnull( spec );

    lams = spec->wavelengths;
    vals = spec->spec;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        cpl_vector_set(vals, ii, (double) ii);
    }

    cpl_test_eq(cpl_vector_get( lams, 0 ), 1.0);
    cpl_test_eq(cpl_vector_get( lams, 99 ), 100.0);
    return spec;
}

static
int cutest_init_spectrum_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/

    return sph_test_nop_code();
}

static
int cutest_clean_spectrum_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_new(void) {
    sph_spectrum* spec = NULL;

    cpl_error_reset();
    /* Setup and run ...*/
    spec = cutest_util_spectrum_new_load();

    /*Verify */
    cpl_test_nonnull( spec );
    sph_spectrum_delete(spec);
    spec = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_save function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_save(void) {
    sph_spectrum* spec = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cpl_error_reset();
    spec = cutest_util_spectrum_new_load();
    cpl_test_nonnull( spec );

    /* Run ... */
    rerr = sph_spectrum_save(spec, "test_spec.fits");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    sph_spectrum_delete(spec);
    spec = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_save function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_load(void) {
    sph_spectrum* spec = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cpl_error_reset();

    /* Run ... */
    spec = sph_spectrum_load("test_spec.fits");
    cpl_test_nonnull( spec );

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( spec->wavelengths );
    cpl_test_nonnull( spec->spec );
    cpl_test_eq(cpl_vector_get( spec->wavelengths, 0 ), 1.0);
    cpl_test_eq(cpl_vector_get( spec->wavelengths, 99 ), 100.0);
    cpl_test_eq(cpl_vector_get( spec->spec, 0 ), 0.0);
    cpl_test_eq(cpl_vector_get( spec->spec, 99 ), 99.0);

    sph_spectrum_delete(spec);
    spec = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_fit_wavs_wav_inverse(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double val = 0.0;
    double lam = 0.0;
    double sig = 0.4;
    cpl_vector* lines = NULL;
    int ll = 0;
    double linl = 0.0;
    cpl_vector* wavs = NULL;
    cpl_vector* wavs_inverse = NULL;
    int np = 0;

    lines = cpl_vector_new(5);
    cpl_test_nonnull( lines );
    cpl_vector_set(lines, 0, 2.24);
    cpl_vector_set(lines, 1, 8.4);
    cpl_vector_set(lines, 2, 12.1);
    cpl_vector_set(lines, 3, 19.9);
    cpl_vector_set(lines, 4, 25.2);

    spec = sph_spectrum_new(30, 1.0, 30.0);
    wavs_inverse = cpl_vector_duplicate(spec->wavelengths);
    np = cpl_vector_get_size(wavs_inverse);
    for (ii = 0; ii < cpl_vector_get_size(spec->wavelengths); ++ii) {
        cpl_vector_set(spec->wavelengths, ii,
                cpl_vector_get(wavs_inverse, np - 1 - ii));
    }

    cpl_test_nonnull( spec );
    for (ii = 0; ii < 30; ++ii) {
        lam = cpl_vector_get(spec->wavelengths, ii);
        val = 0.0;
        for (ll = 0; ll < cpl_vector_get_size(lines); ++ll) {
            linl = cpl_vector_get(lines, ll);
            //linl = linl - 5.0 + 0.1 * linl + 0.003 * linl * linl;
            //linl = linl - 5.0 + 0.2 * linl;
            val += exp(-1.0 * pow((linl - lam) / (2.0 * sig), 2.0));
        }
        cpl_vector_set(spec->spec, ii, val);
    }

    //sph_spectrum_dump(spec);

    wavs = sph_spectrum_fit_wavs(spec, lines, 2, 5, NULL, NULL, NULL);
    cpl_test_nonnull( wavs );
    cpl_vector_delete(spec->wavelengths);
    spec->wavelengths = wavs;

    //sph_spectrum_dump(spec);

    sph_spectrum_delete(spec);
    cpl_vector_delete(lines);
    cpl_vector_delete(wavs_inverse);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_fit_wavs_no_deriv_no_diff(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double val = 0.0;
    double lam = 0.0;
    double sig = 0.4;
    cpl_vector* lines = NULL;
    int ll = 0;
    double linl = 0.0;
    cpl_vector* wavs = NULL;

    lines = cpl_vector_new(5);
    cpl_test_nonnull( lines );
    cpl_vector_set(lines, 0, 2.24);
    cpl_vector_set(lines, 1, 8.4);
    cpl_vector_set(lines, 2, 12.1);
    cpl_vector_set(lines, 3, 19.9);
    cpl_vector_set(lines, 4, 25.2);

    spec = sph_spectrum_new(30, 1.0, 30.0);
    cpl_test_nonnull( spec );
    for (ii = 0; ii < 30; ++ii) {
        lam = cpl_vector_get(spec->wavelengths, ii);
        val = 0.0;
        for (ll = 0; ll < cpl_vector_get_size(lines); ++ll) {
            linl = cpl_vector_get(lines, ll);
            //linl = linl - 5.0 + 0.1 * linl + 0.003 * linl * linl;
            //linl = linl - 5.0 + 0.2 * linl;
            val += exp(-1.0 * pow((linl - lam) / (2.0 * sig), 2.0));
        }
        cpl_vector_set(spec->spec, ii, val);
    }

    //sph_spectrum_dump(spec);

    wavs = sph_spectrum_fit_wavs(spec, lines, 2, 5, NULL, NULL, NULL);
    cpl_test_nonnull( wavs );
    cpl_vector_delete(spec->wavelengths);
    spec->wavelengths = wavs;

    //sph_spectrum_dump(spec);

    sph_spectrum_delete(spec);
    cpl_vector_delete(lines);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_fit_wavs_deriv_diff(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double val = 0.0;
    double lam = 0.0;
    double sig = 0.4;
    cpl_vector* lines = NULL;
    int ll = 0;
    double linl = 0.0;
    cpl_vector* wavs = NULL;
    cpl_vector* derivs = NULL;
    cpl_vector* diffs = NULL;

    lines = cpl_vector_new(5);
    cpl_test_nonnull( lines );
    cpl_vector_set(lines, 0, 2.24);
    cpl_vector_set(lines, 1, 8.4);
    cpl_vector_set(lines, 2, 12.1);
    cpl_vector_set(lines, 3, 19.9);
    cpl_vector_set(lines, 4, 25.2);

    spec = sph_spectrum_new(30, 1.0, 30.0);
    cpl_test_nonnull( spec );
    for (ii = 0; ii < 30; ++ii) {
        lam = cpl_vector_get(spec->wavelengths, ii);
        val = 0.0;
        for (ll = 0; ll < cpl_vector_get_size(lines); ++ll) {
            linl = cpl_vector_get(lines, ll);
            //linl = linl - 5.0 + 0.1 * linl + 0.003 * linl * linl;
            linl = linl + 1.2; // + 0.2 * linl;
            val += exp(-1.0 * pow((linl - lam) / (2.0 * sig), 2.0));
        }
        cpl_vector_set(spec->spec, ii, val);
    }

    wavs = sph_spectrum_fit_wavs(spec, lines, 2, 3, &diffs, &derivs, NULL);
    cpl_test_nonnull( wavs );
    cpl_vector_delete(spec->wavelengths);
    spec->wavelengths = wavs;

    sph_spectrum_delete(spec);

    cpl_test_nonnull( derivs );
    //cpl_vector_dump(derivs,stdout);
    cpl_test_nonnull( diffs );
    //cpl_vector_dump(diffs,stdout);
    cpl_vector_delete(derivs);
    derivs = NULL;
    cpl_vector_delete(diffs);
    diffs = NULL;
    cpl_vector_delete(lines);
    lines = NULL;
    return;
}
static
void cutest_spectrum_interpolate(void) {
    sph_spectrum* spec = NULL;
    sph_spectrum* spec2 = NULL;
    sph_spectrum* nspec = NULL;

    spec = sph_spectrum_new(10, 1.0, 100.0);
    cpl_vector_delete(spec->spec);
    spec->spec = cpl_vector_duplicate(spec->wavelengths);
    cpl_test_nonnull( spec );
    spec2 = sph_spectrum_new(20, 1.0, 100.0);
    cpl_test_nonnull( spec2 );
    nspec = sph_spectrum_new_interpolate(spec, spec2->wavelengths);
    cpl_test_nonnull( nspec );

    cpl_test_abs(cpl_vector_get(nspec->spec,0), 1.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->spec,19), 100.0, 0.01);

    sph_spectrum_delete(spec);
    sph_spectrum_delete(spec2);
    sph_spectrum_delete(nspec);
}

static
void cutest_spectrum_get_resolving_power(void) {
    sph_spectrum* spec = NULL;
    double respower = 0.0;
    int i = 0;

    cpl_error_reset();
    cpl_test_abs(sph_spectrum_get_resolving_power(NULL), 0.0,
            0.000001);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_reset();

    spec = sph_spectrum_new(10, 1.0, 100.0);
    cpl_test_nonnull( spec );
    for (i = 0; i < 10; ++i) {
        cpl_vector_set(spec->wavelengths, i, 1.0 + i);
        respower += (1.0 + i);
    }
    respower /= 10.0;
    cpl_test_abs(sph_spectrum_get_resolving_power(spec), respower,
            0.001);
    sph_spectrum_delete(spec);
}

static
void cutest_spectrum_interpolate_rms_interpolated(void) {
    sph_spectrum* spec = NULL;
    sph_spectrum* spec2 = NULL;
    sph_spectrum* nspec = NULL;

    spec = sph_spectrum_new(10, 1.0, 100.0);
    cpl_vector_delete(spec->spec);
    cpl_vector_delete(spec->rms);
    spec->spec = cpl_vector_duplicate(spec->wavelengths);
    spec->rms = cpl_vector_duplicate(spec->wavelengths);
    cpl_test_nonnull( spec );
    spec2 = sph_spectrum_new(20, 1.0, 100.0);
    cpl_test_nonnull( spec2 );
    nspec = sph_spectrum_new_interpolate(spec, spec2->wavelengths);
    cpl_test_nonnull( nspec );

    cpl_test_abs(cpl_vector_get(nspec->spec,0), 1.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->spec,19), 100.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->rms,19), 100.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->rms,0), 1.0, 0.01);

    sph_spectrum_delete(spec);
    sph_spectrum_delete(spec2);
    sph_spectrum_delete(nspec);
}
static
void cutest_spectrum_interpolate_rms_interpolated_test2(void) {
    sph_spectrum* spec = NULL;
    sph_spectrum* spec2 = NULL;
    sph_spectrum* nspec = NULL;

    spec = sph_spectrum_new(10, 1.0, 100.0);
    cpl_vector_delete(spec->spec);
    cpl_vector_delete(spec->rms);
    spec->spec = cpl_vector_duplicate(spec->wavelengths);
    spec->rms = cpl_vector_duplicate(spec->wavelengths);
    cpl_vector_fill(spec->rms, 10.0);
    cpl_test_nonnull( spec );
    spec2 = sph_spectrum_new(20, 1.0, 100.0);
    cpl_test_nonnull( spec2 );
    nspec = sph_spectrum_new_interpolate(spec, spec2->wavelengths);
    cpl_test_nonnull( nspec );

    cpl_test_abs(cpl_vector_get(nspec->spec,0), 1.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->spec,19), 100.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->rms,19), 10.0, 0.01);
    cpl_test_abs(cpl_vector_get(nspec->rms,0), 10.0, 0.01);

    sph_spectrum_delete(spec);
    sph_spectrum_delete(spec2);
    sph_spectrum_delete(nspec);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_spectrum_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_spectrum_delete(void) {
    sph_spectrum* spec = NULL;

    spec = sph_spectrum_new(100, 1.0, 100.0);
    cpl_test_nonnull( spec );

    sph_spectrum_delete(spec);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("spectrum_test", cutest_init_spectrum_testsuite,
            cutest_clean_spectrum_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_spectrum_new", cutest_spectrum_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_save", cutest_spectrum_save)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_load", cutest_spectrum_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_fit_wavs_no_deriv_no_diff",
                    cutest_spectrum_fit_wavs_no_deriv_no_diff)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_fit_wavs_deriv_diff",
                    cutest_spectrum_fit_wavs_deriv_diff)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_fit_wav_inverse",
                    cutest_spectrum_fit_wavs_wav_inverse)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_interpolate",
                    cutest_spectrum_interpolate)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_interpolate rms ok",
                    cutest_spectrum_interpolate_rms_interpolated)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_interpolate test2",
                    cutest_spectrum_interpolate_rms_interpolated_test2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_get_resolving_power",
                    cutest_spectrum_get_resolving_power)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_spectrum_delete",
                    cutest_spectrum_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
