/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/


#include "sph_triple_image.h"
#include "sph_common_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_filemanager.h"

#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include <cpl.h>

#include <math.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_triple_image Unit test for the sph_triple_image module
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static sph_master_frame* create_mframe(double value) {
    sph_master_frame* mframe;
    int nx = 256;
    int ny = 256;
    mframe = sph_master_frame_new(nx, ny);
    cpl_image_delete(mframe->badpixelmap);
    mframe->badpixelmap = NULL;
    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_delete(mframe->rmsmap);
    mframe->rmsmap = NULL;
    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    cpl_image_delete(mframe->ncombmap);
    mframe->ncombmap = NULL;
    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    cpl_image_delete(mframe->image);
    mframe->image = NULL;
    mframe->image = sph_test_image_tools_create_flat_image_double(nx, ny,
            value);
    return mframe;
}

static
int cutest_init_sph_triple_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/

    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static
int cutest_clean_sph_triple_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_new_empty function.
 */
/*----------------------------------------------------------------------------*/
static void cutest_sph_triple_image_new_empty(void) {
    sph_triple_image* trimage = sph_triple_image_new_empty();
    cpl_test_nonnull( trimage );

    sph_triple_image_delete(trimage);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_triple_image_new(void) {
    sph_triple_image* trimage = NULL;
    const int nx = 100;
    const int ny = 200;

    trimage = sph_triple_image_new(nx, ny);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( trimage );

    cpl_test_eq(cpl_image_get_size_x( trimage->iframe->image ), nx);
    cpl_test_eq(cpl_image_get_size_y( trimage->iframe->badpixelmap ), ny);
    cpl_test_eq(cpl_image_get_size_x( trimage->qframe->image ), nx);
    cpl_test_eq(cpl_image_get_size_y( trimage->qframe->badpixelmap ), ny);
    cpl_test_eq(cpl_image_get_size_x( trimage->uframe->image ), nx);
    cpl_test_eq(cpl_image_get_size_y( trimage->uframe->badpixelmap ), ny);

    sph_triple_image_delete(trimage);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_new_from_master_frames function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_triple_image_new_from_master_frames(void) {
    sph_triple_image* trimage = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    mframe1 = create_mframe(10.0);
    mframe2 = create_mframe(20.0);
    mframe3 = create_mframe(30.0);

    trimage = sph_triple_image_new_from_master_frames(mframe1, mframe2,
            mframe3);
    cpl_test_nonnull( trimage );

    rerr = sph_triple_image_quality_check(trimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            30.0, 1e-20);
    sph_triple_image_delete(trimage);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_duplicate function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_triple_image_duplicate(void) {
    sph_triple_image* trimage = NULL;
    sph_triple_image* trimage_copy = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    mframe1 = create_mframe(10.0);
    mframe2 = create_mframe(20.0);
    mframe3 = create_mframe(30.0);

    trimage = sph_triple_image_new_from_master_frames(mframe1, mframe2,
            mframe3);
    cpl_test_nonnull( trimage );
    trimage_copy = sph_triple_image_duplicate(trimage);
    sph_triple_image_delete(trimage);

    rerr = sph_triple_image_quality_check(trimage_copy);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage_copy->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage_copy->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage_copy->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            30.0, 1e-20);
    sph_triple_image_delete(trimage_copy);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_save_load function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_triple_image_save_load(void) {
    sph_triple_image* trimage = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    mframe1 = create_mframe(10.0);
    mframe2 = create_mframe(20.0);
    mframe3 = create_mframe(30.0);

    trimage = sph_triple_image_new_from_master_frames(mframe1, mframe2,
            mframe3);
    cpl_test_nonnull( trimage );

    //save triple image without qc parameters
    rerr = sph_triple_image_save(trimage, "test_trimage.fits.tmp", NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    sph_triple_image_delete(trimage);

    //load triple image & calculate qc params to test save/load triple image functions
    trimage = sph_triple_image_load("test_trimage.fits.tmp", 0);

    rerr = sph_triple_image_quality_check(trimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            30.0, 1e-20);

    //save triple image with qc params to check save/load triple image functions (with qc params)
    rerr = sph_triple_image_save(trimage, "test_trimage.fits.tmp", NULL);
    sph_triple_image_delete(trimage);
    trimage = NULL;
    trimage = sph_triple_image_load("test_trimage.fits.tmp", 0);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            30.0, 1e-20);
    sph_triple_image_delete(trimage);
    trimage = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_triple_image_save_load function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_triple_image_add_subtract_divide(void) {
    sph_triple_image* trimage = NULL;
    sph_triple_image* trimage2 = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    mframe1 = create_mframe(10.0);
    mframe2 = create_mframe(20.0);
    mframe3 = create_mframe(30.0);

    trimage = sph_triple_image_new_from_master_frames(mframe1, mframe2,
            mframe3);
    cpl_test_nonnull( trimage );
    trimage2 = sph_triple_image_new_from_master_frames(create_mframe(100.0),
            create_mframe(200.0), create_mframe(300.0));
    cpl_test_nonnull( trimage2 );

    rerr = sph_triple_image_add_triple_image(trimage, trimage2);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_triple_image_quality_check(trimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            110.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            220.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            330.0, 1e-20);

    rerr = sph_triple_image_subtract_triple_image(trimage, trimage2);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_triple_image_quality_check(trimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            10.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            30.0, 1e-20);

    rerr = sph_triple_image_divide_double(trimage, 10.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_triple_image_quality_check(trimage);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            1.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            2.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(trimage->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            3.0, 1e-20);

    sph_triple_image_delete(trimage);
    trimage = NULL;
    sph_triple_image_delete(trimage2);
    trimage2 = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_triple_image",
            cutest_init_sph_triple_image_testsuite,
            cutest_clean_sph_triple_image_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_new_empty",
                    cutest_sph_triple_image_new_empty)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_new",
                    cutest_sph_triple_image_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_new_from_master_frames",
                    cutest_sph_triple_image_new_from_master_frames)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_duplicate",
                    cutest_sph_triple_image_duplicate)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_save_load",
                    cutest_sph_triple_image_save_load)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_triple_image_add_subtract_divide",
                    cutest_sph_triple_image_add_subtract_divide)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
