/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>

#include <string.h>
#include <stdio.h>
#ifdef HAVE_UNISTD_H
/* Needed for access() */
#include <unistd.h>
#endif

#include "cutest_sph_filter_generation.h"

/*-----------------------------------------------------------------------------
 Define
 -----------------------------------------------------------------------------*/


#ifndef SPH_ZPL_STAR_PHOT_TABLE_ZIMPOL
#define SPH_ZPL_STAR_PHOT_TABLE_ZIMPOL "spher-calib/zpl/sph_zpl_phot_star_table.fits"
#endif

#define PRO_RECi_PIPE_ID          "ESO PRO REC1 PIPE ID"
#define PRO_RECi_PIPE_ID_COMMENT "Pipeline (unique) identifier"

#define PRO_RECi_ID              "ESO PRO REC1 ID"
#define PRO_RECi_ID_COMMENT      "Pipeline recipe (unique)"


/*-----------------------------------------------------------------------------
 Function declarations
 -----------------------------------------------------------------------------*/

typedef struct {
    double dec;
    double ra;
    const char * main_id;
    const char * otype_3;
    const char * sp_type;
    const char * hd_number;
    const char * hip_number;
    double flux_i;
    double flux_r;
    double flux_v;
    double std_v_r_i;
}phot_star;

static void sph_zpl_phot_table_create(const char *, const char *, const char *);
static double convert_deg_or_h_min_sec(const double degs_or_hours,
        const double min, const double sec);
static double convert_ra(const double deg, const double min, const double sec);
static double convert_dec(const double deg, const double min, const double sec);
static cpl_table * convert_stars_to_table(const phot_star * stars,
        const cpl_size num_stars);
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[]) {

    char * base;
    char *end;
    int ndir = 5; /* Depth of source directory */

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    cpl_test_lt(0, argc);

    base = cpl_strdup(argv[0]);

    while ((end = strrchr(base, '/')) > base && --ndir >= 0) {
        *end = 0;
    }
    if (end != NULL) {
        const char *pipeid = strrchr(argv[0], '/');

        end[1] = 0; /* Truncate after '/' */

        const cpl_boolean doit = argc == 2;

        if(doit)
            sph_zpl_phot_table_create(base, SPH_ZPL_STAR_PHOT_TABLE_ZIMPOL,
                            pipeid ? pipeid + 1 : NULL);
        else
            cpl_msg_info(cpl_func, "Creation of the table disabled, test skipped");
    }

    cpl_free(base);

    return cpl_test_end(0);
}

/**@}*/

static double convert_deg_or_h_min_sec(const double degs_or_hours,
        const double min, const double sec){

    cpl_boolean is_neg = degs_or_hours < 0;

    const double to_ret = abs(degs_or_hours) + (min + sec / 60.0) / 60.0;

    if(is_neg) return -to_ret;
    return to_ret;
}

static double convert_ra(const double h, const double min, const double sec){
    static const double hours_in_deg = 360.0 / 24.0;
    return hours_in_deg * convert_deg_or_h_min_sec(h, min, sec);
}

static double convert_dec(const double deg, const double min, const double sec){
    return convert_deg_or_h_min_sec(deg, min, sec);
}

static cpl_table * convert_stars_to_table(const phot_star * stars,
        const cpl_size num_stars){

    const char * RA_CLMN = "RA";
    const char * DEC_CLMN = "DEC";
    const char * MAIN_ID_CLMN= "Target";
    const char * OTYPE_CLMN = "OTp3";
    const char * SPTYPE_CLMN = "SpT";
    const char * HD_CLMN = "HdNum";
    const char * HIP_CLMN = "HipNum";
    const char * FLX_I_CLMN = "N_I";
    const char * FLX_R_CLMN = "N_R";
    const char * FLX_V_CLMN = "V";
    const char * STD_CLMN = "STD_V_R_I";

    cpl_table * tb = cpl_table_new(num_stars);

    cpl_table_new_column(tb, MAIN_ID_CLMN, CPL_TYPE_STRING);
    cpl_table_new_column(tb, SPTYPE_CLMN, CPL_TYPE_STRING);
    cpl_table_new_column(tb, RA_CLMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tb, DEC_CLMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tb, OTYPE_CLMN, CPL_TYPE_STRING);
    cpl_table_new_column(tb, HD_CLMN, CPL_TYPE_STRING);
    cpl_table_new_column(tb, HIP_CLMN, CPL_TYPE_STRING);
    cpl_table_new_column(tb, FLX_I_CLMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tb, FLX_R_CLMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tb, FLX_V_CLMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(tb, STD_CLMN, CPL_TYPE_DOUBLE);

    for(cpl_size i = 0; i < num_stars; ++i){
        const phot_star * star = stars + i;

        cpl_table_set(tb, RA_CLMN, i, star->ra);
        cpl_table_set(tb, DEC_CLMN, i, star->dec);

        cpl_table_set(tb, FLX_I_CLMN, i, star->flux_i);
        cpl_table_set(tb, FLX_R_CLMN, i, star->flux_r);
        cpl_table_set(tb, FLX_V_CLMN, i, star->flux_v);
        cpl_table_set(tb, STD_CLMN, i, star->std_v_r_i);

        cpl_table_set_string(tb, MAIN_ID_CLMN, i, star->main_id);
        cpl_table_set_string(tb, OTYPE_CLMN, i, star->otype_3);
        cpl_table_set_string(tb, SPTYPE_CLMN, i, star->sp_type);
        cpl_table_set_string(tb, HD_CLMN, i, star->hd_number);
        cpl_table_set_string(tb, HIP_CLMN, i, star->hip_number);
    }


    return tb;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Generate the std flux table for zeropoint calculation for ZIMPOL
 @param path   Initial path to table frame
 @param base   Name of FITS file
 @param pipeid Name of FITS producing "recipe"
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_zpl_phot_table_create(const char * path,
                                    const char * base,
                                    const char * pipeid)
{
    char * file = cpl_sprintf("%s%s", path, base);

    const cpl_size num_stars = 11;
    /*this is a test, no big risk of stack overflow, hence we allocate on the stack to avoid memory management*/
    phot_star stars[num_stars];

    stars[0] =  (phot_star){convert_dec(-33, 58, 00.289), convert_ra(3 , 39, 02.9032), "HD  22861"  , "*"  ,  "A0V"       , "HD  22861", "HIP  17030",    9.442000389, 9.329999924, 9.359999657, 0.057977306};
    stars[1] =  (phot_star){convert_dec(-31, 45, 24.565), convert_ra(5 , 27, 24.7603), "HD  36022"  , "*"  ,  "A0V"       , "HD  36022",  NULL,            9.163999557, 9.119999886, 9.119999886, 0.025403222};
    stars[2] =  (phot_star){convert_dec(-29, 59, 32.781), convert_ra(8 ,  5, 36.5230), "CD-29  5507", "*"  ,  "A0V"       , NULL,         "HIP  39606",    9.071000099, 8.880000114, 8.899999619, 0.104977886};
    stars[3] =  (phot_star){convert_dec(-46,  2, 29.504), convert_ra(8 , 46,  1.6439), "* a Vel"    , "*"  ,  "A0II"      , "HD  75063",  "HIP  43023",    3.730000019, 3.859999895, 3.910000086, 0.092915742};
    stars[4] =  (phot_star){convert_dec(-17, 05, 03.756), convert_ra(10, 28, 11.9124), "HD  90723"  , "*"  ,  "A0V"       , "HD  90723",   NULL  ,        9.237999916, 9.229999542, 9.279999733, 0.026857679};
    stars[5] =  (phot_star){convert_dec(-17, 05, 03.756), convert_ra(10, 28, 11.9124), "HD  90723"  , "*"  ,  "A0V"       , "HD  90723",   NULL  ,        9.237999916, 9.229999542, 9.279999733, 0.026857679};
    stars[6] =  (phot_star){convert_dec(-16, 30, 55.552), convert_ra(12, 29, 51.8551), "* del Crv"  , "PM*",    "A0IV(n)kB9" , "HD 108767",    "HIP  60965",    3.029999971, 2.99000001 , 2.940000057, 0.045092455};
    stars[7] =  (phot_star){convert_dec(-35, 44, 38.835), convert_ra(13,  9, 36.6782), "HD 114225"  , "*"  ,    "A0IV"       , "HD 114225",    NULL,           9.420000076, 9.420000076, 9.409999847, 0.005773635};
    stars[8] =  (phot_star){convert_dec(-46, 30, 20.461), convert_ra(17, 35, 39.5895), "sig Ara"    , "*"  ,  "A0V"       , "HD 159217",    "HIP  86092",    4.650000095, 4.599999905, 4.590000153, 0.032145521};
    stars[9] =  (phot_star){convert_dec(-16, 06, 10.870), convert_ra(18, 32, 18.6645), "HD 170884"  , "*"  ,    "A0III"      , "HD 170884",    NULL,           9.340999603, 9.18999958 , 9.369999886, 0.096645515};
    stars[10] = (phot_star){convert_dec(-10, 40, 40.623), convert_ra(22, 30, 38.8154), "* sig Aqr"  , "*"  , "A0IVs",       "HD 213320",    "HIP 111123",    4.849999905, 4.809999943, 4.809999943, 0.023093989};

    cpl_table * tb = convert_stars_to_table(stars, num_stars);

    cpl_propertylist * plist = cpl_propertylist_new();

    sph_filter_plist_append_string_value_and_comment(plist, "INSTRUME", "SPHERE", "Instrument used");

    sph_filter_plist_append_double_value_and_comment(plist, "MJD-OBS",
            57113.0, "This calib cannot be used before this date");

    sph_filter_plist_append_string_value_and_comment(plist, "ESO PRO CATG",
            "ZPL_PHOT_STAR_TABLE", NULL);

    sph_filter_plist_append_string_value_and_comment(plist, "ESO SEQ ARM",
            "ZIMPOL", "Name of the sub-system");


    cpl_error_code code = cpl_table_save(tb, plist, NULL, file, CPL_IO_CREATE);
    cpl_test_fits(file);
    if(code){
        cpl_msg_error(cpl_func, "Failed to write %s aborting. (pipeid=%s)", file,
                      pipeid ? pipeid : "<NULL>");
        cpl_test_eq(code, CPL_ERROR_NONE);
    }

    cpl_table_delete(tb);
    cpl_propertylist_delete(plist);

    cpl_free(file);
}
