from edps import (
    task, qc0, calchecker
)

from . import zimpol_rules as rules
from .zimpol_classification import (
    ZPL_MASTER_DARK_IMAGING_CAM1, ZPL_MASTER_DARK_IMAGING_CAM2,
    ZPL_INT_FLAT_FIELD_IMAGING_CAM1, ZPL_INT_FLAT_FIELD_IMAGING_CAM2
)
from .zimpol_datasources import zpl_raw_science_center_img
from .zimpol_imaging_subwkf import (
    process_zimpol_science_imaging, process_zimpol_standard_imaging, process_zpl_img_internal_calib_img
)

"""
# Naming conventions
- Classification rules for raw frames start with `cls_`
- Classification rules for products have the same name as the PRO.CATG
- DataSources start with `raw_`
- Tasks start with `task_zpl_`
"""
__title__ = "SPHERE ZIMPOL imaging workflow"
########
# TASK #
########
# imaging cascade
# --- Determination of bias level and read-out noise (monitoring only, not needed in science reduction) ---------------
# --- Determination of dark current ----------------------------------------------------------------------------------
# --- Determination of pixel-to-pixel variation in sensitivity (quantum efficiency, filter defects, etc) -------------
# different raw data may be needed to match the filters used in CAM2 and CAM2, therefore we have 2 tasks and
# filter the products per camera
task_zpl_bias_img, task_zpl_dark_img, task_zpl_flat_img_cam1, task_zpl_flat_img_cam2 \
    = process_zpl_img_internal_calib_img()

# processing of on-sky data
# calibrations are optional because the pipeline can process the data without them
# flat fielding may sometimes improve the quality
# more information can be found in https://www.aanda.org/articles/aa/pdf/2018/11/aa33620-18.pdf
task_zpl_std_astrometry_img, task_zpl_std_flux_img = process_zimpol_standard_imaging(task_zpl_dark_img,
                                                                                     task_zpl_flat_img_cam1,
                                                                                     task_zpl_flat_img_cam2,
                                                                                     task_zpl_bias_img)

# Determine the star's position behind the coronagraph, needed for precise definition of center of
# rotation in pupil-stabilized data
task_zpl_coronagraph_center_img = (task("zimpol_coronagraph_center_imaging")
                                   .with_recipe("sph_zpl_star_center_img")
                                   .with_main_input(zpl_raw_science_center_img)
                                   .with_associated_input(task_zpl_dark_img, [ZPL_MASTER_DARK_IMAGING_CAM1,
                                                                              ZPL_MASTER_DARK_IMAGING_CAM2], min_ret=0)
                                   .with_associated_input(task_zpl_flat_img_cam1, [ZPL_INT_FLAT_FIELD_IMAGING_CAM1],
                                                          min_ret=0, match_rules=rules.zpl_ff_cam1)
                                   .with_associated_input(task_zpl_flat_img_cam2, [ZPL_INT_FLAT_FIELD_IMAGING_CAM2],
                                                          min_ret=0, match_rules=rules.zpl_ff_cam2)
                                   .with_meta_targets([qc0, calchecker])
                                   .build())

# --- process raw science imaging ------------------------------------------------------------------------------------
task_zpl_science_img, task_zpl_science_flux_img = process_zimpol_science_imaging(task_zpl_dark_img,
                                                                                 task_zpl_flat_img_cam1,
                                                                                 task_zpl_flat_img_cam2,
                                                                                 task_zpl_coronagraph_center_img)
