from adari_core.data_libs.master_detmon import MasterDetmonReport
from adari_core.plots.text import TextPlot

from adari_core.utils.esodet import strip_prescan_overscan_uves
from adari_core.utils.utils import fetch_kw_or_default
from .uves_util import UvesSetupInfo

from . import UvesReportMixin


class UvesDetmonReport(UvesReportMixin, MasterDetmonReport):
    files_needed = {
        "on": "ON_RAW",
        "off": "OFF_RAW",
        "gain_info": "GAIN_INFO",
        "detlin_info": "DET_LIN_INFO",
        "coeffs_cube": "COEFFS_CUBE",
    }

    def __init__(self):
        super().__init__("uves_detmon", self.files_needed)

    def access_port(self, im_hdu, port):
        return strip_prescan_overscan_uves(im_hdu)

    def generate_panels(self, **kwargs):
        panels = {}

        # Setup for master report
        data_info = {}
        
        data_info["exptime_key"] = "EXPTIME"
        data_info["gain_nominal_key"] = "HIERARCH ESO DET OUT1 CONAD"
        data_info["lintime_key"] = "EXPTIME"

        rotation_kwargs = {"rotate": 270}
        exts = []
        # Check if this is a blue or a red set of frames
        try:
            first_on = self.hdus[0][self.generate_file_key("on", 0)]
        except KeyError:  # Couldn't find the first RAW_ON file
            raise RuntimeError("Unable to locate a RAW_ON file in given SOF")
        
        arm = UvesSetupInfo.get_arm_info(first_on)
        if len(first_on) == 1:
            exts = ["PRIMARY"]
        else:
            if "RED" in arm:
                rotation_kwargs["flip"] = "y"
                read_speed = first_on[0].header.get(
                    "HIERARCH ESO DET READ SPEED", "None"
                )
                if (
                    "50kHz" in read_speed
                    and first_on[0].header.get("HIERARCH ESO DET WIN1 BINY") == 3
                ):
                    exts = ["WIN1", "WIN2"]
                else:
                    exts = ["CCD-20", "CCD-44"]
            else:
                exts = ["WIN1", "WIN2"]  # 625kHz_2pt case

        for ext in exts:
            # We know each UVES chip has only one port, so no need to calculate
            # or loop over ports
            new_panels = super().generate_panels(
                data_info=data_info,
                ext=ext,
                port=1,
                # raw_im_rotation_kwargs=rotation_kwargs
            )

            for panel, panel_descr in new_panels.items():
                i = panel_descr["hdus_i"]
                # Add UVES-specific metadata
                text2 = TextPlot(v_space=self.METADATA_V_SPACING)
                text2.add_data(
                    [
                        f"DET.WIN1.BINX: {fetch_kw_or_default(self.hdus[i]['detlin_info'][0], 'HIERARCH ESO DET WIN1 BINX', 'UNKNOWN')}",
                        f"DET.WIN1.BINY: {fetch_kw_or_default(self.hdus[i]['detlin_info'][0], 'HIERARCH ESO DET WIN1 BINY', 'UNKNOWN')}",
                        f"DET.READ.SPEED: {fetch_kw_or_default(self.hdus[i]['detlin_info'][0], 'HIERARCH ESO DET READ SPEED', 'UNKNOWN')}",
                    ]
                )
                panel.assign_plot(text2, 1, 0)

                # Append INS PATH to report filename
                panel_descr[
                    "report_name"
                ] += f"_{fetch_kw_or_default(self.hdus[i]['detlin_info'][0], 'HIERARCH ESO INS PATH', 'ARM_UNKNOWN')}"

            panels = {**panels, **new_panels}

        return panels


rep = UvesDetmonReport()
