from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.utils.esodet import (
    strip_prescan_overscan_uves,
    correct_port_headers_uves,
)
from adari_core.plots.text import TextPlot

# from adari_core.data_libs.master_raw_cuts import Panel
from astropy.io import fits

import os

from . import UvesReportMixin
from .uves_util import UvesSetupInfo

center_size = 200


class UvesMasterBiasReport(UvesReportMixin, MasterRawCutsReport):
    def __init__(self):
        super().__init__("uves_master_bias")
        self.center_size = center_size

    def remove_raw_scan(self, im_hdu, **kwargs):
        return strip_prescan_overscan_uves(im_hdu, debug=True, **kwargs)

    def parse_sof(self):
        # Need to generate two report sets:
        # MASTER_BIAS
        master_bias_b = None
        master_bias_rl = None
        master_bias_ru = None
        first_raw_b = None
        first_raw_r = None

        for filename, catg in self.inputs:
            if catg == "MASTER_BIAS_BLUE" and master_bias_b is None:
                master_bias_b = filename
            elif catg == "MASTER_BIAS_REDL" and master_bias_rl is None:
                master_bias_rl = filename
            elif catg == "MASTER_BIAS_REDU" and master_bias_ru is None:
                master_bias_ru = filename
            elif catg == "BIAS_BLUE" and first_raw_b is None:
                first_raw_b = filename
            elif catg == "BIAS_RED" and first_raw_r is None:
                first_raw_r = filename

        # Build and return the file name list
        file_lists = []
        if first_raw_b is not None:
            if master_bias_b is not None:
                file_lists.append(
                    {
                        "master_im": master_bias_b,
                        "raw_im": first_raw_b,
                    }
                )
        else:
            if master_bias_ru is not None:
                file_lists.append(
                    {
                        "master_im": master_bias_ru,
                        "raw_im": first_raw_r,
                    }
                )
            if master_bias_rl is not None:
                file_lists.append(
                    {
                        "master_im": master_bias_rl,
                        "raw_im": first_raw_r,
                    }
                )
        return file_lists

    def generate_panels(self, **kwargs):
        if len(self.hdus[0]["raw_im"]) == 1:  # Blue raw bias is single ext
            exts = "PRIMARY"
            rotation_kwargs = {"rotate": 270}
        else:
            exts = []
            rotation_kwargs = {"rotate": 270, "flip": "y"}
            for file_lists in self.hdus:
                arm = UvesSetupInfo.get_arm_info(file_lists["master_im"])
                all_hdu_names = [hdu.name for hdu in file_lists["raw_im"]]
                if "REDL" in arm:
                  # Red raw has one prim, two data exts
                    exts.append("CCD-44")
                else:  # If not REDL, must be REDU
                    if "CCID-20" in all_hdu_names :
                        exts.append("CCID-20")
                    else :
                        exts.append("CCD-20")

        # Correct the headers in the raw_im
        for file_lists in self.hdus:
            correct_port_headers_uves(file_lists["raw_im"], debug=True)

        new_panels = super().generate_panels(
            raw_im_ext=exts,
            raw_im_rotation=rotation_kwargs,
            raw_title="First raw",
            master_im_ext="PRIMARY",
            master_title="Master bias",
        )
        for panel, panel_descr in new_panels.items():
            panel_descr["report_description"] = (
                f"UVES bias panel - "
                f"{os.path.basename(panel_descr['master_im'])}, "
                f"{os.path.basename(panel_descr['raw'])}, "
                f"{os.path.basename(panel_descr['master_procatg'])}, "
                f"{panel_descr['master_im_ext']}"
            )

            master_im = fits.open(panel_descr["master_im"])
            raw_im = fits.open(panel_descr["raw"])
            raw_im_ext = panel_descr["raw_im_ext"]

            # Text Plot
            px = 0
            py = 0
            # which hdul and ext to use
            vspace = 0.3
            fname = os.path.basename(str(master_im.filename()))
            t1 = TextPlot(columns=1, v_space=vspace)

            col1 = (
                str(master_im["PRIMARY"].header.get("INSTRUME")),
                "EXTNAME: " + str(raw_im[raw_im_ext].header.get("EXTNAME", "N/A")),
                "PRO CATG: "
                + str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    master_im["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")
                ),
            )
            t1.add_data(col1)

            panel.assign_plot(t1, px, py, xext=2)

            px = px + 2
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = (
                "DET.WIN1.BINX: "
                + str(master_im["PRIMARY"].header.get("HIERARCH ESO DET WIN1 BINX")),
                "DET.WIN1.BINY: "
                + str(master_im["PRIMARY"].header.get("HIERARCH ESO DET WIN1 BINY")),
                "DET.READ.SPEED: "
                + str(master_im["PRIMARY"].header.get("HIERARCH ESO DET READ SPEED")),
            )
            t2.add_data(col2)
            panel.assign_plot(t2, px, py, xext=1)

            hist = panel.retrieve(4, 1)
            hist.bins = 15

        return new_panels


rep = UvesMasterBiasReport()
