from adari_core.plots.points import ScatterPlot
from adari_core.plots.text import TextPlot
from adari_core.data_libs.master_orderdef import MasterOrderdefReport
from adari_core.utils.esodet import (
    strip_prescan_overscan_uves,
    correct_port_headers_uves,
)

import os

from . import UvesReportMixin
from .uves_util import UvesSetupInfo


class UvesOrderdefFlatFieldReport(UvesReportMixin, MasterOrderdefReport):
    def __init__(self):
        super().__init__("uves_orderdef")

    def remove_raw_scan(self, im_hdu, **kwargs):
        return strip_prescan_overscan_uves(im_hdu, debug=False, **kwargs)

    def parse_sof(self):
        # building one report set
        order_table_b = None
        order_table_ru = None
        order_table_rl = None

        first_raw_b = None
        first_raw_r = None

        for filename, catg in self.inputs:
            # order_table
            if catg == "ORDER_TABLE_BLUE":
                order_table_b = filename
            if catg == "ORDER_TABLE_REDU":
                order_table_ru = filename
            if catg == "ORDER_TABLE_REDL":
                order_table_rl = filename
            # Used in FLAMES UVES
            if catg == "FIB_ORD_TAB_REDU":
                order_table_rl = filename
            if catg == "FIB_ORD_TAB_REDL":
                order_table_ru = filename

            if catg == "ORDER_FLAT_BLUE" and first_raw_b is None:
                first_raw_b = filename
            if catg == "ORDER_FLAT_RED" and first_raw_r is None:
                first_raw_r = filename
            # used in FLAMES UVES
            if catg == "FIB_ORDEF_RED" and first_raw_r is None:
                first_raw_r = filename

        # Build and return the (one) file name list
        file_lists = []
        if order_table_b is not None and first_raw_b is not None:
            file_lists.append(
                {
                    "order_table": order_table_b,
                    "order_def": first_raw_b,
                    # 'main': order_table_b
                }
            )
        if order_table_rl is not None and first_raw_r is not None:
            file_lists.append(
                {
                    "order_table": order_table_rl,
                    "order_def": first_raw_r,
                    # 'main': order_table_rl
                }
            )
        if order_table_ru is not None and first_raw_r is not None:
            file_lists.append(
                {
                    "order_table": order_table_ru,
                    "order_def": first_raw_r,
                    # 'main': order_table_ru
                }
            )
        return file_lists

    def generate_panels(self, **kwargs):
        # ext here refers to the extension in the order_table
        raw_rotation_kwargs = {"rotate": 270}
        arm = UvesSetupInfo.get_arm_info(self.hdus[0]["order_table"])
        if "BLUE" in arm:
            exts = "PRIMARY"
        else:
            exts = []
            raw_rotation_kwargs["flip"] = "y"
            
            for file_lists in self.hdus:
                arm = UvesSetupInfo.get_arm_info(file_lists["order_table"])
                all_hdu_names = [hdu.name for hdu in file_lists["order_def"]]
                if "REDL" in arm:
                    exts.append("CCD-44")
                else:  # Red raw has one prim, two data exts
                    if "CCID-20" in all_hdu_names :
                        exts.append("CCID-20")
                    else :
                        exts.append("CCD-20")

        # Repair headers as required
        for file_lists in self.hdus:
            # Correct the headers in the raw_im
            for file_lists in self.hdus:
                correct_port_headers_uves(file_lists["order_def"], debug=False)

        new_panels = super().generate_panels(
            raw_im_ext=exts,
            master_im_ext="ORDER_TAB",
            main_image="raw",
            raw_im_rotation_kwargs=raw_rotation_kwargs,
        )
        for panel, panel_descr in new_panels.items():
            i = panel_descr["hdus_i"]
            order_table_hdul = self.hdus[i]["order_table"]
            order_def_hdul = self.hdus[i]["order_def"]
            panel_descr["report_description"] = (
                f"UVES orderdef panel - "
                f"{os.path.basename(panel_descr['order_def'])}, "
                f"{os.path.basename(panel_descr['order_table'])}, "
                f"{panel_descr['ext']}"
            )
            panel_descr["report_name"] = (
                f"UVES_{order_table_hdul['PRIMARY'].header.get('HIERARCH ESO PRO CATG').lower()}_"
                f"{(panel_descr['raw_im_ext'])}"
            )

            px = 0
            py = 0
            vspace = 0.3
            t1 = TextPlot(columns=1, v_space=vspace)
            fname = os.path.basename(str(order_table_hdul.filename()))
            col1 = (
                str(order_table_hdul["PRIMARY"].header.get("INSTRUME")),
                "EXTNAME: " + str(order_def_hdul[0].header.get("EXTNAME", "N/A")),
                "PRO CATG: "
                + str(order_table_hdul["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    order_table_hdul["PRIMARY"].header.get(
                        "HIERARCH ESO PRO REC1 RAW1 NAME"
                    )
                ),
            )
            t1.add_data(col1)
            panel.assign_plot(t1, px, py, xext=2)
            px = px + 2

            if "FIB" not in self.hdus[i]["order_table"]["PRIMARY"].header.get(
                "HIERARCH ESO PRO CATG"
            ):  # standard UVES
                t2 = TextPlot(columns=1, v_space=vspace, xext=1)

                hdr = order_def_hdul["PRIMARY"].header
                g1 = "HIERARCH ESO INS GRAT1 NAME"
                g2 = "HIERARCH ESO INS GRAT2 NAME"
                w1 = "HIERARCH ESO INS GRAT1 WLEN"
                w2 = "HIERARCH ESO INS GRAT2 WLEN"
                col2 = (
                    "DET.WIN1.BINX: " + str(hdr.get("HIERARCH ESO DET WIN1 BINX")),
                    "DET.WIN1.BINY: " + str(hdr.get("HIERARCH ESO DET WIN1 BINY")),
                    "DET.READ.SPEED: " + str(hdr.get("HIERARCH ESO DET READ SPEED")),
                    "INS.MODE: " + str(hdr.get("HIERARCH ESO INS MODE")),
                    "INS.GRAT1.NAME: " + str(hdr.get(g1))
                    if g1 in hdr
                    else "INS.GRAT2.NAME: " + str(hdr.get(g2)),
                    "INS.GRAT1.WLEN: " + str(hdr.get(w1))
                    if g1 in hdr
                    else "INS.GRAT2.WLEN: " + str(hdr.get(w2)),
                )
                t2.add_data(col2)
                panel.assign_plot(t2, px, py, xext=1)

            else:  # FLAMES-UVES
                t2 = TextPlot(columns=1, v_space=vspace, xext=1)

                hdr = order_def_hdul["PRIMARY"].header
                g1 = "HIERARCH ESO INS GRAT1 NAME"
                g2 = "HIERARCH ESO INS GRAT2 NAME"
                w1 = "HIERARCH ESO INS GRAT1 WLEN"
                w2 = "HIERARCH ESO INS GRAT2 WLEN"
                col2 = (
                    "DET.WIN1.BINX: "
                    + str(
                        order_table_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO DET WIN1 BINX"
                        )
                    ),
                    "DET.WIN1.BINY: "
                    + str(
                        order_table_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO DET WIN1 BINY"
                        )
                    ),
                    "DET.READ.SPEED: "
                    + str(
                        order_table_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO DET READ SPEED"
                        )
                    ),
                    "INS.SLIT3.PLATE: "
                    + str(
                        order_table_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO INS SLIT3 PLATE"
                        )
                    ),
                    "INS.GRAT1.WLEN: " + str(hdr.get(w1))
                    if g1 in hdr
                    else "INS.GRAT2.WLEN: " + str(hdr.get(w2)),
                )
                t2.add_data(col2)
                panel.assign_plot(t2, px, py, xext=1)

            # Alter the aspect of the primary plot image for blue
            main_img = panel.retrieve(0, 1)
            if exts[i] == "PRIMARY":
                main_img.aspect = 2.0
            else:
                main_img.aspect = 3.3

            rplot_size = 0.2
            # Additional plots for UVES
            # Residuals of fit against X pos
            res_x_plot = ScatterPlot(
                title="Residuals of fit (X)",
                x_label="X",
                y_label="Residual",
                markersize=rplot_size,
                legend=False,
            )
            res_x_plot.add_data(
                (
                    order_table_hdul[panel_descr["ext"]].data["X"],
                    order_table_hdul[panel_descr["ext"]].data["Residual"],
                ),
                label="X",
                color="black",
            )
            panel.assign_plot(res_x_plot, 2, 2)

            # Residuals of fit against Y pos
            res_y_plot = ScatterPlot(
                title="Residuals of fit (Y)",
                x_label="Y",
                y_label="Residual",
                markersize=rplot_size,
                legend=False,
            )
            res_y_plot.add_data(
                (
                    order_table_hdul[panel_descr["ext"]].data["Y"],
                    order_table_hdul[panel_descr["ext"]].data["Residual"],
                ),
                label="Y",
                color="black",
            )
            panel.assign_plot(res_y_plot, 3, 2)

            # YFIT vs X
            yfit_x_plot = ScatterPlot(
                title="Yfit",
                x_label="X",
                y_label="Yfit",
                markersize=rplot_size,
                legend=False,
            )
            yfit_x_plot.add_data(
                (
                    order_table_hdul[panel_descr["ext"]].data["X"],
                    order_table_hdul[panel_descr["ext"]].data["Yfit"],
                ),
                label="Y",
                color="black",
            )
            panel.assign_plot(yfit_x_plot, 3, 1)

        return new_panels


rep = UvesOrderdefFlatFieldReport()
