/*                                                                              *
 *   This file is part of the ESO UVES Pipeline                                 *
 *   Copyright (C) 2004,2005 European Southern Observatory                      *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or modify       *
 *   it under the terms of the GNU General Public License as published by       *
 *   the Free Software Foundation; either version 2 of the License, or          *
 *   (at your option) any later version.                                        *
 *                                                                              *
 *   This program is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *   GNU General Public License for more details.                               *
 *                                                                              *
 *   You should have received a copy of the GNU General Public License          *
 *   along with this program; if not, write to the Free Software                *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA       *
 *                                                                              */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup uves_idp  Recipe: Science Reduction
 *
 * This recipe reduces a science frame.
 * See man-page for details.
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <uves_utl_idp.h>
#include <uves_parameters.h>
#include <uves_recipe.h>
#include <uves.h>
#include <uves_error.h>
#include <uves_msg.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/
static int
uves_idp_define_parameters(cpl_parameterlist *parameters);

/*-----------------------------------------------------------------------------
                            Recipe standard code
 -----------------------------------------------------------------------------*/
#define cpl_plugin_get_info uves_idp_get_info
UVES_RECIPE_DEFINE(
    UVES_IDP_ID, UVES_IDP_DOM, uves_idp_define_parameters,
    "Jonas M. Larsen", "cpl@eso.org",
    uves_idp_desc_short,
    uves_idp_desc);

/**@{*/
/*-----------------------------------------------------------------------------
                              Functions code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    parameters        the parameterlist to fill
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
uves_idp_define_parameters(cpl_parameterlist *parameters)
{
    const char * recipe_id = make_str(UVES_IDP_ID);
    uves_idp_define_parameters_body(parameters, recipe_id);

    const char *subcontext = NULL;  // used by uves_par_new_enum macro
    uves_par_new_enum("velocity_correction", CPL_TYPE_STRING, "Whether to "
                      "apply velocity correction or not, and which if so",
                      "none", 3, "none", "barycentric", "heliocentric");

    return (cpl_error_get_code() != CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the command line options and execute the data reduction
  @param    parameters  the parameters list
  @param    frames      the frames list
  @return   CPL_ERROR_NONE if everything is ok
 */
/*----------------------------------------------------------------------------*/
static void
UVES_CONCAT2X(UVES_IDP_ID,exe)(cpl_frameset *frames,
        const cpl_parameterlist *parameters,
        const char *starttime)
{
    /* Do science reduction while pretending to be 'make_str(UVES_IDP_ID)' */
    uves_reduce_idp(frames, parameters, make_str(UVES_IDP_ID), starttime);

    return;
}


/**@}*/
