from edps import Job, List, ClassifiedFitsFile, JobParameters, get_parameter

from . import uves_keywords as kwd


################################
### Job processing functions ###
################################

### Functions to set recipe parameters depending on the properties of input files.


# The recipe parameter detmon_opt_lg.exts will be set to 0 or -1 depending
# on the input file format:
#  - all data in primary extension (NAXIS != 0)          --> detmon_opt_lg.exts=0,
#  - primary header plus several extensions (NAXIS == 0) --> detmon_opt_lg.exts=-1)
def detmon_extension(job: Job):
    extensions = f'detmon.{job.command}.exts'
    nax = job.input_files[0].get_keyword_value("NAXIS", None)
    extension = -1 if nax == 0 else 0
    job.parameters.recipe_parameters[extensions] = extension


def object_type(job: Job):
    # Setting some recipe parameters in uves_obs_scired depending if the main input is extended or point-like

    reduce_extract_method = f'{job.command}.reduce.extract.method'
    reduce_backsub_method = f'{job.command}.reduce.backsub.mmethod'
    dp_types = [f.get_keyword_value(kwd.dpr_type, None) for f in job.input_files]
    if "OBJECT,EXTENDED" in dp_types or "OBJECT,EXTENDED,EXTENDED" in dp_types:
        job.parameters.recipe_parameters[reduce_extract_method] = "2d"
        job.parameters.recipe_parameters[reduce_backsub_method] = "minimum"
    elif "OBJECT,POINT" in dp_types or "OBJECT,POINT,POINT" in dp_types:
        job.parameters.recipe_parameters[reduce_extract_method] = "optimal"
        job.parameters.recipe_parameters[reduce_backsub_method] = "median"


def set_minmaxlines(job: Job):
    # Set the maxlines recipe parameter in uves_cal_wavecal, depending on the INS.GRAT2.WLEN and SLIT3 width.
    maxlines = f'{job.command}.search.maxlines'
    minlines = f'{job.command}.search.minlines'

    wavelenghts = [f.get_keyword_value(kwd.grat2_wlen, None) for f in job.input_files]
    widths = [f.get_keyword_value(kwd.slit3_wid, None) for f in job.input_files]
    catgs = [f.get_keyword_value(kwd.dpr_type, None) for f in job.input_files]
    chips = [f.get_keyword_value(kwd.det_chips, None) for f in job.input_files]
    read_speed = [f.get_keyword_value(kwd.det_readout, None) for f in job.input_files]
    binx = [f.get_keyword_value(kwd.det_binx, None) for f in job.input_files]
    biny = [f.get_keyword_value(kwd.det_biny, None) for f in job.input_files]
    insmode = [f.get_keyword_value(kwd.ins_mode, None) for f in job.input_files]
    job.parameters.recipe_parameters[maxlines] = 0
    job.parameters.recipe_parameters[minlines] = 0
    if 580 in wavelenghts and 0.3 in widths and 2 in chips and "LAMP,WAVE" in catgs:
        job.parameters.recipe_parameters[maxlines] = 1200
        job.parameters.recipe_parameters[minlines] = 1000
    if 860 in wavelenghts and ("LAMP,WAVE,OzPoz" in catgs or "LAMP,WAVE,SimCal" in catgs):
        job.parameters.recipe_parameters[minlines] = 900

    if ("2pts/225kHz/lg" in read_speed and 1 in binx and 1 in biny and "RED" in insmode and "LAMP,WAVE" in catgs and
            0.3 in widths and 580 in wavelenghts):
        job.parameters.recipe_parameters[maxlines] = 1500


def set_threshold(job: Job):
    # set the recipe parameter trace.minthresh depending on the values of det.chis and ins.grat2.len
    minthresh = f'{job.command}.trace.minthresh'
    chips = [f.get_keyword_value(kwd.det_chips, None) for f in job.input_files]
    wavelenghts = [f.get_keyword_value(kwd.grat2_wlen, None) for f in job.input_files]
    job.parameters.recipe_parameters[minthresh] = 0.2  # recipe default
    # For these setups traces are fainer and the threshold need to be decreased
    # to select a sufficient number of points to trace the orders properly
    if (580 in wavelenghts or 520 in wavelenghts) and 2 in chips:
        job.parameters.recipe_parameters[minthresh] = 0.1


def set_threshold_mos(job: Job):
    # set the recipe parameter trace.minthresh depending on the values of det.chis and ins.grat2.len
    minthresh = f'{job.command}.trace.minthresh'
    readout = [f.get_keyword_value(kwd.det_readout, None) for f in job.input_files]
    binx = [f.get_keyword_value(kwd.det_binx, None) for f in job.input_files]
    biny = [f.get_keyword_value(kwd.det_biny, None) for f in job.input_files]
    wavelengths = [f.get_keyword_value(kwd.grat2_wlen, None) for f in job.input_files]
    ins3_plate = [f.get_keyword_value(kwd.slit3_plate, None) for f in job.input_files]

    job.parameters.recipe_parameters[minthresh] = 0.2  # recipe default
    # For these 2 setups the minimum trace threshold needs to be decreased. Traces are fainter
    # and a lower threshold have to be used to select a number of points for a propert order tracing
    setup_1 = (580 in wavelengths and '2pts/225kHz/lg' in readout and 1 in binx and 1 in biny and 1 in ins3_plate)
    setup_2 = (580 in wavelengths and '2pts/225kHz/lg' in readout and 1 in binx and 1 in biny and 2 in ins3_plate)
    if setup_1 or setup_2:
        job.parameters.recipe_parameters[minthresh] = 0.1


# Set recipe parameter save_flat_size for input files that contain the string 6FIB
# in ins.slit3.mode
def set_flat_size(job: Job):
    save_flat_size = f'{job.command}.save_flat_size'
    job.parameters.recipe_parameters[save_flat_size] = -1
    slit_mode = [f.get_keyword_value(kwd.slit3_mode, None) for f in job.input_files]
    if "6FIB" in slit_mode:
        job.parameters.recipe_parameters[save_flat_size] = 1


##############################################################################
### Function that changes the classification tags of the files             ###
##############################################################################

# Need to drop the ending character 1/2/3 for MASTER_SFLAT_REDL/U-1/2/3
# otherwise the recipe does not recognize them.
def change_master_sflat(job: Job):
    for f in job.associated_files:
        if f.classification.startswith("MASTER_SFLAT_REDL"):
            f.classification = "MASTER_SFLAT_REDL"
        elif f.classification.startswith("MASTER_SFLAT_REDU"):
            f.classification = "MASTER_SFLAT_REDU"


##############################################################################
### Functions to assign a value to a dynamic parameter based on the        ###
### properties of the main input files                                     ###
##############################################################################

def which_arm(files: List[ClassifiedFitsFile]):
    # Determines if the observation was done on the Blue or Red arm, and set the return value
    # accordingly.
    det = files[0].get_keyword_value(kwd.det_chips, None)
    path = files[0].get_keyword_value(kwd.ins_path, None)
    return "Blue" if ((det == 1) or (path == "BLUE")) else "Red" if ((det == 2) or (path == "RED")) else None


def which_lamp(files: List[ClassifiedFitsFile]):
    # Check which lamp was used, IODINE or REGULAR, and set the return value accordingly
    a = files[0].get_keyword_value(kwd.ins_lamp11_st, None)  # INS.LAMP11.ST=T
    b = files[0].get_keyword_value(kwd.ins_lamp11_name, None)
    c = files[0].get_keyword_value(kwd.ins_opti1_name, None)
    return "IODINE" if (a is True and b == "IODINE_CELL" and c == "IN") else "REGULAR"


def which_observation_type(files: List[ClassifiedFitsFile]):
    # Check whether the observation type is SIM or OBS nd set the return value accordingly
    sim = files[0].get_keyword_value(kwd.dpr_type, None)
    return "SIM" if sim == "OBJECT,SimCal" else "OBS"


###################################################################
### Functions that define conditions depending on the           ###
### values of the the dynamic parameters                        ###
###################################################################


def is_blue(params: JobParameters) -> bool:
    return get_parameter(params, "arm_used") == "Blue"


def is_red(params: JobParameters) -> bool:
    return get_parameter(params, "arm_used") == "Red"


def use_iodine_flat(params: JobParameters) -> bool:
    return get_parameter(params, "use_iodine_flat") == "YES" and get_parameter(params, "lamp_used") == "IODINE"


def use_iodine_flat_blue(params: JobParameters) -> bool:
    return is_blue(params) and use_iodine_flat(params)


def use_iodine_flat_red(params: JobParameters) -> bool:
    return is_red(params) and use_iodine_flat(params)


def is_sim(params: JobParameters) -> bool:
    return get_parameter(params, "simulation") == "SIM"


def combine_flats(params: JobParameters) -> bool:
    return get_parameter(params, "combine_flats") == "TRUE"


def use_combined_flat_blue(params: JobParameters) -> bool:
    return is_blue(params) and combine_flats(params)


def use_regular_flat_blue(params: JobParameters) -> bool:
    return is_blue(params) and not combine_flats(params)


def use_combined_flat_in_science_blue(params: JobParameters) -> bool:
    return use_combined_flat_blue(params) and not use_iodine_flat(params)


def use_regular_flat_in_science_blue(params: JobParameters) -> bool:
    return use_regular_flat_blue(params) and not use_iodine_flat(params)


def use_regular_flat_in_science_red(params: JobParameters) -> bool:
    return is_red(params) and not use_iodine_flat(params)


def use_flats_in_arc(params: JobParameters) -> bool:
    return get_parameter(params, "use_flats_in_arc") == "YES"


def use_combined_flat_in_arc_blue(params: JobParameters) -> bool:
    return is_blue(params) and use_flats_in_arc(params) and combine_flats(params)


def use_regular_flat_in_arc_blue(params: JobParameters) -> bool:
    return is_blue(params) and use_flats_in_arc(params) and not combine_flats(params)


def use_regular_flat_in_arc_red(params: JobParameters) -> bool:
    return is_red(params) and use_flats_in_arc(params)


def use_master_response(params: JobParameters) -> bool:
    return get_parameter(params, "response") in ("master", "MASTER")


def use_master_response_blue(params: JobParameters) -> bool:
    return is_blue(params) and use_master_response(params)


def use_master_response_red(params: JobParameters) -> bool:
    return is_red(params) and use_master_response(params)


def use_night_response(params: JobParameters) -> bool:
    return get_parameter(params, "response") in ("night", "NIGHT")


def use_night_response_blue(params: JobParameters) -> bool:
    return is_blue(params) and use_night_response(params)


def use_night_response_red(params: JobParameters) -> bool:
    return is_red(params) and use_night_response(params)
