from edps import QC1_CALIB, SCIENCE, CALCHECKER, QC0
from edps import task, alternative_associated_inputs, ReportInput

from .uves_datasources import *
from .uves_job_processing import *
from .uves_monitoring import monitoring
from .uves_science import uves_science_slit

__title__ = "UVES workflow"

IDP = "idp"

# --- PROCESSING TASKS -------------------------------------------------------------------

# --- SUBWORKFLOW with tasks used to monitor the detector and the instrument focus ---
cdalign, ccdtest, focus = monitoring()
# ------------------------------------------------------------------------------------

# ------ Process BIAS frames ---------------------------------------------------------
bias = (task("bias")
        .with_recipe("uves_cal_mbias")
        .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
        .with_report('uves_master_bias', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
        .with_main_input(raw_bias)
        .with_associated_input(pattern_detection_redl, condition=is_red, min_ret=0)
        .with_associated_input(pattern_detection_redu, condition=is_red, min_ret=0)
        .with_associated_input(pattern_detection_blue, condition=is_blue, min_ret=0)
        .with_dynamic_parameter("arm_used", which_arm)
        .with_meta_targets([QC1_CALIB])
        .build())

# The bias to associate depends on the arm (RED/BLUE) of the input files.
# Science exposures need different rules to associate a bias with respect to other calibrations
# (validity time range is different).
bias_calibrations = (alternative_associated_inputs()
                     .with_associated_input(bias, [MASTER_BIAS_BLUE], condition=is_blue)
                     .with_associated_input(bias, [MASTER_BIAS_REDL, MASTER_BIAS_REDU], condition=is_red))

bias_calibrations_for_science = (alternative_associated_inputs()
                                 .with_associated_input(bias, [MASTER_BIAS_BLUE], condition=is_blue,
                                                        match_rules=associate_bias_to_science)
                                 .with_associated_input(bias, [MASTER_BIAS_REDL, MASTER_BIAS_REDU], condition=is_red,
                                                        match_rules=associate_bias_to_science))

# ----------------------------------------------------------------------------------------


# ------ Process DARK frames -------------------------------------------------------------
# measuring dark current
dark = (task("dark")
        .with_recipe("uves_cal_mdark")
        .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
        .with_report('uves_master_dark', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
        .with_main_input(raw_dark)
        .with_alternative_associated_inputs(bias_calibrations)
        .with_dynamic_parameter("arm_used", which_arm)  # defines the arm of the input file
        .with_input_filter(MASTER_BIAS_REDL, MASTER_BIAS_REDU, MASTER_BIAS_BLUE)
        .with_meta_targets([QC1_CALIB])
        .build())
# measuring "parasitic" light (for monitoring only)
pdark = (task("pdark")
         .with_recipe("uves_cal_mdark")
         .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
         .with_report('uves_master_dark', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
         .with_main_input(raw_pdark)
         .with_alternative_associated_inputs(bias_calibrations)
         .with_dynamic_parameter("arm_used", which_arm)
         .with_input_filter(MASTER_BIAS_REDL, MASTER_BIAS_REDU, MASTER_BIAS_BLUE)
         .with_meta_targets([QC1_CALIB])
         .build())

# ------ Process DARK frames -------------------------------------------------------------


# ------ Tasks for prediction the position of spectra -----
# Generation of guess order and line tables for slit spectra
predict = (task("predict")
           .with_recipe("uves_cal_predict")
           .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
           .with_report('uves_formatcheck', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
           .with_main_input(raw_arc_lamp_form)
           .with_associated_input(line_refer_table)
           .with_associated_input(masterform_redl, condition=is_red, min_ret=0)
           .with_associated_input(masterform_redu, condition=is_red, min_ret=0)
           .with_associated_input(masterform_blue, condition=is_blue, min_ret=0)
           .with_dynamic_parameter("arm_used", which_arm)
           .with_meta_targets([QC1_CALIB])
           .build())

# Generation of guess order and line tables for fibre spectra
predict_mos = (task("predict_mos")
               .with_recipe("flames_cal_predict")
               .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
               .with_report('uves_formatcheck', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
               .with_main_input(raw_arc_lamp_form_mos)
               .with_associated_input(line_refer_table)
               .with_associated_input(masterform_redl_mos, min_ret=0)
               .with_associated_input(masterform_redu_mos, min_ret=0)
               .with_meta_targets([QC1_CALIB])
               .build())

# Several tasks need the order guess tables which the predict task generates.
# The associated input for these tasks is defined here.
#
# The correct calibration to associate depends on the arm (RED/BLUE) of the input data.
predict_calibrations = (alternative_associated_inputs()
                        .with_associated_input(predict, [ORDER_GUESS_TABLE_BLUE], condition=is_blue)
                        .with_associated_input(predict, [ORDER_GUESS_TABLE_REDL, ORDER_GUESS_TABLE_REDU],
                                               condition=is_red))

# The arc task needs the line guess tables from the output of the predict task. Therefore, we
# define here a separate version of the associated input that contains the line guess table.
#
# The correct calibration to associate depends on the arm (RED/BLUE) of the input data.
predict_for_arcs = (alternative_associated_inputs()
                    .with_associated_input(predict, [LINE_GUESS_TABLE_BLUE], condition=is_blue)
                    .with_associated_input(predict, [LINE_GUESS_TABLE_REDL, LINE_GUESS_TABLE_REDU],
                                           condition=is_red))

# ------------------------------------------------------------------------------------


# -------- Tasks to define the order positions  --------------------------------------
# For slit spectra
orderpos = (task("orderpos")
            .with_recipe("uves_cal_orderpos")
            .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
            .with_report('uves_orderdef_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
            .with_main_input(raw_order_flat)
            .with_alternative_associated_inputs(predict_calibrations)
            .with_dynamic_parameter("arm_used", which_arm)
            .with_meta_targets([QC1_CALIB, CALCHECKER])
            .build())

# For fibre spectra
orderpos_mos = (task("orderpos_mos")
                .with_recipe("flames_cal_orderpos")
                .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
                .with_report('uves_orderdef_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
                .with_main_input(raw_order_flat_mos)
                .with_associated_input(predict_mos, [FIB_ORDER_GUE_REDL, FIB_ORDER_GUE_REDU])
                .with_meta_targets([QC1_CALIB, CALCHECKER])
                .build())

# The calibration with table of order position to associate depends on the arm (RED/BLUE).
orderpos_calibrations = (alternative_associated_inputs()
                         .with_associated_input(orderpos, [ORDER_TABLE_BLUE], condition=is_blue)
                         .with_associated_input(orderpos, [ORDER_TABLE_REDL, ORDER_TABLE_REDU],
                                                condition=is_red))

# Validity range of orderpos for response calibrations are different
orderpos_calibrations_for_standards = (alternative_associated_inputs()
                                       .with_associated_input(orderpos, [ORDER_TABLE_BLUE],
                                                              match_rules=orderpos_response_rules,
                                                              condition=is_blue)
                                       .with_associated_input(orderpos, [ORDER_TABLE_REDL, ORDER_TABLE_REDU],
                                                              match_rules=orderpos_response_rules,
                                                              condition=is_red))

# ------------------------------------------------------------------------------------

# ------ Tasks for processing the flat fields  ---------------------------------------

# Regular flats (slit spectra).
flat = (task("flat")
        .with_recipe("uves_cal_mflat")
        .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
        .with_report('uves_echelle_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
        .with_main_input(raw_flat)
        .with_alternative_associated_inputs(bias_calibrations)
        .with_alternative_associated_inputs(orderpos_calibrations)
        .with_dynamic_parameter("arm_used", which_arm)
        .with_meta_targets([QC1_CALIB, CALCHECKER])
        .build())

# Iodine-cell flats (slit spectra).
iflat = (task("iflat")
         .with_recipe("uves_cal_mflat")
         .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
         .with_report('uves_echelle_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
         .with_main_input(raw_iflat)
         .with_alternative_associated_inputs(bias_calibrations)
         .with_alternative_associated_inputs(orderpos_calibrations)
         .with_dynamic_parameter("arm_used", which_arm)
         .with_input_filter(MASTER_BIAS_REDL, MASTER_BIAS_REDU, MASTER_BIAS_BLUE,
                            ORDER_TABLE_BLUE, ORDER_TABLE_REDL, ORDER_TABLE_REDU)
         .with_meta_targets([QC1_CALIB, CALCHECKER])
         .build())

# 2D lamp flats, only for BLUE arm 346 setting  (slit spectra).
dflat = (task("dflat")
         .with_recipe("uves_cal_mflat")
         .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
         .with_report('uves_echelle_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
         .with_main_input(raw_dflat)
         .with_alternative_associated_inputs(bias_calibrations)
         .with_alternative_associated_inputs(orderpos_calibrations)
         .with_dynamic_parameter("arm_used", which_arm)
         .with_meta_targets([QC1_CALIB, CALCHECKER])
         .build())

# Task to combine regular and 2D blue lamp flats (slit spectra)
combined_flat = (task("combined_flat")
                 .with_recipe("uves_cal_mflat_combine")
                 .with_main_input(flat)
                 .with_associated_input(dflat, min_ret=0)
                 .with_associated_input(orderpos, [ORDER_TABLE_BLUE], condition=is_blue)
                 .with_dynamic_parameter("arm_used", which_arm)
                 .with_input_filter(MASTER_FLAT_BLUE, MASTER_FLAT_REDL, MASTER_FLAT_REDU, ORDER_TABLE_BLUE,
                                    ORDER_TABLE_REDL, ORDER_TABLE_REDU).build())
# flats for fibre spectra
flat_mos = (task("flat_mos")
            .with_recipe("flames_cal_mkmaster")
            .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
            .with_report('uves_fibre_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
            .with_main_input(raw_sflat_mos)
            .with_associated_input(bias, [MASTER_BIAS_REDL, MASTER_BIAS_REDU])
            .with_associated_input(orderpos_mos, [FIB_ORD_TAB_REDL, FIB_ORD_TAB_REDU])
            .with_meta_targets([QC1_CALIB, CALCHECKER])
            .build())

# The flat calibration to associate depends on the arm (RED/BLUE) and on user's choice (combined or regular flats),
# expressed in the parameter file (parameter "combine_flats": "TRUE"/"FALSE")
flat_calibrations = (alternative_associated_inputs()
                     .with_associated_input(combined_flat, [MASTER_FLAT_BLUE], condition=use_combined_flat_blue)
                     .with_associated_input(flat, [MASTER_FLAT_BLUE], condition=use_regular_flat_blue)
                     .with_associated_input(flat, [MASTER_FLAT_REDL, MASTER_FLAT_REDU], condition=is_red))

# For the arcs, the flat calibration to associate depends on the arm (RED/BLUE) and on user choice in the parameter file.
# The choice of the flat to use (regular or combined) is determined by the parameter "combine_flats": "TRUE"/"FALSE"
# The choice of using flat fields for arcs is determined by the parameter parameter "use_flats_in_arc": "YES"/"NO"
flat_for_arcs = (alternative_associated_inputs()
                 .with_associated_input(combined_flat, [MASTER_FLAT_BLUE],
                                        condition=use_combined_flat_in_arc_blue, min_ret=0)
                 .with_associated_input(flat, [MASTER_FLAT_BLUE],
                                        condition=use_regular_flat_in_arc_blue, min_ret=0)
                 .with_associated_input(flat, [MASTER_FLAT_REDL, MASTER_FLAT_REDU],
                                        condition=use_regular_flat_in_arc_red, min_ret=0))

# For science reduction, the flat to associate depends on the arm (RED/BLUE) and on user's choice
# The choice of the flats to use (regular, combined, or iodine) is determined in the parameter file
# combine_flats: "FALSE" (has preference)
# use_iodine_flat: "YES"
flat_for_science = (alternative_associated_inputs()
                    .with_associated_input(combined_flat, [MASTER_FLAT_BLUE],
                                           condition=use_combined_flat_in_science_blue,
                                           match_rules=associate_flat_to_science)
                    .with_associated_input(flat, [MASTER_FLAT_BLUE], condition=use_regular_flat_in_science_blue,
                                           match_rules=associate_flat_to_science)
                    .with_associated_input(flat, [MASTER_FLAT_REDL, MASTER_FLAT_REDU],
                                           condition=use_regular_flat_in_science_red,
                                           match_rules=associate_flat_to_science)
                    .with_associated_input(iflat, [MASTER_IFLAT_BLUE],
                                           condition=use_iodine_flat_blue, match_rules=associate_flat_to_science)
                    .with_associated_input(iflat, [MASTER_IFLAT_REDL, MASTER_IFLAT_REDU],
                                           condition=use_iodine_flat_red, match_rules=associate_flat_to_science))
# ------------------------------------------------------------------------------------


# ---- Fibre & order position determination task --------------------------------

sff_ofpos_mos = (task("sff_ofpos_mos")
                 .with_recipe("flames_cal_prep_sff_ofpos")
                 .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
                 .with_report('uves_fibre_flatfield', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
                 .with_main_input(raw_fib_ff_odd_mos)
                 .with_associated_input(raw_fib_ff_even_mos)
                 .with_associated_input(raw_fib_ff_all_mos)
                 .with_associated_input(bias, [MASTER_BIAS_REDL, MASTER_BIAS_REDU])
                 .with_associated_input(orderpos_mos,
                                        [FIB_ORD_TAB_REDL, FIB_ORDEF_REDL,
                                         FIB_ORD_TAB_REDU, FIB_ORDEF_REDU],
                                        match_rules=orderpos_mos_rules)
                 .with_associated_input(flat_mos, [MASTER_SFLAT_REDL1, MASTER_SFLAT_REDU1,
                                                   MASTER_SFLAT_REDL2, MASTER_SFLAT_REDU2,
                                                   MASTER_SFLAT_REDL3, MASTER_SFLAT_REDU3])
                 .with_job_processing(set_flat_size)
                 .with_job_processing(change_master_sflat)
                 .with_meta_targets([QC1_CALIB, CALCHECKER])
                 .build())
# ------------------------------------------------------------------------------------


# ------ Tasks for wavelength calibration --------------------------------------------
# slit spectra
arc = (task("arc")
       .with_recipe("uves_cal_wavecal")
       .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
       .with_report('uves_wavelength', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
       .with_main_input(raw_arc_lamp)
       .with_alternative_associated_inputs(predict_for_arcs)
       .with_alternative_associated_inputs(orderpos_calibrations)
       .with_associated_input(line_refer_table)
       .with_associated_input(line_intmon_table)
       .with_alternative_associated_inputs(flat_for_arcs)
       .with_dynamic_parameter("arm_used", which_arm)
       .with_job_processing(set_minmaxlines)
       .with_meta_targets([QC1_CALIB, CALCHECKER])
       .build())

# fibre spectra
arc_mos = (task("arc_mos")
           .with_recipe("flames_cal_wavecal")
           .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
           .with_report('uves_wavelength', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
           .with_main_input(raw_arc_lamp_mos)
           .with_associated_input(predict_mos,
                                  [FIB_ORDER_GUE_REDL, FIB_ORDER_GUE_REDU, FIB_LIN_GUE_TAB_REDU,
                                   FIB_LIN_GUE_TAB_REDL])
           .with_associated_input(sff_ofpos_mos, [FIB_ORDEF_TABLE_REDL, FIB_ORDEF_TABLE_REDU])
           .with_associated_input(line_refer_table)
           .with_associated_input(corvel_mask)
           .with_associated_input(line_intmon_table)
           .with_job_processing(set_minmaxlines)
           .with_meta_targets([QC1_CALIB, CALCHECKER])
           .build())

# The arc calibration to associate depends on the arm (RED/BLUE)
arc_calibrations = (alternative_associated_inputs()
                    .with_associated_input(arc, [LINE_TABLE_BLUE], condition=is_blue)
                    .with_associated_input(arc, [LINE_TABLE_REDL, LINE_TABLE_REDU], condition=is_red))
# arcs for science and calibration files have different validity ranges
arc_calibrations_for_science = (alternative_associated_inputs()
                                .with_associated_input(arc, [LINE_TABLE_BLUE], condition=is_blue,
                                                       match_rules=associate_arc_to_science)
                                .with_associated_input(arc, [LINE_TABLE_REDL, LINE_TABLE_REDU], condition=is_red,
                                                       match_rules=associate_arc_to_science))
# ------------------------------------------------------------------------------------

# Telluric lamp flat fields (for instrument monitoring only)
tflat = (task("tflat")
         .with_recipe("uves_cal_tflat")
         .with_main_input(raw_tflat)
         .with_alternative_associated_inputs(bias_calibrations)
         .with_alternative_associated_inputs(orderpos_calibrations)
         .with_alternative_associated_inputs(flat_calibrations)
         .with_alternative_associated_inputs(arc_calibrations)
         .with_dynamic_parameter("arm_used", which_arm)
         .with_input_filter(MASTER_BIAS_REDL, MASTER_BIAS_REDU, MASTER_BIAS_BLUE,
                            ORDER_TABLE_BLUE, ORDER_TABLE_REDL, ORDER_TABLE_REDU,
                            LINE_TABLE_REDL, LINE_TABLE_REDU, LINE_TABLE_BLUE,
                            MASTER_FLAT_REDL, MASTER_FLAT_REDU, MASTER_FLAT_BLUE)
         .with_meta_targets([QC1_CALIB])
         .build())
# ------------------------------------------------------------------------------------

# ---------  Task for response curve computation -------------------------------------
response = (task("response")
            .with_recipe("uves_cal_response")
            .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
            .with_report("uves_specphot_std", ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
            .with_main_input(raw_flux_std)
            .with_alternative_associated_inputs(bias_calibrations)
            .with_alternative_associated_inputs(orderpos_calibrations_for_standards)
            .with_alternative_associated_inputs(flat_calibrations)
            .with_alternative_associated_inputs(arc_calibrations)
            .with_associated_input(resp_fit_points_catalog)
            .with_associated_input(flux_std_table)
            .with_associated_input(extcoeff_table)
            .with_dynamic_parameter("arm_used", which_arm)
            .with_meta_targets([QC0, QC1_CALIB])
            .build())

# Task to process telluric standards
telluric = (task("telluric")
            .with_recipe("uves_cal_response")
            .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
            .with_report("uves_specphot_std", ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
            .with_main_input(raw_telluric)
            .with_alternative_associated_inputs(bias_calibrations)
            .with_alternative_associated_inputs(orderpos_calibrations_for_standards)
            .with_alternative_associated_inputs(flat_calibrations)
            .with_alternative_associated_inputs(arc_calibrations)
            .with_associated_input(resp_fit_points_catalog)
            .with_associated_input(flux_std_table)
            .with_associated_input(extcoeff_table)
            .with_dynamic_parameter("arm_used", which_arm)
            .with_meta_targets([QC0, QC1_CALIB])
            .build())

# Task to process standards for efficiency only
efficiency = (task("efficiency")
              .with_recipe("uves_cal_response")
              .with_report('uves_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
              .with_report("uves_specphot_std", ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
              .with_main_input(raw_efficiency)
              .with_alternative_associated_inputs(bias_calibrations)
              .with_alternative_associated_inputs(orderpos_calibrations_for_standards)
              .with_alternative_associated_inputs(flat_calibrations)
              .with_alternative_associated_inputs(arc_calibrations)
              .with_associated_input(resp_fit_points_catalog)
              .with_associated_input(flux_std_table)
              .with_associated_input(extcoeff_table)
              .with_dynamic_parameter("arm_used", which_arm)
              .with_meta_targets([QC0, QC1_CALIB])
              .build())

# The associated response curve calibration depends on the arm (RED/BLUE) and whether one wants to use a
# response curve provided by CalSelector or the one computed using the standard star observed at night. This
# choice is expressed in the parameter file (parameter:  response: "master"/"night")
# If the parameter response is "master", then response_calibrations are not associated.
# master_response are associated instead (the association is defined in the object task).
response_calibrations = (alternative_associated_inputs()
                         .with_associated_input(response, [INSTR_RESPONSE_FINE_BLUE],
                                                condition=use_night_response_blue, min_ret=0)
                         .with_associated_input(response,
                                                [INSTR_RESPONSE_FINE_REDL, INSTR_RESPONSE_FINE_REDU],
                                                condition=use_night_response_red, min_ret=0))

# ---- Science processing tasks  ----------------------------------------------------

# Subworkflow to reduce long slit spectra of generic science targets and reflected solar spectra
science_slit, solar_slit = uves_science_slit(bias_calibrations_for_science, orderpos_calibrations,
                                             flat_calibrations, flat_for_science, arc_calibrations_for_science,
                                             response_calibrations)

# Task to reduce fibre observations
science_mos = (task("object_mos")
               .with_recipe("flames_obs_scired")
               .with_main_input(raw_fib_science_mos)
               .with_associated_input(bias, [MASTER_BIAS_REDL, MASTER_BIAS_REDU], match_rules=associate_bias_to_science)
               .with_associated_input(sff_ofpos_mos, [FIB_FF_BPC_REDL, FIB_FF_COM_REDL,
                                                      FIB_FF_DTC_REDL, FIB_FF_NOR_REDL,
                                                      FIB_FF_NSG_REDL, FIB_FF_SGC_REDL,
                                                      FIB_FF_BPC_REDU, FIB_FF_COM_REDU,
                                                      FIB_FF_DTC_REDU, FIB_FF_NOR_REDU,
                                                      FIB_FF_NSG_REDU, FIB_FF_SGS_REDU,
                                                      SLIT_FF_BPC_REDL, SLIT_FF_COM_REDL,
                                                      SLIT_FF_DTC_REDL, SLIT_FF_NOR_REDL,
                                                      SLIT_FF_BNC_REDL, SLIT_FF_SGC_REDL,
                                                      SLIT_FF_BPC_REDU, SLIT_FF_COM_REDU,
                                                      SLIT_FF_DTC_REDU, SLIT_FF_NOR_REDU,
                                                      SLIT_FF_BNC_REDU, SLIT_FF_SGC_REDU,
                                                      FIB_ORDEF_TABLE_REDL, FIB_ORDEF_TABLE_REDU])
               .with_associated_input(arc_mos, [FIB_LINE_TABLE_REDL, FIB_LINE_TABLE_REDU],
                                      match_rules=associate_arc_to_science_mos)
               .with_associated_input(corvel_mask, condition=is_sim, min_ret=0)
               .with_dynamic_parameter("simulation", which_observation_type)
               .with_meta_targets([SCIENCE, QC0, CALCHECKER, IDP])
               .build())
# ------------------------------------------------------------------------------------
