from adari_core.data_libs.master_rawdisp import MasterRawdispReport
import os
import numpy as np
from .uves_util import UvesSetupInfo

from . import UvesReportMixin


class UvesRawdispReport(UvesReportMixin, MasterRawdispReport):
    raw_extensions = {
        "PRIMARY": "blue",
        "CCD-20": "redu",
        "CCD-44": "redl",
        # Special case for old red data
        "CCID-20": "redu",
        # Special case for red 2x3 binning
        "WIN1": "2x3bin1",
        "WIN2": "2x3bin2",
    }

    def __init__(self):
        super().__init__("uves_rawdisp")
        self.extensions = []
        self.tasks = {
            #
            "BIAS_BLUE": "bias",
            "BIAS_RED": "bias",
            #
            "DARK_BLUE": "dark",
            "DARK_RED": "dark",
            #
            "PDARK_BLUE": "dark",
            "PDARK_RED": "dark",
            #
            "OFF_RAW": "ccdtest",
            "ON_RAW": "ccdtest",
            #
            "CD_ALIGN_BLUE": "cd_align_blue",
            "CD_ALIGN_RED": "cd_align_red",
            #
            "ARC_LAMP_FORM_BLUE": "predict",
            "ARC_LAMP_FORM_RED": "predict",
            #
            "ORDER_FLAT_BLUE": "orderpos",
            "ORDER_FLAT_RED": "orderpos",
            #
            "FLAT_BLUE": "flat",
            "FLAT_RED": "flat",
            "IFLAT_BLUE": "iflat",
            "IFLAT_RED": "iflat",
            "DFLAT_BLUE": "dflat",
            "DFLAT_RED": "dflat",
            #
            "ARC_LAMP_BLUE": "arc",
            "ARC_LAMP_RED": "arc",
            #
            "STANDARD_BLUE": "response",
            "STANDARD_RED": "response",
            #
            "FIB_ARC_LAMP_FORM_RED": "predict_mos",
            #
            "FIB_ORDEF_RED": "orderpos_mos",
            #
            "SFLAT_RED": "flat_mos",
            #
            "FIB_FF_ODD_RED": "sff_ofpos_mos",
            "FIB_FF_EVEN_RED": "sff_ofpos_mos",
            "FIB_FF_ALL_RED": "sff_ofpos_mos",
            #
            "FIB_ARC_LAMP_RED": "arc_mos",
        }
        self.setup_info = UvesSetupInfo

    def parse_sof(self):
        # we building multiple report sets, so we append multiple reports to file_lists
        # get a list of tags
        tags = list(self.tasks.keys())
        added = {}
        file_lists = []
        for filename, catg in self.inputs:
            if catg in tags:
                if filename is not None and catg not in added:
                    file_lists.append({"filename": filename})
                    added[catg] = self.tasks[catg]
                    self.sof_tag.append(catg)
        return file_lists

    def get_extensions(self):
        """Find the data extensions required for each UVES files.

        Description
        -----------
        After the SOF has been parsed, this method iterates over the different
        HDUS files to find which extension(s) contains the data.
        """
        new_list_hdus = []
        new_list_tags = []
        for i, filedict in enumerate(self.hdus):
            hdul = filedict["filename"]
            all_hdu_names = [hdu.name for hdu in hdul]
            detector_name = hdul["PRIMARY"].header.get("HIERARCH ESO DET NAME")
            if "uvesb" in detector_name.lower():
                # Check if the data was read in fast-mode (binning)
                if "WIN1" in all_hdu_names:
                    # Stitch both extensions and save them in the PRIMARY
                    # to ensure consistency between all blue files
                    data = np.concatenate(
                        (hdul["WIN1"].data, hdul["WIN2"].data), axis=1
                    )
                    hdul["PRIMARY"].data = data
                self.extensions.append("PRIMARY")
                new_list_hdus.append(filedict)
                new_list_tags.append(self.sof_tag[i])
            else:
                # Check if the data was read in fast-mode (binning)
                if "WIN1" in all_hdu_names:
                    self.extensions.append("WIN1")
                    new_list_tags.append(self.sof_tag[i])
                    new_list_hdus.append(filedict)
                    self.extensions.append("WIN2")
                    new_list_hdus.append(filedict)
                    new_list_tags.append(self.sof_tag[i])
                else:
                    self.extensions.append("CCD-44")
                    new_list_hdus.append(filedict)
                    new_list_tags.append(self.sof_tag[i])
                    # Check if this is old data format
                    if "CCID-20" in all_hdu_names:
                        self.extensions.append("CCID-20")
                        new_list_hdus.append(filedict)
                        new_list_tags.append(self.sof_tag[i])
                    else:
                        self.extensions.append("CCD-20")
                        new_list_hdus.append(filedict) 
                        new_list_tags.append(self.sof_tag[i])

        self.hdus = new_list_hdus
        self.sof_tag = new_list_tags
        
    def generate_panels(self, **kwargs):
        panels = {}
        self.get_extensions()
        new_panels = super().generate_panels(ext=self.extensions)
        for i, (panel, panel_descr) in enumerate(new_panels.items()):
            # Alter the cut pos, or remove CutPlot(s) completely,
            # depending on task name
            try:
                task_name = panel_descr["task_name"]
            except KeyError:
                raise RuntimeError(
                    "A report has been created by "
                    "MasterRawdispReport that did "
                    "not come back with a task name "
                    "attached!"
                )
            panel_descr["report_name"] = "uves_rawdisp_{}_{}_{}".format(
                task_name,
                self.raw_extensions[self.extensions[i]],
                self.sof_tag[i].lower(),
            )
            panel_descr["report_description"] = (
                f"UVES rawdisp panel - "
                f"{panel_descr['task_name']}, "
                f"{panel_descr['tag']}, "
                f"{os.path.basename(panel_descr['filename'])}, "
                f"{panel_descr['ext']}"
            )
        panels = {**panels, **new_panels}

        return panels


rep = UvesRawdispReport()
