/*                                                                              *
 *   This file is part of the ESO UVES Pipeline                                 *
 *   Copyright (C) 2004,2005 European Southern Observatory                      *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or modify       *
 *   it under the terms of the GNU General Public License as published by       *
 *   the Free Software Foundation; either version 2 of the License, or          *
 *   (at your option) any later version.                                        *
 *                                                                              *
 *   This program is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *   GNU General Public License for more details.                               *
 *                                                                              *
 *   You should have received a copy of the GNU General Public License          *
 *   along with this program; if not, write to the Free Software                *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA       *
 *                                                                              */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 * @addtogroup uves_idp
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/


/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <uves_utl_idp.h>

#include <uves_reduce.h>
#include <uves_reduce_utils.h>
#include <uves_corrbadpix.h>

#include <uves_chip.h>
#include <uves_plot.h>
#include <uves_dfs.h>
#include <uves_pfits.h>
#include <uves_parameters.h>
#include <uves_msg.h>
#include <uves_utils.h>
#include <uves_utils_wrappers.h>
#include <uves_qclog.h>
#include <uves_error.h>
#include <uves_merge.h>
#include <uves.h>
#include <uves_dump.h>

#include <cpl.h>
#include <string.h>
#include <../hdrl/hdrl.h>
#include <uves_response_utils.h>

/**
 * Linear interpolation of spec_res.
 * @param x   The x value to interpolate for.
 * @param is_blue Is the blue arm (true) or red arm (false).
 * @return    Interpolated y value.
 */
double interp_spec_res(const uves_propertylist *header, enum uves_chip chip)
{

    double slit_widths[20] = {
        0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1,
        1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0, 2.1};

    double red_resolving_powers[20] = {
        128600, 107200, 87410, 74450, 66320, 56990, 51690, 45990, 42310, 37820,
        34540, 33290, 31220, 30360, 27890, 25820, 24160, 22500, 20840, 18770};

    double blue_resolving_powers[20] = {
        81950, 77810, 71050, 65030, 58640, 53750, 49620, 43980, 40970, 39090,
        36840, 33830, 31950, 31200, 29000, 26310, 24810, 23680, 21800, 19540};

    bool is_blue = (chip == UVES_CHIP_BLUE);

    const double *xp = slit_widths;
    double *fp = is_blue ? blue_resolving_powers : red_resolving_powers;

    cpl_size n = 20;

    double x = uves_pfits_get_slitwidth(header, chip);

    
    // Extrapolate to left
    if (x <= xp[0])
        return fp[0];
    // Extrapolate to right
    if (x >= xp[n - 1])
        return fp[n - 1];

    // Find interval xp[i] <= x < xp[i+1]
    for (cpl_size i = 0; i < n - 1; ++i)
    {
        if (x < xp[i + 1])
        {
            double t = (x - xp[i]) / (xp[i + 1] - xp[i]);
            return fp[i] + t * (fp[i + 1] - fp[i]);
        }
    }
    // Should not reach here
    return fp[n - 1];
}

static void replace_spaces_with_underscores(char *str) 
{     

    for (int i = 0; str[i] != '\0'; i++) {         
         if (str[i] == ' ') {             
              str[i] = '_'; // Replace space with an underscore         
         }     
     } 
} 

/*-----------------------------------------------------------------------------
                            Implementation
 -----------------------------------------------------------------------------*/


const char * const uves_idp_desc_short = "Produce an idp file";
const char * const uves_idp_desc =
"This recipe take a set of input science frames (\n"
"SCI_POINT_BLUE or SCI_POINT_RED, or \n"
"SCI_SLICER_BLUE or SCI_SLICER_RED) \n"
"and produces a set of idp frames.\n"
"\n"
"For each arm (xxxx = BLUE, REDL, REDU) the recipe produces a combination of "
"of idp products\n";

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    parameters        the parameterlist to fill
  @param    recipe_id         name of calling recipe
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
int uves_idp_define_parameters_body(cpl_parameterlist *parameters,
                       const char *recipe_id)
{

    /*****************
     *    General    *
     *****************/
    if (uves_define_global_parameters(parameters))
    {
        return -1;
    }

    /*******************
     *  Reduce.        *
     ******************/
    /*if (uves_propagate_parameters_step(UVES_IDPR_ID, parameters, recipe_id, 0))
    {
        return -1;
    }*/

    return (cpl_error_get_code() != CPL_ERROR_NONE);
}

cpl_error_code uves_extend_array(cpl_array *dest_array,
                                            const cpl_array *ins_array, const uves_trim_bounds *bounds)
{
    if (!dest_array || !ins_array)
    {
        uves_msg("Cannot extend NULL array");
        return CPL_ERROR_NULL_INPUT;
    }

    cpl_size len = cpl_array_get_size(dest_array);

    cpl_array *tmp_array = cpl_array_extract(ins_array, bounds->start, bounds->count);
    cpl_error_code err = cpl_array_insert(dest_array, tmp_array, len);
    cpl_array_delete(tmp_array);

    return err;
}

cpl_type uves_wave_precision(int naxis1, double cdelt1, double crval1, double crpix1) {
    double maxwave, binfrac;
    double val1 = (1.0 - crpix1) * cdelt1;
    double val2 = (naxis1 - crpix1) * cdelt1;
    double maxval = (val1 > val2) ? val1 : val2;
    maxwave = crval1 + maxval;
    binfrac = fabs(cdelt1 / maxwave);
    if (0.02 * binfrac < 1e-7) {
        // printf("wave format: double precision\n");
        return CPL_TYPE_DOUBLE;
    } else {
        // printf("wave format: single precision\n");
        return CPL_TYPE_FLOAT;
    }
}

/**
 * @brief Create a cpl_array of wavelength values from WCS parameters.
 * @param naxis1  Number of wavelength points (array length)
 * @param cdelt1  Wavelength increment per pixel
 * @param crval1  Wavelength value at reference pixel
 * @param crpix1  Reference pixel (1-based)
 * @return        Pointer to cpl_array of length naxis1 (type double), or NULL on error
 */
cpl_array *uves_wavelength_array(const uves_propertylist *header, uves_trim_bounds *bounds)
{
    double crval1 = uves_propertylist_get_float(header, "CRVAL1");
    if (crval1 == 0.0) {
        uves_msg("Invalid CRVAL1 value: %f", crval1);
        return NULL;
    }
    double crpix1 = uves_propertylist_get_float(header, "CRPIX1");

    double cdelt1 = uves_propertylist_get_float(header,"CDELT1");
    if (cdelt1 == 0.0) {
        uves_msg("Invalid CDELT1 value: %f", cdelt1);
        return NULL;
    }

    cpl_type wave_type = uves_wave_precision(bounds->count, cdelt1, crval1, crpix1);

    cpl_array *wave = cpl_array_new(bounds->count, wave_type);
    if (!wave)
        return NULL;

    if (wave_type == CPL_TYPE_FLOAT)
    {
        for (int i = bounds->start; i < bounds->count + bounds->start; ++i)
        {
            cpl_array_set_float(wave, i - bounds->start,
                                (float)(crval1 + ((i + 1) - crpix1) * cdelt1));
        }
    }
    else if (wave_type == CPL_TYPE_DOUBLE)
    {
        for (int i = bounds->start; i < bounds->count + bounds->start; ++i)
        {
            cpl_array_set_double(wave, i - bounds->start,
                                 crval1 + ((i + 1) - crpix1) * cdelt1);
        }
    }
    return wave;
}


/**
 * Find the index of the first or last non-zero element in a cpl_array.
 * 
 * @param arr      Pointer to cpl_array (CPL_TYPE_FLOAT or CPL_TYPE_DOUBLE)
 * @param forward  If true, search from the front ('f'), else from the back ('b')
 * @return         Index of first (or last) non-zero value, or -1 if all zero or error
 */
int uves_trim_zeros_index(cpl_array *arr, int forward)
{

    if (!arr) return -1;
    cpl_type arr_type = cpl_array_get_type(arr);
    cpl_size n = cpl_array_get_size(arr);


    if (arr_type == CPL_TYPE_FLOAT) {
        
        const float *arr_data = cpl_array_get_data_float(arr);
        if (forward) {
            for (cpl_size i = 0; i < n; ++i)
                if (arr_data[i] != 0.0f) return i;
        } else {
            for (cpl_size i = n; i-- > 0; )
                if (arr_data[i] != 0.0f) return i;
        }
    } else if (arr_type == CPL_TYPE_DOUBLE) {
        const double *arr_data = cpl_array_get_data_double(arr);
        if (forward) {
            for (cpl_size i = 0; i < n; ++i)
                if (arr_data[i] != 0.0) return i;
        } else {
            for (cpl_size i = n; i-- > 0; )
                if (arr_data[i] != 0.0) return i;
        }
    } else {
        // Unsupported type
        return -1;
    }
    // All zeros
    return -1;
}

uves_trim_bounds  uves_get_trim_bounds(cpl_array *array, enum uves_chip chip)
{
    uves_trim_bounds bounds;
    cpl_size n = cpl_array_get_size(array);

    bounds.start = 0;
    bounds.count = n;

    if (chip == UVES_CHIP_REDU) {
        int lower = uves_trim_zeros_index(array, 1); // forward
        bounds.start = (lower >= 0) ? lower : 0;
        bounds.count = n - bounds.start;
    } else if (chip == UVES_CHIP_REDL) {
        int upper = uves_trim_zeros_index(array, 0); // backward
        bounds.start = 0;
        bounds.count = (upper >= 0) ? (upper + 1) : n;
    }
    // For UVES_CHIP_BLUE, use full array (start=0, count=n)
    return bounds;
}

int uves_reduced_arrays_find_chip(const uves_reduced_arrays *arr, enum uves_chip chip)
{
    if (!arr || !arr->tags) return -1;
    for (int i = 0; i < arr->nvec; ++i) {
        if (arr->tags[i] && arr->tags[i]->chip == chip)
            return i;
    }
    return -1;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the command line options and execute the data reduction
  @param    parameters  the parameters list
  @param    frames      the frames list
  @param    recipe_id   the recipe name (will be written to FITS headers)
  @param    starttime   time when calling recipe started
  @return   CPL_ERROR_NONE if everything is ok
 */
/*----------------------------------------------------------------------------*/
void uves_reduce_idp(cpl_frameset *frames, const cpl_parameterlist *parameters,
        const char *recipe_id, const char *starttime)
{



    /* Local variables */
    cpl_frame **raw_frames = NULL;  /* Array of cpl_frame pointers */
    char *title_string = NULL;
    uves_propertylist *table_header = NULL;
    uves_propertylist *rec_header = NULL;

    int frameset_size = 0;   /* Keeps track of number of raw_header pointers allocated */

    check( frameset_size = cpl_frameset_get_size(frames),
       "Error reading frameset size");

    uves_reduced_arrays reduced_science = {0};
    uves_reduced_arrays reduced_science_error = {0};
    uves_reduced_arrays merged_sky = {0};
    uves_reduced_arrays fluxcal_science = {0};
    uves_reduced_arrays fluxcal_error = {0};

    cpl_array * idp_wave = NULL; 
    cpl_array * idp_reduced_science = NULL;
    cpl_array * idp_reduced_science_error = NULL; 
    cpl_array * idp_merged_sky = NULL; 
    cpl_array * idp_fluxcal_science = NULL; 
    cpl_array * idp_fluxcal_error = NULL;

    char *idp_tag = NULL;
    char *idp_filename = NULL;

    /* Load raw image and header, and identify input frame as red or blue */
    check(uves_load_reduce_arrays(frames, "FLUX_REDUCED", &reduced_science),
          "Error loading raw frame");
    check(uves_load_reduce_arrays(frames, "ERR_REDUCED", &reduced_science_error),
          "Error loading raw frame");
    check(uves_load_reduce_arrays(frames, "BGFLUX_REDUCED", &merged_sky),
          "Error loading raw frame");
    check(uves_load_reduce_arrays(frames, "FLUX", &fluxcal_science),
          "Error loading raw frame");
    check(uves_load_reduce_arrays(frames, "ERR", &fluxcal_error),
          "Error loading raw frame");

    if (reduced_science.nvec == 0) {
        uves_msg("No FLUX_REDUCED frames found in frameset");
        //cpl_error_set_code(CPL_ERROR_NO_DATA);
        goto cleanup;
    }

    for (enum uves_chip chip = uves_chip_get_first(true);
         chip != UVES_CHIP_INVALID;
         chip = uves_chip_get_next_all(chip))
    {
        int i = uves_reduced_arrays_find_chip(&reduced_science, chip);
        if (i < 0)
        {
            uves_msg("No FLUX_REDUCED frames found for chip %s", uves_chip_tostring_upper(chip));
            continue; // Skip this chip
        }

        uves_trim_bounds bounds = uves_get_trim_bounds(reduced_science.arrays[i], reduced_science.tags[i]->chip);
        if (!idp_reduced_science)
        {
            rec_header = uves_propertylist_new();
            uves_propertylist_copy_property_regexp_rename(rec_header,
                                                          reduced_science.headers[i],
                                                          "ESO PRO .*"
                                                          "|ESO DET OUT.*"
                                                          "|ESO QC ORD.*"
                                                          "|ESO QC OBJ.*"
                                                          "|ESO QC EX NORD.*"
                                                          "|ESO BNOISE.*",
                                                          0, i + 1);

            const char *dispelem = NULL;
            if (chip == UVES_CHIP_BLUE)
            {
                dispelem = uves_propertylist_get_string(reduced_science.headers[i], "ESO INS GRAT1 NAME");
            }
            else
            {
                dispelem = uves_propertylist_get_string(reduced_science.headers[i], "ESO INS GRAT2 NAME");
            }
            if (dispelem && strlen(dispelem) > 0)
            {
                check(uves_propertylist_append_string(rec_header, "DISPELEM", dispelem),
                      "Error setting DISPELEM property");
            }
            else
            {
                uves_msg("No DISPELEM found in header for chip %s", uves_chip_tostring_upper(chip));
            }

            idp_wave = uves_wavelength_array(reduced_science.headers[i], &bounds);
            idp_reduced_science = cpl_array_extract(reduced_science.arrays[i], bounds.start, bounds.count);
            i = uves_reduced_arrays_find_chip(&reduced_science_error, chip);
            if (i < 0)
            {
                uves_msg("No ERR_REDUCED frames found for chip %s", uves_chip_tostring_upper(chip));
                goto cleanup;
            }
            idp_reduced_science_error = cpl_array_extract(reduced_science_error.arrays[i], bounds.start, bounds.count);
            i = uves_reduced_arrays_find_chip(&merged_sky, chip);
            if (i >= 0)
            {
                idp_merged_sky = cpl_array_extract(merged_sky.arrays[i], bounds.start, bounds.count);
            }
            i = uves_reduced_arrays_find_chip(&fluxcal_science, chip);
            if (i >= 0)
            {
                idp_fluxcal_science = cpl_array_extract(fluxcal_science.arrays[i], bounds.start, bounds.count);
                i = uves_reduced_arrays_find_chip(&fluxcal_error, chip);
                if (i < 0)
                {
                    uves_msg("No ERR frames found for chip %s", uves_chip_tostring_upper(chip));
                    goto cleanup;
                }
                idp_fluxcal_error = cpl_array_extract(fluxcal_error.arrays[i], bounds.start, bounds.count);
            }
            check(idp_filename = uves_idp_filename(chip),
                 "Error getting filename");
        }
        else
        {
            uves_propertylist_copy_property_regexp_rename(rec_header,
                                                          reduced_science.headers[i],
                                                          "ESO PRO .*"
                                                          "|ESO DET OUT.*"
                                                          "|ESO QC ORD.*"
                                                          "|ESO QC OBJ.*"
                                                          "|ESO QC EX NORD"
                                                          "|ESO BNOISE",
                                                          0, i+1);

            cpl_size n_idp = cpl_array_get_size(idp_wave);
            cpl_array *tmp_wave = uves_wavelength_array(reduced_science.headers[i], &bounds);
            check(cpl_array_insert(idp_wave, tmp_wave, n_idp),"Error extending WAVE array");
            cpl_array_delete(tmp_wave);

            check(uves_extend_array(idp_reduced_science, reduced_science.arrays[i], &bounds),"Error extending FLUX_REDUCED array");
            i = uves_reduced_arrays_find_chip(&reduced_science_error, chip);
            if (i < 0)
            {
                uves_msg("No ERR_REDUCED frames found for chip %s", uves_chip_tostring_upper(chip));
                goto cleanup;
            }
            check(uves_extend_array(idp_reduced_science_error, reduced_science_error.arrays[i], &bounds),"Error extending ERR_REDUCED array");
            i = uves_reduced_arrays_find_chip(&merged_sky, chip);
            if (i >= 0)
            {
                check(uves_extend_array(idp_merged_sky, merged_sky.arrays[i], &bounds),"Error extending BGFLUX_REDUCED array");
            }
            i = uves_reduced_arrays_find_chip(&fluxcal_science, chip);
            if (i >= 0)
            {
                check(uves_extend_array(idp_fluxcal_science, fluxcal_science.arrays[i], &bounds),"Error extending FLUX array");
                i = uves_reduced_arrays_find_chip(&fluxcal_error, chip);
                if (i < 0)
                {
                    uves_msg("No ERR frames found for chip %s", uves_chip_tostring_upper(chip));
                    goto cleanup;
                }
                check(uves_extend_array(idp_fluxcal_error, fluxcal_error.arrays[i], &bounds),
                      "Error extending ERR array");
            }
        }
        //break;
    }

    if (!idp_reduced_science)
    {
        uves_msg("No FLUX_REDUCED frames found in frameset");
        goto cleanup;
    }
    check(uves_propertylist_append_string(rec_header, "PRODCATG", "SCIENCE.SPECTRUM"),
          "Error setting PRODCATG property");

    check(uves_propertylist_append_int(rec_header, "PRODLVL", 2),
          "Error setting PRODLVL property");
    check(uves_propertylist_set_comment(rec_header, "PRODLVL",
          "Product level: 1-raw 2-science grade 3-advanced"),
        "Error setting PRODLVL comment");

    double exptime = uves_pfits_get_exptime(reduced_science.headers[0]);
    check(uves_propertylist_append_double(rec_header, "TEXPTIME", exptime),
          "Error setting TEXPTIME property");

    double mjd_obs = uves_pfits_get_mjdobs(reduced_science.headers[0]);
    double mjd_end = mjd_obs + exptime / 86400.0;
    check(uves_propertylist_append_double(rec_header, "MJD-END", mjd_end),
          "Error setting MJD-END property");

    const char *prog_id = uves_pfits_get_prog_id(reduced_science.headers[0]);
    check(uves_propertylist_append_string(rec_header, "PROG_ID", prog_id),
          "Error setting PROG_ID property");

    int obs_id = uves_pfits_get_obs_id(reduced_science.headers[0]);
    check(uves_propertylist_append_int(rec_header, "OBID1", obs_id),
          "Error setting OBID1 property");

    check(uves_propertylist_append_int(rec_header, "NCOMBINE", 1),
          "Error setting NCOMBINE property");

    const bool is_slice = strstr(reduced_science.tags[0]->tag, "SCI_SLIC");
    const bool abscell = strstr(reduced_science.tags[0]->tag, "ABSCELL");

    char *obstech = NULL;
    const char *slice_name = NULL;
    if (is_slice)
    {
        slice_name = uves_pfits_get_slit1_name(reduced_science.headers[0]);
    }
    if (abscell)
    {
        if (is_slice)
        {
            obstech = cpl_sprintf("ECHELLE,ABSORPTION-CELL,%s", slice_name);
        }
        else
        {
            obstech = cpl_strdup("ECHELLE,ABSORPTION-CELL");
        }
    }
    else if (is_slice)
    {
        obstech = cpl_sprintf("ECHELLE,%s", slice_name);
    }
    else
    {
        obstech = cpl_strdup("ECHELLE");
    }
    check(uves_propertylist_append_string(rec_header, "OBSTECH", obstech),
          "Error setting OBSTECH property");
    cpl_free(obstech);

    if (idp_fluxcal_science)
    {
        check(uves_propertylist_append_string(rec_header, "FLUXCAL", "ABSOLUTE"),
              "Error setting FLUXCAL property");
    }
    else
    {
        check(uves_propertylist_append_string(rec_header, "FLUXCAL", "UNCALIBRATED"),
              "Error setting FLUXCAL property");
    }

    const char *proc_soft = uves_pfits_get_proc_soft(reduced_science.headers[0]);
    check(uves_propertylist_append_string(rec_header, "PROCSOFT", proc_soft),
          "Error setting PROCSOFT property");

    const char *prov1 = uves_pfits_get_prov(reduced_science.headers[0]);
    check(uves_propertylist_append_string(rec_header, "PROV1", prov1),
          "Error setting PROV1 property");

    check(uves_propertylist_append_string(rec_header, "SPECSYS", "TOPOCENT"),
          "Error setting SPECSYS property");

    check(uves_propertylist_append_bool(rec_header, "EXT_OBJ", false),
          "Error setting EXT_OBJ property");

    check(uves_propertylist_append_bool(rec_header, "CONTNORM", false),
          "Error setting CONTNORM property");

    check(uves_propertylist_append_bool(rec_header, "TOT_FLUX", false),
          "Error setting TOT_FLUX property");

    check(uves_propertylist_append_int(rec_header, "FLUXERR", -2),
          "Error setting FLUXERR property");

    double wave_min = cpl_array_get_min(idp_wave);
    wave_min = round(wave_min / 10.0 * 10000.0) / 10000.0;
    double wave_max = cpl_array_get_max(idp_wave);
    wave_max = round(wave_max / 10.0 * 10000.0) / 10000.0;
    check(uves_propertylist_append_double(rec_header, "WAVELMIN", wave_min),
          "Error setting WAVE_MIN property");
    check(uves_propertylist_append_double(rec_header, "WAVELMAX", wave_max),
          "Error setting WAVE_MAX property");

    double cdelt1 = uves_pfits_get_cdelt1(reduced_science.headers[0]);
    cdelt1 = round(cdelt1 / 10.0 * 10000.0) / 10000.0;
    check(uves_propertylist_append_double(rec_header, "SPEC_BIN", cdelt1),
          "Error setting SPEC_BIN property");

    double crder1 = uves_pfits_get_crder1(reduced_science.headers[0]);
    crder1 = round(crder1 / 10.0 * 100000.0) / 100000.0;
    check(uves_propertylist_append_double(rec_header, "SPEC_ERR", crder1),
          "Error setting SPEC_ERR property");

    double csyer1 = uves_pfits_get_csyer1(reduced_science.headers[0]);
    csyer1 = round(csyer1 / 10.0 * 100000.0) / 100000.0;
    check(uves_propertylist_append_double(rec_header, "SPEC_SYE", csyer1),
          "Error setting SPEC_SYE property");

    double snr = 0;
    for (int i = 0; i < reduced_science.nvec; i++)
    {
        snr += uves_pfits_get_qc_msnr(reduced_science.headers[i]);
    }
    snr /= reduced_science.nvec;
    check(uves_propertylist_append_double(rec_header, "SNR", snr),
          "Error setting SNR property");

    double spec_res = interp_spec_res(reduced_science.headers[0], reduced_science.tags[0]->chip);
    check(uves_propertylist_append_double(rec_header, "SPEC_RES", spec_res),
          "Error setting SPEC_RES property");

    table_header = uves_propertylist_new(); // Temporary header for WCS and other properties

    uves_propertylist_copy_property_regexp(table_header,
                                           reduced_science.headers[0],
                                           "^OBJECT$"
                                           "|^RA$"
                                           "|^DEC$",
                                           0);

    int nelem = cpl_array_get_size(idp_wave);
    check(uves_propertylist_append_int(table_header, "NELEM", nelem),
          "Error setting NELEM property");

    check(uves_propertylist_append_string(table_header, "VOCLASS", "SPECTRUM v2.0"),
          "Error setting VOCLASS property");

    check(uves_propertylist_append_string(table_header, "VOPUB", "ESO/SAF"),
          "Error setting VOPUB property");

    title_string = cpl_sprintf("%s_%d_%s",
                                     uves_propertylist_get_string(reduced_science.headers[0], "OBJECT"),
                                     obs_id,
                                     uves_propertylist_get_string(reduced_science.headers[0], "DATE-OBS"));

    replace_spaces_with_underscores(title_string);

    check(uves_propertylist_append_string(table_header, "TITLE", title_string),
          "Error setting TITLE property");

    double aper = uves_pfits_get_slitwidth(reduced_science.headers[0], reduced_science.tags[0]->chip);
    aper = round(aper / (60 * 60) * 100000.0) / 100000.0;
    check(uves_propertylist_append_double(table_header, "APERTURE", aper),
          "Error setting APERTURE property");

    double telapse = (mjd_end - mjd_obs) * 86400.0;
    telapse = round(telapse * 1000.0) / 1000.0;
    check(uves_propertylist_append_double(table_header, "TELAPSE", telapse),
          "Error setting TELAPSE property");

    double tmid = (mjd_obs + mjd_end) / 2.0;
    tmid = round(tmid * 1e11) / 1e11;
    check(uves_propertylist_append_double(table_header, "TMID", tmid),
          "Error setting TMID property");

    double spec_val = (wave_min + wave_max) / 2.0;
    spec_val = round(spec_val * 1e4) / 1e4; // Round to 4 decimal places
    check(uves_propertylist_append_double(table_header, "SPEC_VAL", spec_val),
          "Error setting SPEC_VAL property");
    double spec_bw = wave_max - wave_min;
    spec_bw = round(spec_bw * 1e4) / 1e4;
    check(uves_propertylist_append_double(table_header, "SPEC_BW", spec_bw),
          "Error setting SPEC_BW property");

    check(uves_propertylist_append_string(table_header, "EXTNAME", "SPECTRUM"),
          "Error setting SPECTRUM property");


    const char * specflux = "spec:Data.FluxAxis.Value";
    const char * esoflux = "eso:Data.FluxAxis.Value";


    const char * specfluxerr = "spec:Data.FluxAxis.Accuracy.StatError";
    const char * esofluxerr = "eso:Data.FluxAxis.Accuracy.StatError";

    const char * specbackmod = "spec:Data.BackgroundModel.Value";
    const char *esobackmod = "eso:Data.BackgroundModel.Value";

    const char *tucd_r = "phot.flux.density;em.wl;src.net;stat.uncalib";
    const char *tucd_rerr = "stat.error;phot.flux.density;em.wl;src.net;stat.uncalib";

    const char *tucd_rs = "phot.flux.density;em.wl;stat.uncalib";
    const char *tucd_rserr = "stat.error;phot.flux.density;em.wl;stat.uncalib";

    //check(idp_filename = uves_idp_filename(UVES_CHIP_REDL),
    //      "Error getting filename");

    char *idp_tag_red = uves_pfits_get_pro_catg(reduced_science.headers[0]);
    size_t tag_len = strlen(idp_tag_red);
    idp_tag = uves_sprintf("%.*s", (int)(tag_len - 1), idp_tag_red);
    cpl_size n_idp = cpl_array_get_size(idp_reduced_science);
    cpl_table *idp_product = cpl_table_new(1);

    if (cpl_array_get_type(idp_wave) == CPL_TYPE_FLOAT)
    {
        cpl_table_new_column_array(idp_product, "WAVE", CPL_TYPE_FLOAT, n_idp);
    }
    else
    {
        cpl_table_new_column_array(idp_product, "WAVE", CPL_TYPE_DOUBLE, n_idp);
    }

    cpl_table_set_array(idp_product, "WAVE", 0, idp_wave);
    cpl_table_set_column_unit(idp_product, "WAVE", "angstrom");
    check(uves_propertylist_append_string(table_header, "TCOMM1", "Array computed from original WCS information"),
          "Error setting TCOMM1 property");
    check(uves_propertylist_append_string(table_header, "TUTYP1", "spec:Data.SpectralAxis.Value"),
          "Error setting TUTYP1 property");
    check(uves_propertylist_append_string(table_header, "TUCD1", "em.wl;obs.atmos"),
          "Error setting TUCD1 property");
    check(uves_propertylist_set_comment(table_header, "TUCD1", "Air wavelength"),
          "Error setting TUCD1 comment");
    check(uves_propertylist_append_double(table_header, "TDMIN1", cpl_array_get_min(idp_wave)),
          "Error setting TDMIN1 property");
    check(uves_propertylist_append_double(table_header, "TDMAX1", cpl_array_get_max(idp_wave)),
          "Error setting TDMAX1 property");

    cpl_table_new_column_array(idp_product, "FLUX_REDUCED", CPL_TYPE_FLOAT, n_idp);
    cpl_table_set_array(idp_product, "FLUX_REDUCED", 0, idp_reduced_science);
    cpl_table_set_column_unit(idp_product, "FLUX_REDUCED", "adu");
    check(uves_propertylist_append_string(table_header, "TCOMM2", reduced_science.filenames[0]),
          "Error setting TCOMM2 property");
    check(uves_propertylist_append_string(table_header, "TUTYP2", idp_fluxcal_science ? esoflux : specflux),
          "Error setting TUTYP2 property");
    check(uves_propertylist_append_string(table_header, "TUCD2", is_slice ? tucd_rs : tucd_r),
          "Error setting TUCD2 property");
    check(uves_propertylist_append_double(table_header, "TDMIN2", cpl_array_get_min(idp_reduced_science)),
          "Error setting TDMIN2 property");
    check(uves_propertylist_append_double(table_header, "TDMAX2", cpl_array_get_max(idp_reduced_science)),
          "Error setting TDMAX2 property");


    cpl_table_new_column_array(idp_product, "ERR_REDUCED", CPL_TYPE_FLOAT, n_idp);
    cpl_table_set_array(idp_product, "ERR_REDUCED", 0, idp_reduced_science_error);
    cpl_table_set_column_unit(idp_product, "ERR_REDUCED", "adu");
    check(uves_propertylist_append_string(table_header, "TCOMM3", reduced_science_error.filenames[0]),
          "Error setting TCOMM3 property");
    check(uves_propertylist_append_string(table_header, "TUTYP3", idp_fluxcal_science ? esofluxerr : specfluxerr),
          "Error setting TUTYP3 property");
    check(uves_propertylist_append_string(table_header, "TUCD3", is_slice ? tucd_rserr : tucd_rerr),
          "Error setting TUCD3 property");
    check(uves_propertylist_append_double(table_header, "TDMIN3", cpl_array_get_min(idp_reduced_science_error)),
          "Error setting TDMIN3 property");
    check(uves_propertylist_append_double(table_header, "TDMAX3", cpl_array_get_max(idp_reduced_science_error)),
          "Error setting TDMAX3 property");



    if (idp_merged_sky)
    {
        cpl_table_new_column_array(idp_product, "BGFLUX_REDUCED", CPL_TYPE_FLOAT, n_idp);
        cpl_table_set_array(idp_product, "BGFLUX_REDUCED", 0, idp_merged_sky);
        cpl_table_set_column_unit(idp_product, "BGFLUX_REDUCED", "adu");
        check(uves_propertylist_append_string(table_header, "TCOMM4", merged_sky.filenames[0]),
            "Error setting TCOMM4 property");
        check(uves_propertylist_append_string(table_header, "TUTYP4", idp_fluxcal_science ? esobackmod : specbackmod),
            "Error setting TUTYP4 property");
    check(uves_propertylist_append_string(table_header, "TUCD4", "phot.flux.density;em.wl;stat.uncalib"),
          "Error setting TUCD4 property");
    check(uves_propertylist_append_double(table_header, "TDMIN4", cpl_array_get_min(idp_merged_sky)),
          "Error setting TDMIN4 property");
    check(uves_propertylist_append_double(table_header, "TDMAX4", cpl_array_get_max(idp_merged_sky)),
          "Error setting TDMAX4 property");

    }

    if (idp_fluxcal_science)
    {

        const char *idp_tag_flux = uves_pfits_get_pro_catg(fluxcal_science.headers[0]);
        tag_len = strlen(idp_tag_flux);
        cpl_free(idp_tag);
        idp_tag = uves_sprintf("%.*s", (int)(tag_len - 1), idp_tag_flux);

        const char *tucd_f = "phot.flux.density;em.wl;src.net;meta.main";
        const char *tucd_ferr = "stat.error;phot.flux.density;em.wl;src.net;meta.main";

        const char *tucd_fs = "phot.flux.density;em.wl;meta.main";
        char *tucd_fserr = "stat.error;phot.flux.density;em.wl;meta.main";

        cpl_table_new_column_array(idp_product, "FLUX", CPL_TYPE_FLOAT, n_idp);
        cpl_table_set_array(idp_product, "FLUX", 0, idp_fluxcal_science);

        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TCOMM5" : "TCOMM4", 
            fluxcal_science.filenames[0]), "Error setting TCOMM5 property");
        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TUTYP5" : "TUTYP4", 
            "spec:Data.FluxAxis.Value"), "Error setting TUTYP5 property");
        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TUCD5" : "TUCD4", 
            is_slice ? tucd_fs : tucd_f), "Error setting TUCD5 property");
        check(uves_propertylist_append_double(table_header, idp_merged_sky ? "TDMIN5" : "TDMIN4" , 
            cpl_array_get_min(idp_fluxcal_science)), "Error setting TDMIN5 property");
        check(uves_propertylist_append_double(table_header, idp_merged_sky ? "TDMAX5" : "TDMAX4",
            cpl_array_get_max(idp_fluxcal_science)), "Error setting TDMAX5 property");


        cpl_table_new_column_array(idp_product, "ERR", CPL_TYPE_FLOAT, n_idp);
        cpl_table_set_array(idp_product, "ERR", 0, idp_fluxcal_error);
        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TCOMM6" : "TCOMM5", 
            fluxcal_error.filenames[0]), "Error setting TCOMM6 property");
        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TUTYP6" : "TUTYP5", 
            "spec:Data.FluxAxis.Accuracy.StatError"), "Error setting TUTYP6 property");
        check(uves_propertylist_append_string(table_header, idp_merged_sky ? "TUCD6" : "TUCD5", 
            is_slice ? tucd_fserr : tucd_ferr), "Error setting TUCD6 property");
        check(uves_propertylist_append_double(table_header, idp_merged_sky ? "TDMIN6" : "TDMIN5" , 
            cpl_array_get_min(idp_fluxcal_error)), "Error setting TDMIN6 property");
        check(uves_propertylist_append_double(table_header, idp_merged_sky ? "TDMAX6" : "TDMAX5",
            cpl_array_get_max(idp_fluxcal_error)), "Error setting TDMAX6 property");


        const char *bunit = NULL;

        if (uves_propertylist_contains(fluxcal_science.headers[0], UVES_BUNIT))
        {
            bunit = uves_pfits_get_bunit(fluxcal_science.headers[0]);
            if (bunit && strncmp(bunit, "10^-16", 6) == 0)
            {
                cpl_table_set_column_unit(idp_product, "FLUX", "10**(-16)erg.cm**(-2).s**(-1).angstrom**(-1)");
                cpl_table_set_column_unit(idp_product, "ERR", "10**(-16)erg.cm**(-2).s**(-1).angstrom**(-1)");
            }
            else if (bunit && strncmp(bunit, "erg", 3) == 0)
            {
                cpl_table_set_column_unit(idp_product, "FLUX", "10**(-16)erg.cm**(-2).s**(-1).angstrom**(-1)");
                cpl_table_set_column_unit(idp_product, "ERR", "10**(-16)erg.cm**(-2).s**(-1).angstrom**(-1)");
                cpl_array* flux_arr = (cpl_array*) cpl_table_get_array(idp_product, "FLUX", 0);
                cpl_array* err_arr = (cpl_array*) cpl_table_get_array(idp_product, "ERR", 0);
                cpl_array_multiply_scalar(flux_arr, 1e16);
                cpl_array_multiply_scalar(err_arr, 1e16);
            }
            else
            {
                cpl_table_set_column_unit(idp_product, "FLUX", " ");
                cpl_table_set_column_unit(idp_product, "ERR", " ");
            }
        }
        else
        {
            cpl_table_set_column_unit(idp_product, "FLUX", " ");
            cpl_table_set_column_unit(idp_product, "ERR", " ");
        }
    }

    check(uves_frameset_insert(frames,
                               idp_product,
                               CPL_FRAME_GROUP_PRODUCT,
                               CPL_FRAME_TYPE_TABLE,
                               CPL_FRAME_LEVEL_FINAL,
                               idp_filename,
                               idp_tag,
                               reduced_science.headers[0],
                               rec_header,
                               table_header,
                               parameters,
                               recipe_id,
                               PACKAGE "/" PACKAGE_VERSION,
                               NULL,
                               starttime, false, 0),
          // CPL_STATS_MIN | CPL_STATS_MAX),
          "Could not add idp science "
          "spectrum error '%s' (%s) to frameset",
          idp_filename, idp_tag);

    uves_msg("IDP spectrum "
             "'%s' (%s) added to frameset",
             idp_filename, idp_tag);

    cpl_table_delete(idp_product);

cleanup:

    cpl_free(idp_filename);
    cpl_free(idp_tag);
    cpl_free(title_string);

    uves_free_reduced_arrays(&reduced_science);
    uves_free_reduced_arrays(&reduced_science_error);
    uves_free_reduced_arrays(&merged_sky);
    uves_free_reduced_arrays(&fluxcal_science);
    uves_free_reduced_arrays(&fluxcal_error);

    uves_propertylist_delete(rec_header);
    uves_propertylist_delete(table_header);

    cpl_array_delete(idp_wave);
    cpl_array_delete(idp_reduced_science);
    cpl_array_delete(idp_reduced_science_error);
    cpl_array_delete(idp_merged_sky);
    cpl_array_delete(idp_fluxcal_science);
    cpl_array_delete(idp_fluxcal_error);

    uves_print_rec_status(18);
    return;
}

/**@}*/
