from edps import data_source, match_rules
from edps.generator.time_range import *

from .uves_classification import *

# Convention for Data sources Association rule levels:
# Each data source can have several match function which correspond to different
# quality levels for the selected data. The level is specified as a number that
# follows this convention:
#   level < 0: more restrictive than the calibration plan
#   level = 0 follows the calibration plan
#   level = 1 quality sufficient for QC1 certification
#   level = 2 probably still acceptable quality
#   level = 3 significant risk of bad quality results


arm = [kwd.ins_path]
binning = [kwd.det_binx, kwd.det_biny]
arm_binning = arm + binning
instrument = [kwd.instrume]
setup_slit = [kwd.slit1_name, kwd.grat1_wlen, kwd.grat2_wlen, kwd.slit2_wid, kwd.slit3_wid]
setup_mos = binning + [kwd.det_readout, kwd.slit3_mode, kwd.slit3_plate, kwd.grat2_wlen]

# matching keywords:
match_detector = arm_binning + [kwd.det_readout, kwd.instrume]
match_bias_dark = binning + [kwd.det_readout, kwd.instrume, kwd.det_chips]
match_grating_detector = arm_binning + [kwd.instrume, kwd.grat1_wlen, kwd.grat2_wlen]
match_fib_ordeftable = [kwd.slit3_mode, kwd.slit3_plate, kwd.grat2_wlen]
match_slice = match_grating_detector + [kwd.slit1_name]

#################################################################################################
# Raw files data sources
#################################################################################################

# Files for detector monitoring (off exposures)
raw_detmon_off = (data_source()
                  .with_classification_rule(detmon_off_class)
                  .with_grouping_keywords([kwd.tpl_start])
                  .with_setup_keywords([kwd.det_binx, kwd.det_biny, kwd.det_readout])
                  .build())

# Files for detector monitoring (on exposures)
raw_detmon_on = (data_source()
                 .with_classification_rule(detmon_on_class)
                 .with_match_keywords([kwd.instrume, kwd.tpl_start] + arm, time_range=ONE_DAY, level=0)
                 .build())

# Files for measuring the cross disperser position
cd_align = (data_source('CD_ALIGN')
            .with_classification_rule(cd_align_red_class)
            .with_classification_rule(cd_align_blue_class)
            .with_grouping_keywords([kwd.det_chips, kwd.tpl_start])
            .with_setup_keywords([kwd.grat1_name, kwd.grat2_name])
            .with_min_group_size(2)
            .build())

# biases
raw_bias = (data_source('BIAS')
            .with_classification_rule(bias_red_class)
            .with_classification_rule(bias_blue_class)
            .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
            .with_min_group_size(3)
            .with_setup_keywords(binning + [kwd.det_readout])
            .with_match_keywords(match_bias_dark, time_range=ONE_WEEK, level=0)
            .with_match_keywords(match_bias_dark, time_range=UNLIMITED, level=3)
            .build())

# Bias for science exposures have different validity ranges than biases for calibrations
associate_bias_to_science = (match_rules()
                             .with_match_keywords(match_bias_dark, time_range=FOUR_DAYS, level=0)
                             .with_match_keywords(match_bias_dark, time_range=ONE_WEEK, level=1)
                             .with_match_keywords(match_bias_dark, time_range=UNLIMITED, level=3))

# -- darks --
# To measure dark current
raw_dark = (data_source('DARK')
            .with_classification_rule(dark_red_class)
            .with_classification_rule(dark_blue_class)
            .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
            .with_setup_keywords(binning + [kwd.det_readout])
            .with_match_keywords(match_bias_dark, time_range=RelativeTimeRange(-90, 90), level=0)
            .with_match_keywords(match_bias_dark, time_range=UNLIMITED, level=3)
            .build())

# To measure "parasitic" light
raw_pdark = (data_source('PDARK')
             .with_classification_rule(pdark_red_class)
             .with_classification_rule(pdark_blue_class)
             .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
             .with_setup_keywords(binning + [kwd.det_readout])
             .build())
# ----

# -- Raw files for format check and order tracing guess --
# slit spectra
raw_arc_lamp_form = (data_source('ARC_FORM')
                     .with_classification_rule(arc_lamp_form_blue_class)
                     .with_classification_rule(arc_lamp_form_red_class)
                     .with_grouping_keywords([kwd.unique])
                     .with_setup_keywords(binning + [kwd.det_readout, kwd.grat1_wlen, kwd.grat2_wlen, kwd.ins_mode])
                     .with_match_keywords(match_grating_detector + [kwd.ins_mode], time_range=ONE_DAY, level=-1)
                     .with_match_keywords(match_grating_detector + [kwd.ins_mode], time_range=THREE_DAYS, level=0)
                     .with_match_keywords(match_grating_detector + [kwd.ins_mode], time_range=ONE_MONTH, level=2)
                     .with_match_keywords(match_grating_detector + [kwd.ins_mode], time_range=UNLIMITED, level=3)
                     .build())

# fiber spectra
raw_arc_lamp_form_mos = (data_source()
                         .with_classification_rule(arc_lamp_form_mos_class)
                         .with_grouping_keywords([kwd.unique])
                         .with_setup_keywords(binning + [kwd.det_readout, kwd.slit3_plate, kwd.grat2_wlen])
                         .with_match_keywords(match_grating_detector + [kwd.slit3_plate], time_range=ONE_DAY, level=-1)
                         .with_match_keywords(match_grating_detector + [kwd.slit3_plate], time_range=THREE_DAYS,
                                              level=0)
                         .with_match_keywords(match_grating_detector + [kwd.slit3_plate], time_range=ONE_MONTH, level=2)
                         .with_match_keywords(match_grating_detector + [kwd.slit3_plate], time_range=UNLIMITED, level=3)
                         .build())
# --

# -- Raw files to detect orders
# slit spectra
raw_order_flat_match_keywords = arm_binning + [kwd.ins_mode, kwd.grat1_wlen, kwd.grat2_wlen]
raw_order_flat = (data_source('ORDER_FLAT')
                  .with_classification_rule(order_flat_blue_class)
                  .with_classification_rule(order_flat_red_class)
                  .with_grouping_keywords([kwd.unique])
                  .with_setup_keywords(binning + [kwd.det_readout, kwd.grat1_wlen, kwd.grat2_wlen, kwd.ins_mode])
                  .with_match_keywords(raw_order_flat_match_keywords, time_range=ONE_DAY, level=-1)
                  .with_match_keywords(raw_order_flat_match_keywords, time_range=THREE_DAYS, level=0)
                  .with_match_keywords(raw_order_flat_match_keywords, time_range=ONE_WEEK, level=1)
                  .with_match_keywords(raw_order_flat_match_keywords, time_range=UNLIMITED, level=3)
                  .build())

# Response calibrations needs order_flat (orderpos) with a different validity range
orderpos_response_rules = (match_rules()
                           .with_match_keywords(raw_order_flat_match_keywords, time_range=ONE_DAY, level=-1)
                           .with_match_keywords(raw_order_flat_match_keywords, time_range=ONE_WEEK, level=0)
                           .with_match_keywords(raw_order_flat_match_keywords, time_range=TWO_WEEKS, level=2)
                           .with_match_keywords(raw_order_flat_match_keywords, time_range=UNLIMITED, level=3))

# fiber spectra
raw_order_flat_mos_match_keywords = [kwd.instrume, kwd.grat2_wlen]
raw_order_flat_mos = (data_source()
                      .with_classification_rule(order_flat_mos_class)
                      .with_grouping_keywords([kwd.unique])
                      .with_setup_keywords(binning + [kwd.det_readout, kwd.slit3_plate, kwd.grat2_wlen])
                      .with_match_keywords(raw_order_flat_mos_match_keywords, time_range=ONE_DAY, level=-1)
                      .with_match_keywords(raw_order_flat_mos_match_keywords, time_range=THREE_DAYS, level=0)
                      .with_match_keywords(raw_order_flat_mos_match_keywords, time_range=ONE_WEEK, level=1)
                      .with_match_keywords(raw_order_flat_mos_match_keywords, time_range=UNLIMITED, level=3)
                      .build())

# The task sff_ofpos_mos needs an match also on kwd.slit3_plate for the orderpos_mos calibrations. I therefore define
# new association rules that overrides those defined in the data_source raw_order_flat_most
orderpos_mos_match_keywords = [kwd.instrume, kwd.grat2_wlen, kwd.slit3_plate]
orderpos_mos_rules = (match_rules()
                      .with_match_keywords(orderpos_mos_match_keywords, time_range=ONE_DAY, level=-1)
                      .with_match_keywords(orderpos_mos_match_keywords, time_range=THREE_DAYS, level=0)
                      .with_match_keywords(orderpos_mos_match_keywords, time_range=ONE_WEEK, level=1)
                      .with_match_keywords(orderpos_mos_match_keywords, time_range=UNLIMITED, level=3))
# --

# -- Raw files for flat fielding (slit)
# regular flats
raw_flat = (data_source('FLAT')
            .with_classification_rule(flat_blue_class)
            .with_classification_rule(flat_red_class)
            .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
            .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_wlen, kwd.grat2_wlen,
                                            kwd.slit2_wid, kwd.slit3_wid])
            .with_match_keywords(match_grating_detector, time_range=ONE_WEEK, level=0)
            .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3)
            .build())

associate_flat_to_science = (match_rules()
                             .with_match_keywords(match_grating_detector, time_range=ONE_DAY, level=-1)
                             .with_match_keywords(match_grating_detector, time_range=TWO_DAYS, level=0)
                             .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3))

# iodine flats
raw_iflat_match_keywords = arm_binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_wlen, kwd.grat2_wlen, kwd.slit2_wid,
                                          kwd.slit3_wid]
raw_iflat = (data_source('IFLAT')
             .with_classification_rule(iflat_blue_class)
             .with_classification_rule(iflat_red_class)
             .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
             .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.slit1_name, kwd.grat1_name,
                                             kwd.grat2_name, kwd.grat1_wlen, kwd.grat2_wlen, kwd.slit2_wid,
                                             kwd.slit3_wid])
             .with_match_keywords(raw_iflat_match_keywords, time_range=TWO_DAYS, level=0)
             .with_match_keywords(raw_iflat_match_keywords, time_range=UNLIMITED, level=3)
             .build())

# telluric lines flats
raw_tflat = (data_source('TFLAT')
             .with_classification_rule(tflat_blue_class)
             .with_classification_rule(tflat_red_class)
             .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
             .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.slit1_name, kwd.grat1_name,
                                             kwd.grat2_name, kwd.grat2_wlen, kwd.slit2_wid])
             .build())

# 2D flats for BLUE setup 346
raw_dflat = (data_source('DFLAT')
             .with_classification_rule(dflat_blue_class)
             .with_classification_rule(dflat_red_class)
             .with_grouping_keywords([kwd.tpl_start, kwd.det_chips])
             .with_setup_keywords(binning + [kwd.det_readout, kwd.grat1_wlen, kwd.grat2_wlen])
             .with_match_keywords(match_slice, time_range=ONE_DAY, level=-1)
             .with_match_keywords(match_slice, time_range=TWO_DAYS, level=0)
             .with_match_keywords(match_slice, time_range=ONE_MONTH, level=2)
             .with_match_keywords(match_slice, time_range=UNLIMITED, level=3)
             .with_match_keywords(match_grating_detector, time_range=ONE_DAY, level=-1)
             .with_match_keywords(match_grating_detector, time_range=THREE_DAYS, level=0)
             .with_match_keywords(match_grating_detector, time_range=ONE_MONTH, level=2)
             .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3)
             .build())
# --

# -- Raw files for flat fielding (fibre)
raw_sflat_mos = (data_source()
                 .with_classification_rule(flat_mos_class)
                 .with_grouping_keywords([kwd.tpl_start])
                 .with_setup_keywords(binning + [kwd.det_readout, kwd.grat2_wlen])
                 .with_match_keywords(match_grating_detector, time_range=ONE_WEEK, level=0)
                 .with_match_keywords(match_grating_detector, time_range=ONE_MONTH, level=2)
                 .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3)
                 .build())

raw_fib_ff_mos = (data_source('FIBRE_FF')
                  .with_classification_rule(fib_ff_odd_mos_class)
                  .with_classification_rule(fib_ff_even_mos_class)
                  .with_classification_rule(fib_ff_all_mos_class)
                  .with_grouping_keywords([kwd.tpl_start])
                  .with_setup_keywords(binning + [kwd.det_readout, kwd.slit3_mode, kwd.slit3_plate,
                                                  kwd.grat2_wlen])
                  .with_match_keywords(match_detector, time_range=ONE_DAY, level=0)
                  .with_match_keywords(match_detector, time_range=ONE_WEEK, level=1)
                  .with_match_keywords(match_detector, time_range=ONE_MONTH, level=2)
                  .with_match_keywords(match_detector, time_range=UNLIMITED, level=3)
                  .build())

# odd orders
raw_fib_ff_odd_mos = (data_source()
                      .with_classification_rule(fib_ff_odd_mos_class)
                      .with_grouping_keywords([kwd.tpl_start])
                      .with_setup_keywords(binning + [kwd.det_readout, kwd.slit3_mode, kwd.slit3_plate,
                                                      kwd.grat2_wlen])
                      .with_match_keywords(match_fib_ordeftable, time_range=ONE_DAY, level=0)
                      .with_match_keywords(match_fib_ordeftable, time_range=ONE_WEEK, level=1)
                      .with_match_keywords(match_fib_ordeftable, time_range=ONE_MONTH, level=2)
                      .with_match_keywords(match_fib_ordeftable, time_range=UNLIMITED, level=3)
                      .build())

# even orders
raw_fib_ff_even_mos = (data_source()
                       .with_classification_rule(fib_ff_even_mos_class)
                       .with_match_keywords([kwd.tpl_start], time_range=ONE_DAY, level=0)
                       .build())

# all orders
raw_fib_ff_all_mos = (data_source()
                      .with_classification_rule(fib_ff_all_mos_class)
                      .with_match_keywords([kwd.tpl_start], time_range=ONE_DAY, level=0)
                      .build())
# --

# Raw files for wavelength calibration
# slit
raw_arc_lamp = (data_source('ARC')
                .with_classification_rule(wave_blue_class)
                .with_classification_rule(wave_red_class_1)
                .with_classification_rule(wave_red_class_2)
                .with_grouping_keywords([kwd.unique])
                .with_setup_keywords(binning + [kwd.det_readout, kwd.grat1_wlen, kwd.grat2_wlen, kwd.slit2_wid,
                                                kwd.slit3_wid, kwd.ins_mode])
                .with_match_function(rules.assoc_arc, time_range=ONE_WEEK, level=0)
                .with_match_function(rules.assoc_arc, time_range=TWO_WEEKS, level=1)
                .with_match_function(rules.assoc_arc, time_range=UNLIMITED, level=3)
                .build())

associate_arc_to_science = (match_rules()
                            .with_match_function(rules.assoc_arc, time_range=ONE_DAY, level=0)
                            .with_match_function(rules.assoc_arc, time_range=ONE_WEEK, level=1)
                            .with_match_function(rules.assoc_arc, time_range=ONE_MONTH, level=2)
                            .with_match_function(rules.assoc_arc, time_range=UNLIMITED, level=3))
# fibre
raw_arc_lamp_mos = (data_source()
                    .with_classification_rule(wave_mos_class)
                    .with_grouping_keywords([kwd.unique])
                    .with_setup_keywords(setup_mos)
                    .with_match_keywords(match_grating_detector, time_range=ONE_WEEK, level=0)
                    .with_match_keywords(match_grating_detector, time_range=TWO_WEEKS, level=0)
                    .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3)
                    .build())

associate_arc_to_science_mos = (match_rules()
                                .with_match_keywords(match_grating_detector, time_range=ONE_DAY, level=0)
                                .with_match_keywords(match_grating_detector, time_range=ONE_WEEK, level=1)
                                .with_match_keywords(match_grating_detector, time_range=ONE_MONTH, level=2)
                                .with_match_keywords(match_grating_detector, time_range=UNLIMITED, level=3))
# --

# Raw files for instrument focus
raw_focus = (data_source('FOCUS')
             .with_classification_rule(focus_red_class_1)
             .with_classification_rule(focus_red_class_2)
             .with_grouping_keywords([kwd.unique])
             .with_setup_keywords(binning + [kwd.det_readout] + setup_slit)
             .build())

# Standard star for response curve computation
raw_flux_std = (data_source('STANDARD')
                .with_classification_rule(standard_blue_class)
                .with_classification_rule(standard_red_class)
                .with_grouping_keywords([kwd.unique])
                .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_name, kwd.grat1_wlen,
                                                kwd.grat2_wlen])
                .with_match_keywords(arm_binning + [kwd.grat1_wlen, kwd.grat2_wlen], time_range=ONE_DAY, level=-1)
                .with_match_keywords(arm_binning + [kwd.grat1_wlen, kwd.grat2_wlen], time_range=THREE_DAYS, level=0)
                .build())

# Telluric standard:
raw_telluric = (data_source('TELLURIC')
                .with_classification_rule(telluric_blue_class)
                .with_classification_rule(telluric_red_class)
                .with_grouping_keywords([kwd.unique])
                .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_name, kwd.grat1_wlen,
                                                kwd.grat2_wlen])
                .build())

# Standard stars used for efficiency calibration only:
raw_efficiency = (data_source('EFFICIENCY_STD')
                  .with_classification_rule(efficiency_blue_class)
                  .with_classification_rule(efficiency_red_class)
                  .with_grouping_keywords([kwd.unique])
                  .with_setup_keywords(binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_name, kwd.grat1_wlen,
                                                  kwd.grat2_wlen])
                  .build())

# Raw science exposures
# extended calibration targets with reflexted sun spectrum, e.g. moon (slit spectra)
raw_solar = (data_source('SOLAR')
             .with_classification_rule(solar_extended_red_class)
             .with_classification_rule(solar_extended_blue_class)
             .with_setup_keywords(binning + [kwd.det_readout] + setup_slit)
             .with_grouping_keywords([kwd.unique])
             .build())

# All science targets (slit spectra)
raw_science = (data_source('SCIENCE')
               .with_classification_rule(sci_point_red_class)
               .with_classification_rule(sci_extended_red_class)
               .with_classification_rule(sci_point_blue_class)
               .with_classification_rule(sci_extended_blue_class)
               .with_classification_rule(sci_generic_blue_class)
               .with_classification_rule(sci_generic_red_class)
               .with_setup_keywords(binning + [kwd.det_readout] + setup_slit)  # info for QcFlow.
               .with_grouping_keywords([kwd.unique])
               .build())

# fibre spectra
raw_fib_science_mos = (data_source('SCIENCE_MOS')
                       .with_classification_rule(fib_science_mos_class)
                       .with_setup_keywords(setup_mos)  # info for QcFlow.
                       .with_grouping_keywords([kwd.unique])
                       .build())

#################################################################################################
# data sources for static calibrations
#################################################################################################

# Reference list of arc lines
line_refer_table = (data_source()
                    .with_classification_rule(line_refer_table_class)
                    .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                    .build())

# Table listing reference lines of intermediate intensity uniformly distributed on the detectors
line_intmon_table = (data_source()
                     .with_classification_rule(line_intmon_table_class)
                     .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                     .build())

# Masks for radial velocity correction (fibre observations only)
corvel_mask = (data_source()
               .with_classification_rule(corvel_mask_class)
               .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
               .build())

# Table with flux standard reference spectra
flux_std_table = (data_source()
                  .with_classification_rule(flux_std_table_class)
                  .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                  .build())

# Table with extinction coefficients
extcoeff_table = (data_source()
                  .with_classification_rule(extcoeff_table_class)
                  .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                  .build())

# Catalogue of points to fit the response curve
resp_fit_points_catalog = (data_source()
                           .with_classification_rule(resp_fit_points_catalog_class)
                           .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                           .build())

# Table that specifies the regions where to compute instrument efficiency
efficiency_windows = (data_source()
                      .with_classification_rule(efficiency_windows_class)
                      .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                      .build())

# calibrations to define areas of the detector where to compute the image quality, fit
quality_areas = (data_source()
                 .with_classification_rule(quality_areas_class)
                 .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                 .build())

fit_areas = (data_source()
             .with_classification_rule(fit_areas_class)
             .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
             .build())

# catalogue of telluric lines
tell_mod_catalog = (data_source()
                    .with_classification_rule(tell_mod_catalog_class)
                    .with_match_keywords(instrument, time_range=UNLIMITED, level=0)
                    .build())

# Reference format check frames (redl/u, blue, and fibre)
masterform_redl = (data_source()
                   .with_classification_rule(master_form_redl_class)
                   .with_match_keywords(arm_binning + [kwd.det_readout, kwd.ins_mode, kwd.grat2_wlen, kwd.instrume],
                                        time_range=UNLIMITED, level=0)
                   .build())

masterform_redu = (data_source()
                   .with_classification_rule(master_form_redu_class)
                   .with_match_keywords(arm_binning + [kwd.det_readout, kwd.ins_mode, kwd.grat2_wlen, kwd.instrume],
                                        time_range=UNLIMITED, level=0)
                   .build())

masterform_blue = (data_source()
                   .with_classification_rule(master_form_blue_class)
                   .with_match_keywords(arm_binning + [kwd.det_readout, kwd.ins_mode, kwd.grat1_wlen, kwd.instrume],
                                        time_range=UNLIMITED, level=0)
                   .build())

masterform_redl_mos = (data_source()
                       .with_classification_rule(master_form_redl_class)
                       .with_match_keywords(arm_binning + [kwd.ins_mode, kwd.grat2_wlen, kwd.slit3_plate],
                                            time_range=UNLIMITED, level=0)
                       .build())

masterform_redu_mos = (data_source()
                       .with_classification_rule(master_form_redu_class)
                       .with_match_keywords(arm_binning + [kwd.ins_mode, kwd.grat2_wlen, kwd.slit3_plate],
                                            time_range=UNLIMITED, level=0)
                       .build())

# master responses delivered with CalSelector, obtained by merging the response of several nights (redu/l, blue)
# To be used as alternatives to the responses obtained from the standard star observed at night.
master_response_red_match_keywords = [kwd.grat2_id, kwd.grat2_wlen, kwd.instrume]
master_response_redu = (data_source()
                        .with_classification_rule(master_response_redu_class)
                        .with_match_keywords(master_response_red_match_keywords, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(master_response_red_match_keywords, time_range=UNLIMITED, level=3)
                        .build())

master_response_redl = (data_source()
                        .with_classification_rule(master_response_redl_class)
                        .with_match_keywords(master_response_red_match_keywords, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(master_response_red_match_keywords, time_range=UNLIMITED, level=3)
                        .build())

master_response_blue_match_keywords = [kwd.grat1_id, kwd.grat1_wlen, kwd.instrume]
master_response_blue = (data_source()
                        .with_classification_rule(master_response_blue_class)
                        .with_match_keywords(master_response_blue_match_keywords, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(master_response_blue_match_keywords, time_range=UNLIMITED, level=3)
                        .build())

# Files with detector pattern noise (redl/u, blue)
pattern_detection_match_keywords = binning + [kwd.det_readout, kwd.det_chips]
pattern_detection_redl = (data_source()
                          .with_classification_rule(pd_mask_redl_class)
                          .with_match_keywords(pattern_detection_match_keywords, time_range=UNLIMITED, level=0)
                          .build())

pattern_detection_redu = (data_source()
                          .with_classification_rule(pd_mask_redu_class)
                          .with_match_keywords(pattern_detection_match_keywords, time_range=UNLIMITED, level=0)
                          .build())

pattern_detection_blue = (data_source()
                          .with_classification_rule(pd_mask_blue_class)
                          .with_match_keywords(pattern_detection_match_keywords, time_range=UNLIMITED, level=0)
                          .build())
