# AREA A1: the import statements
# import statements needed by esoreflex
import reflex
import sys
from argparse import ArgumentParser
import json
import os
import importlib
import json
import copy
import re

# Import statements needed by the 
# algorithm to execute
from astropy.io import fits as fits
import numpy as np

import sop_force_include_H2O

# END OF AREA A1
# AREA A2: the algorithm(s).
# END OF AREA A2
# AREA A3: the interface.
# This is the main Python script. It contains 2 general parts and a
# customized part, that is responsible of calling the desired
# reduction function.
# ------------------------------------------------------------------------------------------
def do_stuff(
        filename,
        previous_inst_setup='None',
        files=None,
        ParameterInitialization=False,
        recipe=None,
    ):
    hdu = fits.open(filename,method='readonly')
    header = hdu[0].header
    instrument=header['INSTRUME']
    
    try:
        # First try for a user version -- this allows users to "easily" set up support
        # for new instruments placing their python scripts somewhere a 'my_molecfit' directory
        # in a sys.path directory, e.g.:
        # on macOS:
        #   $HOME/Library/Python/3.6/lib/python/site-packages/my_molecfit
        # or on standard Linuxs
        #   $HOME/.local/lib/python3.6/site-packages/my_molecfit
        sys.path.insert(0, "%s/KeplerData/workflows/MyWorkflows/my_molecfit" %(os.environ.get('HOME')))
        inst = importlib.import_module('%s' %(instrument.lower()))
    except ImportError :
        try:
            # Then try the version that should be there in an installation...
            # When installed, the <inst>.py scripts are copied into the same directory
            # as this script...
            inst = importlib.import_module('%s' %(instrument.lower()))
        except ImportError :
            # then try the version that should be in instruments/ because in the
            # source the <inst>.py scripts are still in the instruments/ directory
            try:
                sys.path.insert(0, '%s/instruments' %(os.path.dirname(__file__)))
                inst = importlib.import_module('%s' %(instrument.lower()))
            except ImportError :
                raise Exception('%s not (yet?) supported.' % (instrument))
    except Exception as e :
        raise Exception(e)

    return (
        inst.set_inst_setup(header),
        inst.check_idp(
            hdu,
            previous_inst_setup=previous_inst_setup,
            files=files,
            ParameterInitialization=ParameterInitialization,
            recipe=recipe,
        )
    )

# ------------------------------------------------------------------------------------------
if __name__ == '__main__':

    # ***  PART P1: Input/output ports ***

    #Define inputs/outputs
    parser = reflex.ReflexIOParser()
    parser.add_option("-i", "--in_sof", dest="in_sof")
    parser.add_option("--previous_inst_setup", dest="previous_inst_setup")
    parser.add_option("--ParameterInitialization", dest="ParameterInitialization")
    parser.add_option("-p", "--in_sop", dest="in_sop")

    parser.add_output("-o", "--out_sop", dest="out_sop")
    parser.add_output("--set_init_sop", dest="set_init_sop")
    parser.add_output("-q", "--out_sof", dest="out_sof")
    parser.add_output("--inst_setup", dest="inst_setup")

    inputs = parser.get_inputs()
    outputs = parser.get_outputs()
    in_sof = inputs.in_sof
    files = in_sof.files

    #Define the list of outputs
    output_files=list()
    output_datasetname=in_sof.datasetName

    #Get the name of the output directory
    pattern = '^--products-dir='
    for arg in sys.argv:
        m=re.match(pattern, arg)
        if m is not None :
            output_dir = re.sub(pattern, '', arg)

    # ***  END OF PART P1 ***

    # ***  PART P2: customized part***

    inst_setup=None
    o_attrs={}
    files_dict={}
    mf_cats={
        'REDUCED_IDP_SCI_LSS': "SCIENCE",           # fors
        'REDUCED_IDP_SCI_MOS': "SCIENCE",           # fors
        'REDUCED_IDP_STD_MOS': "SCIENCE",           # fors
        'REDUCED_IDP_SCI_MXU': "SCIENCE",           # fors
        "SCI_SLIT_FLUX_IDP_VIS": "SCIENCE",         # xshoo
        "SCI_SLIT_FLUX_IDP_NIR": "SCIENCE",         # xshoo
        "TELL_SLIT_FLUX_IDP_VIS": "STD_MODEL",      # xshoo
        "TELL_SLIT_FLUX_IDP_NIR": "STD_MODEL",      # xshoo
    }
    for file in files:
        if file.category not in files_dict.keys() :
            # write the first file of each category to files_dict
            # In the case where there would be multiple SCIENCE
            # molecfit_model would anyway only use the first one...
            files_dict[mf_cats.get(file.category,file.category)]=file
    ## molecfit_model only works on a single file, so we only bother normalizing
    ## the one the molecfit_model would select, and it selects in order the first file
    ## it finds in the SOF of STD_MODEL then SCIENCE_CALCTRANS then SCIENCE.
    valid_categories=[]
    recipe=''
    if len(inputs.in_sop) > 0 :
        recipe=inputs.in_sop[0].recipe

    nfiles=None
    if len(files) > 0 :
        if recipe in [
            'molecfit_model',
            'fors_molecfit_model',
            'xsh_molecfit_model'
        ] :
            nfiles=[files_dict.get('STD_MODEL',files_dict.get('SCIENCE_CALCTRANS',files_dict.get('SCIENCE',None))),]
            valid_categories=['SCIENCE','SCIENCE_CALCTRANS','STD_MODEL',]
        elif recipe in [
            'molecfit_calctrans',
            '_fors_molecfit_calctrans',
            '_xsh_molecfit_calctrans',
        ] :
            # Do nothing for 'xsh_molecfit_calctrans', remove _ to enable doing something...
            nfiles=[files_dict.get('SCIENCE_CALCTRANS',files_dict.get('SCIENCE',None)),]
            valid_categories=['SCIENCE','SCIENCE_CALCTRANS',]
        elif recipe in [
            'molecfit_correct',
            '_fors_molecfit_correct',
            '_xsh_molecfit_correct',
        ] :
            # Do nothing for 'xsh_molecfit_correct', remove _ to enable doing something...
            nfiles=files
            valid_categories=['SCIENCE','SCIENCE_CALCTRANS',]

    if nfiles is not None and nfiles != [None,] :
        for file in nfiles:
            if file.category in valid_categories or file.category in mf_cats.keys() :
                while inst_setup is None :
                    inst_setup,o_attrs = do_stuff(
                        file.name,
                        #previous_inst_setup=inputs.previous_inst_setup, # force a fll setup each time now that we have save_params working...
                        files=files,
                        ParameterInitialization=(inputs.ParameterInitialization.lower() in ['true',]),
                        recipe=recipe,
                    )
                output_files.append(reflex.FitsFile(file.name, file.category, None, file.purposes))
        for file in files:
            if file.category not in ['SCIENCE','SCIENCE_CALCTRANS','STD_MODEL',] :
                output_files.append(reflex.FitsFile(file.name, file.category, None, file.purposes))
    else :
        output_files=files

    # *** END OF PART P2***

    # *** PART P3: broadcast products ***
  
    # After the script is completed, the list of products is
    # broadcasted to the output sof.
    new_sof = reflex.SetOfFiles(output_datasetname,output_files)
    outputs.out_sof = new_sof    

    # out_sop with all parameters to pass to molecfit_model...
    out_sop=inputs.in_sop
    # set_init_sop with just modified parameters to pass to
    # reflex.actor.SetInitialLoopParam
    # because that reflex.actor.SetInitialLoopParam is SLOW...
    set_init_sop=[]

    sop_keynames=[rp.name for rp in inputs.in_sop]
    for _d in [o_attrs,o_attrs.get('ignore_if_not_NULL',{})] :
        for k in _d.keys() :
            if _d[k] is not None :
                if k in sop_keynames :
                    if inputs.in_sop[sop_keynames.index(k)].valtype == 'bool' :
                        if isinstance(_d[k],str) :
                            _d[k]=_d[k].lower() == 'true'
                    if _d[k] != inputs.in_sop[sop_keynames.index(k)].value :
                        out_sop[sop_keynames.index(k)].value=_d[k]
                        set_init_sop+=[inputs.in_sop[sop_keynames.index(k)],]
                        set_init_sop[-1].value=_d[k]

    out_sop=sop_force_include_H2O.sop_force_include_H2O( out_sop )
    
    outputs.out_sop=out_sop
    outputs.set_init_sop=set_init_sop
    outputs.inst_setup=inst_setup or 'None'

    # Broadcast outputs on the extra output port.
    # broadcast outputs:
    parser.write_outputs()
    sys.exit()
    # ***  END OF PART P3 ***
    # ***  END OF AREA A3***    
