import os
import re
import numpy as np
from astropy.io import fits
import io

# ---------------------------------------------------------------------------------------
# The following method controls the format of the values written to
# WAVE_INCLUDE & WAVE_EXCLUDE parameters
# This is important when you need LOTS of includes and/or excludes
# because there is some kind of string-truncation error somewhere
# I think in the molecfit/cpl code that cuts off values after a
# certain point
# On macOS by default it writes as %8.6f, but in linux it writes as %18.16f
# Limiting the format to %9.7f should allow twice as many intervals as is possible
# with %18.16f.
def wrfmt(x) :
    if isinstance(x, (int,np.int64)) :
        return "{x:1d}".format(x=x)
    return "{x:9.7f}".format(x=x).rstrip("0")
# ------------------------------------------------------------------------------------------
WL_TO_MICRONS={
    "micron": 1.,
    "nm": 1.e-3,
    "Angstrom": 1.e-4,
    "UNKNOWN": 1.,
}
# ------------------------------------------------------------------------------
def select_by_WL_range(x,wl,r) :
    return x[np.where(np.logical_and(wl>=r[0],wl<=r[1]))]
# ------------------------------------------------------------------------------------------
def wl_to_microns( hdu, default=None, inst_kws=None ) :
    """
    return the conversion factor from WL scale in hdu to micron
    based on scale specified in CUNIT1 or TUNIT1 header keywords in
    hdu.
    default = default value if no
    """
    kws=inst_kws
    if inst_kws is None :
        if isinstance(hdu, fits.BinTableHDU ) :
            kws=['TUNIT1',]
        else :
            kws=['CUNIT1',]
    wlg=default or WL_TO_MICRONS['UNKNOWN']
    found_scale=False
    for kw in kws :
        if kw in hdu.header :
            if hdu.header[kw] not in WL_TO_MICRONS.keys() :
                print('WARN : Unknown wavelength scale from header keyword %s=%s' %(kw,hdu.header[kw]))
            else :
                wlg=WL_TO_MICRONS.get(hdu.header[kw],'UNKNOWN')
                found_scale=True
                break
    if not found_scale :
        print('WARN : could not determine wavelength scale, using default value %f' %(wlg))
    return wlg
# ------------------------------------------------------------------------------------------
def propagate_keys(file, propagate_keys_from_recipes, orig_file=None ) :
    # Get all the keys for the first file...
    propagate_keys_from=file.name
    pkf=fits.open(propagate_keys_from)
    print("Getting PRO RECi keys from %s" %propagate_keys_from)
    # Find the keys for the recipe we need to propagate for...
    hdr=fits.Header()
    match_str={}
    for pkf_recipe in propagate_keys_from_recipes :
        for k in pkf[0].header :
            if (
                re.match('ESO PRO REC[0-9]? ID',k)
                and
                pkf[0].header[k].strip() == pkf_recipe
            ) :
                match_str[pkf_recipe]=re.sub(' ID','',k)
        if pkf_recipe in match_str :
            for k in pkf[0].header :
                if re.match(match_str[pkf_recipe],k) :
                    hdr.set('HIERARCH %s' %k,pkf[0].header[k],pkf[0].header.comments[k])
    pkf.close()
    if orig_file is None :
        # And now add in the RAWi * keys from all the other files...
        for pkf_recipe in ["molecfit_correct",] :
            i=1
            for file in files[1:] :
                pkf=fits.open(file.name)
                kprev="%s RAW%d CATG" %(match_str[pkf_recipe],i)
                i+=1
                for ksuf in ["NAME","CATG",] :
                    print("Getting RAW%d %s keys from %s" %(i,ksuf,file.name))
                    k="%s RAW1 %s" %(match_str[pkf_recipe],ksuf)
                    knew="%s RAW%d %s" %(match_str[pkf_recipe],i,ksuf)
                    try :
                        hdr.set('HIERARCH %s' %knew,pkf[0].header[k],pkf[0].header.comments[k],after=kprev)
                    except ValueError :
                        nchars=len(pkf[0].header[k])-(len(knew)-len(k))
                        hdr.set('HIERARCH %s' %knew,pkf[0].header[k][:nchars],pkf[0].header.comments[k],after=kprev)
                    kprev="%s RAW%d %s" %(match_str[pkf_recipe],i,ksuf)
            pkf.close()
    else :
        i=1
        for ksuf in ["NAME","CATG",] :
            bnfn=os.path.basename(orig_file.name)
            print("Setting RAW%d %s keys from %s" %(i,ksuf,orig_file.name))
            k="%s RAW1 %s" %(match_str[pkf_recipe],ksuf)
            knew="%s RAW%d %s" %(match_str[pkf_recipe],i,ksuf)
            if ksuf == "NAME" :
                nchars=np.min([len(bnfn),80-(len("HIERARCH ")+len(k)+len(" = ''"))])
                hdr.set(
                    'HIERARCH %s' %knew,
                    bnfn[:nchars],
                    pkf[0].header.comments[k]
                )
            else :
                hdr.set(
                    'HIERARCH %s' %knew,
                    re.sub('^ORIG_','',orig_file.category),
                    pkf[0].header.comments[k]
                )

    return hdr
# ------------------------------------------------------------------------------------------
# =====================================================================================

def DER_SNR(flux):
   
# =====================================================================================
   """
   DESCRIPTION This function computes the signal to noise ratio DER_SNR following the
               definition set forth by the Spectral Container Working Group of ST-ECF,
           MAST and CADC. 

               signal = median(flux)      
               noise  = 1.482602 / sqrt(6) median(abs(2 flux_i - flux_i-2 - flux_i+2))
           snr    = signal / noise
               values with padded zeros are skipped

   USAGE       snr = DER_SNR(flux)
   PARAMETERS  none
   INPUT       flux (the computation is unit independent)
   OUTPUT      the estimated signal-to-noise ratio [dimensionless]
   USES        numpy      
   NOTES       The DER_SNR algorithm is an unbiased estimator describing the spectrum 
           as a whole as long as
               * the noise is uncorrelated in wavelength bins spaced two pixels apart
               * the noise is Normal distributed
               * for large wavelength regions, the signal over the scale of 5 or
             more pixels can be approximated by a straight line
 
               For most spectra, these conditions are met.

   REFERENCES  * ST-ECF Newsletter, Issue #42:
               www.spacetelescope.org/about/further_information/newsletters/html/newsletter_42.html
               * Software:
           www.stecf.org/software/ASTROsoft/DER_SNR/
   AUTHOR      Felix Stoehr, ST-ECF
               24.05.2007, fst, initial import
               01.01.2007, fst, added more help text
               28.04.2010, fst, return value is a float now instead of a numpy.float64
   """
   from numpy import array, where, median, abs 

   flux = array(flux)

   # Values that are exactly zero (padded) are skipped
   flux = array(flux[where(flux != 0.0)])
   n    = len(flux)      

   # For spectra shorter than this, no value can be returned
   if (n>4):
      signal = median(flux)

      noise  = 0.6052697 * median(abs(2.0 * flux[2:n-2] - flux[0:n-4] - flux[4:n]))

      return float(signal / noise)  

   else:

      return 0.0

# end DER_SNR -------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
# =====================================================================================
class HDUList(fits.HDUList) :
    '''
    A simple wrapper to cope with the change from clobber to overwrite in
    astropy.io.fits.HDUList.writeto() as of astropy version 2.0 allowing one
    to use overwrite on all version of astropy.
    Usage:
    from astropy.io import fits
    import ins_utils as iu
    iu.HDUList(fits.HDUList()).writeto(..., overwrite=T/F, ...)
    '''
    def __init__(self, *args, **kwargs):
        super(HDUList, self).__init__( *args, **kwargs)

    def writeto(self, *args, **kwargs):
        import astropy
        from distutils.version import StrictVersion
        if StrictVersion(astropy.__version__) < StrictVersion('2.0') :
            if 'overwrite' in kwargs.keys() :
                kwargs['clobber']=kwargs['overwrite']
                del kwargs['overwrite']
        fits.HDUList(self).writeto(*args, **kwargs)
# ------------------------------------------------------------------------------------------
def import_inst(instrument) :
    import sys
    import os
    import importlib
    try:
        # First try for a user version -- this allows users to "easily" set up support
        # for new instruments placing their python scripts somewhere a 'my_molecfit' directory
        # in a sys.path directory, e.g.:
        # on macOS:
        #   $HOME/Library/Python/3.6/lib/python/site-packages/my_molecfit
        # or on standard Linuxs
        #   $HOME/.local/lib/python3.6/site-packages/my_molecfit
        sys.path.insert(0, "%s/KeplerData/workflows/MyWorkflows/my_molecfit" %(os.environ.get('HOME')))
        inst = importlib.import_module('%s' %(instrument.lower()))
    except ImportError :
        try:
            inst = importlib.import_module('%s' %(instrument.lower()))
        except ImportError :
            # ...but when run in devel mode, the <inst>.py scripts are still in the instruments/ directory
            try:
                sys.path.insert(0, '%s/instruments' %(os.path.dirname(__file__)))
                inst = importlib.import_module('%s' %(instrument.lower()))
            except ImportError :
                raise Exception('%s not (yet?) supported.' % (instrument))
    except Exception as e :
        raise(e)
    return inst
# ------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------

# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# From telluriccorr-4.2.0a4/src/mf_spectrum.c
'''
if (!strcmp(frame_type, MF_PARAMETERS_WAVELENGTH_FRAME_VACUUM_RV)) {

    /* The input wavelength has been corrected for the Earth motion relative to an external reference frame */
    for (cpl_size i = 0; i < nrow; i++) {

        double n = (1. + 1.55e-8) * (1. + obs_erf_rv / 299792.458);

        lambda[i] /= n;
    }
} else if (!strcmp(frame_type, MF_PARAMETERS_WAVELENGTH_FRAME_AIR)) {

    /* Converts air wavelengths to vacuum wavelengths by using the formula of Edlen (1966) : Get refractive index and convert wavelengths. */
    for (cpl_size i = 0; i < nrow; i++) {

        double sig2 = pow(lambda[i], -2);
        double n    = 8342.13 + 2406030. / (130. - sig2) + 15997. / (38.9 - sig2);

        n = 1. + 1e-8 * n;

        lambda[i] *= n;
    }
'''
def bary_to_topo( wl, obs_erf_rv) :
    """
    Convert barycentric wl to topocentric wl
    [wl]
    [obs_erf_rv] = km/s
    """
    nrv=(1. + 1.55e-8) * (1. + obs_erf_rv / 299792.458)
    return wl/nrv
# ------------------------------------------------------------------------------------------
def topo_to_bary( wl, obs_erf_rv) :
    """
    Convert topocentric wl to barycentric wl
    [wl]
    [obs_erf_rv] = km/s
    """
    nrv=(1. + 1.55e-8) * (1. + obs_erf_rv / 299792.458)
    return wl*nrv
# ------------------------------------------------------------------------------------------
def air_ref_index( wl ) :
    """
    Calculate refacrtive index of air
    [wl] = micron
    """
    sig2=np.power(wl,-2.)
    n=8342.13 + 2406030. / (130. - sig2) + 15997. / (38.9 - sig2)
    n = 1. + 1e-8 * n
    return n
# ------------------------------------------------------------------------------------------
def air_to_vac( wl, wlg=1. ) :
    """
    Convert wl in air to wl in vacuum
    [wl] = micron
    wlg conversion factor to micron
     [um] 1.
     [nm] 1.e-3
     [AA] 1.e-4
    """
    return wl*air_ref_index(np.array(wl)*wlg)
# ------------------------------------------------------------------------------------------
def vac_to_air( wl, wlg=1. ) :
    """
    Convert wl in vacuum to wl in air
    [wl] = micron
    wlg conversion factor to micron
     [um] 1.
     [nm] 1.e-3
     [AA] 1.e-4
    """
    return wl/air_ref_index(np.array(wl)*wlg)
# ---------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
def heli_topo_correction_factor( gcorr, hcorr ) :
    """
    # from giscience.c
        /* Calculate the Heliocentric correction factor to apply.
         * The gcorr and hcorr values are in km/s, so must be converted to m/s.
         */
        vela = gcorr[i] * 1e3;
        velb = hcorr[i] * 1e3;
        beta = (vela + velb) / CPL_PHYS_C;
        cpl_error_ensure(-1 <= beta && beta <= 1,
                         CPL_ERROR_ILLEGAL_OUTPUT, goto cleanup,
                         "The velocities GCORR = %g and HCORR = %g for spectrum"
                         "%"CPL_SIZE_FORMAT" in file '%s' give invalid"
                         " Heliocentric correction factor values.",
                         gcorr[i], hcorr[i], specindex, flux_filename);
        correction_factor = sqrt((1.0 + beta) / (1.0 - beta));

        /* Calculate and set corrected wavelength array, remembering to cast
         * to the appropriate type. */
        wave_data = cpl_array_get_data_double_const(refwavearray);
        if (wavecoltype == CPL_TYPE_FLOAT) {
            data_float = cpl_malloc(ny * sizeof(float));
            for (j = 0; j < ny; ++j) {
                data_float[j] = wave_data[j] * correction_factor;
            }
            array = cpl_array_wrap_float(data_float, ny);
        } else {
            data_double = cpl_malloc(ny * sizeof(double));
            for (j = 0; j < ny; ++j) {
                data_double[j] = wave_data[j] * correction_factor;
            }
            array = cpl_array_wrap_double(data_double, ny);
        }
        error |= irplib_sdp_spectrum_set_column_data(
                      spectrum, GIALIAS_COLUMN_WAVE, array);
        cpl_array_unwrap(array);
        array = NULL;
    """
    from astropy.constants import c as CPL_PHYS_C
    vela = gcorr * 1e3
    velb = hcorr * 1e3
    beta = (vela + velb) / CPL_PHYS_C.to('m/s').value
    return np.sqrt((1.0 + beta) / (1.0 - beta));
# ------------------------------------------------------------------------------------------
def heli_to_topo( wl, gcorr, hcorr ) :
    """
    wl wavelengths
    gcorr : geo [km/s]
    hcorr : helio [km/s]
    """
    return np.array(wl)/heli_topo_correction_factor(gcorr, hcorr)
# ------------------------------------------------------------------------------------------
def topo_to_heli( wl, gcorr, hcorr ) :
    """
    wl wavelengths
    gcorr : geo [km/s]
    hcorr : helio [km/s]
    """
    return np.array(wl)*heli_topo_correction_factor(gcorr, hcorr)
# ------------------------------------------------------------------------------------------
def heli_air_to_topo_vac( wl, gcorr, hcorr, wlg=1. ) :
    return air_to_vac(heli_to_topo(wl, gcorr, hcorr),wlg)
# ------------------------------------------------------------------------------------------
def topo_vac_to_heli_air( wl, gcorr, hcorr, wlg=1. ) :
    return topo_to_heli(vac_to_air(wl,wlg), gcorr, hcorr)
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
def read_file_contents(*paths, **kwargs):
    """Read the contents of a text file safely.
    >>> read("dpapi", "VERSION")
    '0.1.0'
    >>> read("README.md")
    ...
    """

    content = ""
    with io.open(
        os.path.join(os.path.dirname(__file__), *paths),
        encoding=kwargs.get("encoding", "utf8"),
    ) as open_file:
        content = open_file.read().strip()
    return content
# ------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
