import os
import sys
import numpy as np
from astropy.io import fits
import reflex
import importlib
import re
import copy

try :
    am = importlib.import_module('auto_molecule')
    iu = importlib.import_module('inst_utils')
except ImportError:
    sys.path.insert(0, '%s/instruments' %(os.path.dirname(__file__)))
    am = importlib.import_module('auto_molecule')
    iu = importlib.import_module('inst_utils')
except Exception as e:
    raise(e)

XS_slit_width_keywords={
    "UVB": "ESO INS OPTI3 NAME",
    "VIS": "ESO INS OPTI4 NAME",
    "NIR": "ESO INS OPTI5 NAME",
    "UNKNOWN": "1.0x11",
}
# ------------------------------------------------------------------------------------------
def dataset_chooser_keywords() :
    return 'INSTRUME,OBJECT,SEQ.ARM,MJD-OBS,INS.PATH,INS.OPTI4.NAME,INS.OPTI5.NAME,PRO.CATG'
# ------------------------------------------------------------------------------------------
def set_inst_setup(header=None) :
  if header :
    return "%s_%s" %(
      header.get('HIERARCH ESO SEQ ARM', 'UNKNOWN'),
      header.get(
        XS_slit_width_keywords[header.get("ESO SEQ ARM","UNKNOWN")],
        XS_slit_width_keywords["UNKNOWN"],
      ),
    )
  return 'None'
# ------------------------------------------------------------------------------------------
def check_format(files,output_dir):
    output_files=list()
    files_dict={}
    for file in files:
        if file.category not in files_dict.keys() :
            files_dict[file.category]=[]
        files_dict[file.category]+=[file,]
    has_sci_and_std=(
        ('SCIENCE' in files_dict.keys() or 'SCIENCE_CALCTRANS' in files_dict.keys())
        and
        'STD_MODEL' in files_dict.keys()
    )
    for k in files_dict.keys() :
        for file in files_dict[k] :
            if k in ['STD_MODEL','SCIENCE','SCIENCE_CALCTRANS',] :
                prefix=re.sub('_.*','',k)
                # If this is not an IFU file pass through...
                # If this is a an IFU file then:
                #    a) check for the RBNERRORS file and append data as appropriate...
                #    b) select/prepare a single file for model+calctrans
                #    c) prepare the individual fibres for correct
                hdulist=fits.open(file.name)
                procatg=hdulist[0].header.get('HIERARCH ESO PRO CATG','NULL')
                if procatg in [
                      'SCI_IFU_MERGE3D_DATA_OBJ_VIS',
                      'SCI_IFU_MERGE3D_DATA_OBJ_NIR',
                      'FLUX_IFU_MERGE3D_DATA_OBJ_VIS',
                      'FLUX_IFU_MERGE3D_DATA_OBJ_NIR',
                      'TELL_IFU_MERGE3D_DATA_OBJ_VIS',
                      'TELL_IFU_MERGE3D_DATA_OBJ_NIR',
                    ] :

                    # reformat it into a MolecFit Compliant [_MFC] file...
                    model_file=extract_2d_rows_ifu_pixels(
                        file.name, output_dir,
                    )

                    # broadcast the file to use for molecfit_model...
                    if (
                        ( has_sci_and_std and k == "STD_MODEL" )
                        or
                        ( not has_sci_and_std )
                    ) :
                        output_files.append(reflex.FitsFile(
                            model_file, "STD_MODEL", model_file[1], file.purposes
                        ))

                    # broadcast the file to use for molecfit_calctrans...
                    if (
                        ( has_sci_and_std and k != "STD_MODEL" )
                        or
                        ( not has_sci_and_std )
                    ) :
                        output_files.append(reflex.FitsFile(
                            model_file, "SCIENCE_CALCTRANS", model_file[1], file.purposes
                        ))

                    # broadcast the original frame (for sticking it all back together at the end...)
                    output_files.append(reflex.FitsFile(
                        file.name, "ORIG_%s" %(file.category), None, file.purposes
                    ))
                else :
                    # broadcast the original file it...
                    output_files.append(reflex.FitsFile(file.name, file.category, None, file.purposes ))
            else :
                # broadcast it...
                output_files.append(reflex.FitsFile(file.name, file.category, None, file.purposes ))

    return output_files
# ------------------------------------------------------------------------------------------
def set_parameters(header=None, hdu=None, previous_inst_setup='None', files=None) :
    wframe='AIR'
    list_mol='NULL'
    fit_mol='NULL'
    rel_col='NULL'
    clam='WAVE'
    cflux='FLUX'
    cdflux='ERR'
    fit_tel_back="False"
    arm=header.get('HIERARCH ESO SEQ ARM',"UNKNOWN")
    '''
    if arm == 'NIR':
        fit_tel_back="False"
    else:
        fit_tel_back="False"
    '''
    cont_poly = int(1)
    varkern="True"
    wave_include='NULL'

    return {
        'WAVELENGTH_FRAME': wframe,
        'LIST_MOLEC': list_mol,
        'FIT_MOLEC': fit_mol,
        'REL_COL': rel_col,
        'COLUMN_LAMBDA': clam,
        'COLUMN_FLUX': cflux,
        'COLUMN_DFLUX': cdflux,
        'FIT_TELESCOPE_BACKGROUND': fit_tel_back,
        'CONTINUUM_N': cont_poly,
        'VARKERN': varkern,
        'WAVE_INCLUDE': wave_include,
        'SLIT_WIDTH_KEYWORD': XS_slit_width_keywords[arm],
    }
# ------------------------------------------------------------------------------------------
def check_idp(hdu, previous_inst_setup='None', files=None, ParameterInitialization=False, recipe=None, ) :
    import numpy as np

    inst_setup=set_inst_setup(hdu[0].header)
    if inst_setup != previous_inst_setup :
      print("Detected change of instrument setup.")

    wlg = default_wlg = 0.001 # default value

    no_output=0
    # these are the values to use for XSHOOTER IDP spectra
    clam='WAVE'
    cflux='FLUX'
    cdflux='ERR'
    primary='FALSE'
    errorext=0
    maskext=0
    map_convolve="1,1"
    map_atmosphere="0,1"
    map_correct="0,1"

    if hdu[0].header.get('HIERARCH ESO PRO CATG','NULL') in [
        "SCIENCE.SPECTRUM",
        "SCI_SLIT_FLUX_IDP_VIS",
        "SCI_SLIT_FLUX_IDP_NIR",
        "TELL_SLIT_FLUX_IDP_VIS",
        "TELL_SLIT_FLUX_IDP_NIR",
    ] :
      dd=hdu[1].data['FLUX']
      wl_range=np.array([np.min(hdu[1].data[clam][0]),np.max(hdu[1].data[clam][0])])
      wl_resolution=np.min(hdu[1].data[clam][0][1:]-hdu[1].data[clam][0][0:-1])
      wlg=iu.wl_to_microns(hdu[1],default_wlg)
    else :
      # these are the values to use for XSHOOTER spectra in generic PL format (non IDP)
      # Assume the FLUX is in the primary header...
      primary='TRUE'
      # Check other HDUs, if present, for ERRS and QUAL extensions...
      extnames={
        'ERRS': None,
        'QUAL': None,
      }
      if len(hdu) > 1 :
        for i,h in enumerate(hdu[1:]) :
          if h.header.get('EXTNAME') in extnames.keys() :
            extnames[h.header.get('EXTNAME')] = i+1
      errorext=extnames.get('ERRS',0)
      maskext=extnames.get('QUAL',0)
      map_convolve="1"
      map_atmosphere="1"
      map_correct="1"
      WAVE_AXIS=1
      for i in range(hdu[0].header['NAXIS']) :
         if hdu[0].header.get('CTYPE%d' %(i+1),'').strip() == 'WAVE' :
            WAVE_AXIS=i+1
      try:
        dd=hdu[0].data
        wl_range=np.array([
          hdu[0].header['CRVAL%d' %(WAVE_AXIS)],hdu[0].header['CRVAL%d' %(WAVE_AXIS)]
          +
          hdu[0].header['NAXIS%d' %(WAVE_AXIS)]*hdu[0].header.get('CD%d_%d' %(WAVE_AXIS,WAVE_AXIS),hdu[0].header.get('CDELT%d' %(WAVE_AXIS)))
        ])
        wl_resolution=hdu[0].header.get('CD%d_%d' %(WAVE_AXIS,WAVE_AXIS),hdu[0].header.get('CDELT%d' %(WAVE_AXIS)))
        wlg=iu.wl_to_microns(hdu[0],default_wlg)
      except:         
        dd=hdu[1].data
        wl_range=np.array([hdu[1].header['CRVAL%d' %(WAVE_AXIS)],hdu[1].header['CRVAL%d' %(WAVE_AXIS)]+hdu[1].header['NAXIS%d' %(WAVE_AXIS)]*hdu[1].header.get('CD%d_%d' %(WAVE_AXIS,WAVE_AXIS),hdu[1].header.get('CDELT%d' %(WAVE_AXIS)))])
        wl_resolution=hdu[1].header.get('CD%d_%d' %(WAVE_AXIS,WAVE_AXIS),hdu[1].header.get('CDELT%d' %(WAVE_AXIS)))
        wlg=iu.wl_to_microns(hdu[0],default_wlg)
    median_cont = float(np.median(dd[np.where(~np.isnan(dd))]))

    slit_width=hdu[0].header.get(
      XS_slit_width_keywords[hdu[0].header.get("ESO SEQ ARM","UNKNOWN")],
      XS_slit_width_keywords["UNKNOWN"],
    ).split('x')[0]

    WL_inc_exc_PIX_exc={}
    inst_setup_changed = set_inst_setup(hdu[0].header) != previous_inst_setup
    if ParameterInitialization :
        ## ---------------------------------------------------------------------
        # Automatically set molecules and ranges...
        wl_span=wl_range[1]-wl_range[0]

        WL_ranges_to_include=am.auto_molecules_set_WL_ranges_to_include(
        files,
        wl_range,
        wl_span,
        wl_resolution,
        wl_scale=1./wlg,
        buff=wl_span*0.05,
        )
        WL_inc_exc_PIX_exc={
            'LIST_MOLEC': WL_ranges_to_include['list_mol'],
            'FIT_MOLEC': WL_ranges_to_include['fit_mol'],
            'REL_COL': WL_ranges_to_include['rel_col'],
            'WAVE_INCLUDE': WL_ranges_to_include['wave_include'],
            'WAVE_EXCLUDE': WL_ranges_to_include['wave_exclude'],
            'PIXEL_EXCLUDE': WL_ranges_to_include['pixel_exclude'],
        }
        ## ---------------------------------------------------------------------

    return dict(
        list(({
          # model settings...
          'WLG_TO_MICRON': wlg,
          'COLUMN_LAMBDA': clam,
          'COLUMN_FLUX': cflux,
          'COLUMN_DFLUX': cdflux,
          'USE_ONLY_INPUT_PRIMARY_DATA': primary,
          'USE_DATA_EXTENSION_AS_DFLUX': errorext,
          'USE_DATA_EXTENSION_AS_MASK': maskext,
          'CONTINUUM_CONST': median_cont,
          'SLIT_WIDTH_KEYWORD': "NONE",
          'SLIT_WIDTH_VALUE': slit_width,
          # calctrans settings...
          'MAPPING_CONVOLVE': map_convolve,
          'MAPPING_ATMOSPHERIC': map_atmosphere,
          # correct settings...
          'MAPPING_CORRECT': map_correct,
        }).items())
        +
        list((WL_inc_exc_PIX_exc if inst_setup_changed else {}).items())
    )
# ------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
# -----   XSHOOTER specific methods   ------------------------------------------------------
# ------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------
def extract_2d_rows_ifu_pixels(
      infile, output_dir,
    ) :

    bn_infile=os.path.basename(infile)
    HDUlist=fits.open(infile)

    supported_pro_catgs=[
      'SCI_SLIT_FLUX_MERGE2D_VIS',
      'SCI_SLIT_FLUX_MERGE2D_NIR',
      'SCI_SLIT_MERGE2D_VIS',
      'SCI_SLIT_MERGE2D_NIR',
      'TELL_SLIT_FLUX_MERGE2D_VIS',
      'TELL_SLIT_FLUX_MERGE2D_NIR',
      'TELL_SLIT_MERGE2D_VIS',
      'TELL_SLIT_MERGE2D_NIR',
      #
      'SCI_IFU_MERGE3D_DATA_OBJ_VIS',
      'SCI_IFU_MERGE3D_DATA_OBJ_NIR',
      'FLUX_IFU_MERGE3D_DATA_OBJ_VIS',
      'FLUX_IFU_MERGE3D_DATA_OBJ_NIR',
      'TELL_IFU_MERGE3D_DATA_OBJ_VIS',
      'TELL_IFU_MERGE3D_DATA_OBJ_NIR',
    ]

    ## Check format of file is as expected:
    # 0) Check PRO.CATG
    if HDUlist[0].header.get('HIERARCH ESO PRO CATG','None') not in supported_pro_catgs :
      #logging.error('PRO.CATG = %s in %s not supported' %(HDUlist[0].header.get('HIERARCH ESO PRO CATG','None'),infile))
      sys.exit(1)

    # 1) Check number of HDUs == 3
    if len(HDUlist) != 3 :
      #debug.error('unexpected number of HDUs [%d] in %s' %(len(HDUlist),infile))
      sys.exit(1)

    # 2) Check number the three expected EXTNAMEs are present
    expected_EXTNAMEs={'FLUX': -1, 'ERRS': -1,'QUAL': -1}
    for (i_hdu,HDU) in enumerate(HDUlist) :
      if HDU.header.get('EXTNAME') in expected_EXTNAMEs.keys() :
        expected_EXTNAMEs[HDU.header['EXTNAME']]=i_hdu
    EXTNAME_not_found=False

    for e in expected_EXTNAMEs :
      if expected_EXTNAMEs[e] < 0 :
        #logging.error('no HDU with EXTNAME = %s found in %s' %(e,infile))
        EXTNAME_not_found=True
      if EXTNAME_not_found:
         sys.exit(1)

    # Checks finished

    nHDUlist=copy.deepcopy(HDUlist)
    _HDUlist=copy.deepcopy(HDUlist)
    # Append the HDUs so that we can re-pack...
    # It's really just so that we save the full header info and image format
    for ext in ['FLUX', 'ERRS', 'QUAL'] :
      nHDUlist.append(_HDUlist[expected_EXTNAMEs[ext]])
      nHDUlist[-1].header['EXTNAME']=_HDUlist[expected_EXTNAMEs[ext]].header['EXTNAME']+"_EXTRACT_2D_ROWS_IFU_PIXELS"
      nHDUlist[-1].data=np.array(np.shape(_HDUlist[expected_EXTNAMEs[ext]].data),dtype=int)

    if HDUlist[0].header['NAXIS'] == 2 :
      ## MERGE2D
      # Fix the WAVE scale keys
      for (i_hdu,HDU) in enumerate(HDUlist) :
        for k in ['CTYPE', 'CRPIX', 'CRVAL', 'CDELT', 'CUNIT'] :
          #nHDUlist[i_hdu].header[k+'1']=HDUlist[i_hdu].header[k+'3']
          if k+'2' in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header[k+'2']
      # Remove the CD matrix
        for j in range(2) :
          for k in range(2) :
            if 'CD%d_%d' %(j+1,k+1) in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header['CD%d_%d' %(j+1,k+1)]

      for j in range(HDUlist[0].header['NAXIS2']) :
        for (i_hdu,HDU) in enumerate(HDUlist) :
          nHDUlist[i_hdu].data = HDU.data[j,:]
          nHDUlist[i_hdu].header['HIERARCH ESO PRO CATG']=HDUlist[0].header['HIERARCH ESO PRO CATG'].replace("MERGE2D","MERGE1D")+'_ROW'
          nHDUlist[i_hdu].header['HIERARCH ESO DRS MOLECFIT PRO CATG']="SCIENCE_CALCTRANS"
          nHDUlist[i_hdu].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 2D NAXIS2 ROW']=j
        fname=bn_infile.replace('.fits','_row_%02d.fits' %(j)).replace("MERGE2D","MERGE1D")
        model_file="%s/%s" %(output_dir,fname)
        iu.HDUList(nHDUlist).writeto(model_file, output_verify="fix+warn", overwrite=True, checksum=True)
        #logging.info('Wrote %s' %(fname))

    elif HDUlist[0].header['NAXIS'] == 3 :
      ## IFU MERGE3D
      # Fix the WAVE scale keys...
      for (i_hdu,HDU) in enumerate(HDUlist) :
        for k in ['CTYPE', 'CRPIX', 'CRVAL', 'CDELT', 'CUNIT'] :
          nHDUlist[i_hdu].header[k+'1']=HDUlist[i_hdu].header[k+'3']
          if k+'2' in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header[k+'2']
          if k+'3' in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header[k+'3']
        # Remove the CD matrix...
        for j in range(3) :
          for k in range(3) :
            if 'CD%d_%d' %(j+1,k+1) in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header['CD%d_%d' %(j+1,k+1)]
        # Remove the SPECSYS key...
        if 'SPECSYS' in nHDUlist[i_hdu].header : del nHDUlist[i_hdu].header['SPECSYS']
      for i in range(HDUlist[0].header['NAXIS1']) :
        # FLUX: simple SUM
        sumFLUX=np.sum(HDUlist[expected_EXTNAMEs['FLUX']].data,axis=1)
        # ERRS: quadratic sum i.e. sqrt.(ERRS*ERRS)
        sumERRS = np.sqrt(
          np.sum(
            HDUlist[expected_EXTNAMEs['ERRS']].data*HDUlist[expected_EXTNAMEs['ERRS']].data
            ,axis=1
          )
        )
        # QUAL: simple SUM, till I know better
        sumQUAL = np.sum(HDUlist[expected_EXTNAMEs['FLUX']].data,axis=1)

        for j in range(HDUlist[0].header['NAXIS2']) :
          for (i_hdu,HDU) in enumerate(HDUlist) :
            nHDUlist[i_hdu].data = HDU.data[:,j,i]
            nHDUlist[i_hdu].header['HIERARCH ESO PRO CATG']=HDUlist[0].header['HIERARCH ESO PRO CATG'].replace("MERGE3D","MERGE1D")+'_PIXEL'
            nHDUlist[i_hdu].header['HIERARCH ESO DRS MOLECFIT PRO CATG']="SCIENCE_CALCTRANS"
            nHDUlist[i_hdu].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 3D PIXEL NAXIS1 I']=i
            nHDUlist[i_hdu].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 3D PIXEL NAXIS2 J']=j
          fname=bn_infile.replace('.fits','_pixel_%01d_%02d.fits' %(i,j)).replace("MERGE3D","MERGE1D")
          iu.HDUList(nHDUlist).writeto("%s/%s" %(output_dir,fname), output_verify="fix+warn", overwrite=True, checksum=True)
          for (i_hdu,HDU) in enumerate(HDUlist) :
            del nHDUlist[i_hdu].header['HIERARCH ESO DRS MOLECFIT PRO CATG']
            del nHDUlist[i_hdu].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 3D PIXEL NAXIS2 J']
          #logging.info('Wrote %s' %(fname))
        # Sum each column of slit....
        nHDUlist[expected_EXTNAMEs['FLUX']].data = sumFLUX[:,i]
        nHDUlist[expected_EXTNAMEs['ERRS']].data = sumERRS[:,i]
        nHDUlist[expected_EXTNAMEs['QUAL']].data = sumQUAL[:,i]
        fname=bn_infile.replace('.fits','_sum_%01d.fits' %(i)).replace("MERGE3D","MERGE1D")
        for (i_hdu,HDU) in enumerate(HDUlist) :
            nHDUlist[i_hdu].header['HIERARCH ESO PRO CATG']=HDUlist[0].header['HIERARCH ESO PRO CATG'].replace("MERGE3D","MERGE1D")+'_SUM'
        iu.HDUList(nHDUlist).writeto("%s/%s" %(output_dir,fname), output_verify="fix+warn", overwrite=True, checksum=True)
        #logging.info('Wrote %s' %(fname))
      # Sum all pixels...
      nHDUlist[expected_EXTNAMEs['FLUX']].data = np.sum(sumFLUX,axis=1)
      nHDUlist[expected_EXTNAMEs['ERRS']].data = np.sqrt(np.sum(sumERRS*sumERRS,axis=1))
      nHDUlist[expected_EXTNAMEs['QUAL']].data = np.sum(sumQUAL,axis=1)
      fname=bn_infile.replace('.fits','_sum_sum.fits').replace("MERGE3D","MERGE1D")
      for (i_hdu,HDU) in enumerate(HDUlist) :
        nHDUlist[i_hdu].header['HIERARCH ESO PRO CATG']=HDUlist[0].header['HIERARCH ESO PRO CATG'].replace("MERGE3D","MERGE1D")+'_SUM_SUM'
        del nHDUlist[i_hdu].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 3D PIXEL NAXIS1 I']
      #del nHDUlist[-1].header['HIERARCH ESO EXTRACT_2D_ROWS_IFU_PIXELS 3D PIXEL J']
      model_file="%s/%s" %(output_dir,fname)
      iu.HDUList(nHDUlist).writeto(model_file, output_verify="fix+warn", overwrite=True, checksum=True)
      #logging.info('Wrote %s' %(fname))

    return model_file
# ------------------------------------------------------------------------------------------
def recombine_idps( files=None, orig_files=None ) :
    o_files=[]
    ifu_files=[]
    TELLURIC_CORR_file=None
    for file in orig_files or {} :
        if "ERRORS" in file.category :
            #print(file)
            pass
        if "ORIG_" in file.category[0:5] :
            hdulist=fits.open(file.name)
            procatg=hdulist[0].header.get('HIERARCH ESO PRO CATG','NULL')
            if procatg in [
              'SCI_IFU_MERGE3D_DATA_OBJ_VIS',
              'SCI_IFU_MERGE3D_DATA_OBJ_NIR',
              'FLUX_IFU_MERGE3D_DATA_OBJ_VIS',
              'FLUX_IFU_MERGE3D_DATA_OBJ_NIR',
              'TELL_IFU_MERGE3D_DATA_OBJ_VIS',
              'TELL_IFU_MERGE3D_DATA_OBJ_NIR',
            ] :
              o_files+=[file,]
        elif file.category in [
            "SCIENCE_CUBE_SPECTRA",
            "STD_CUBE_SPECTRA",
        ] :
            ifu_files+=[file]
        elif file.category in [
            "TELLURIC_CORR",
        ] :
            TELLURIC_CORR_file=file
    for file in o_files :
        hdulist=fits.open(file.name)
        return recombine_idps_IFU(
            file,
            hdulist,
            files,
            ifu_files,
            TELLURIC_CORR_file,
        )
    return files
# ------------------------------------------------------------------------------------------
def recombine_idps_IFU(
        orig_file,
        hdulist,
        files=None,
        ifu_files=None,
        TELLURIC_CORR_file=None,
    ) :
    new_files=None
    propagate_keys_file=None
    TELLURIC_CORR={
        "SCIENCE_TELLURIC_CORR": fits.open(hdulist.filename()),
        # Don't bother with the SPECTRUM_TELLURIC_CORR, it seems to be an exact copy
        # of SCIENCE_TELLURIC_CORR...
        #"SPECTRUM_TELLURIC_CORR": copy.deepcopy(hdulist),
    }
    if "SCIENCE_TELLURIC_CORR" in TELLURIC_CORR.keys() :
        #TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].data*=0.

        # Put the SCIENCE_TELLURIC_CORR into the PrimaryHDU
        # and the SPECTRUM_TELLURIC_CORR into an added HDU on the end...
        # For convenience, copy the original MERGE3D_DATA_OBJ into another HDU
        #
        # First the MERGE3D_DATA_OBJ...
        # Insert at position 1 as XSHOOTER IDPs have FLUX, ERRS and QUAL extensions...
        insert_pos=1
        TELLURIC_CORR["SCIENCE_TELLURIC_CORR"].insert(insert_pos, fits.ImageHDU(data=copy.deepcopy(hdulist[0].data), header=hdulist[0].header))
        TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][insert_pos].header['EXTNAME']='FLUX_NO_TELLCORR'
        #TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][insert_pos].header['EXTNAME']=TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][insert_pos].header['ESO PRO CATG']
        #del TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][insert_pos].header['ESO PRO CATG']
        del TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][insert_pos].header['PIPEFILE']
        
        if "SPECTRUM_TELLURIC_CORR" in TELLURIC_CORR.keys() :
            # And prepare another HDU for the SPECTRUM_TELLURIC_CORR...
            TELLURIC_CORR["SCIENCE_TELLURIC_CORR"].append(fits.ImageHDU(data=copy.deepcopy(hdulist[0].data), header=hdulist[0].header))
            TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][-1].header['EXTNAME']="SPECTRUM_TELLURIC_CORR"
            del TELLURIC_CORR["SCIENCE_TELLURIC_CORR"]["SPECTRUM_TELLURIC_CORR"].header['ESO PRO CATG']
            del TELLURIC_CORR["SCIENCE_TELLURIC_CORR"]["SPECTRUM_TELLURIC_CORR"].header['PIPEFILE']
            TELLURIC_CORR["SCIENCE_TELLURIC_CORR"]["SPECTRUM_TELLURIC_CORR"].data*=0.
        # set PRO.CATG
        TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].header['HIERARCH ESO PRO CATG']="SCIENCE_TELLURIC_CORR"

    if "SPECTRUM_TELLURIC_CORR" in TELLURIC_CORR.keys() :
        TELLURIC_CORR["SPECTRUM_TELLURIC_CORR"][0].header['HIERARCH ESO PRO CATG']="SPECTRUM_TELLURIC_CORR"
        TELLURIC_CORR["SPECTRUM_TELLURIC_CORR"][0].data*=0.
    
    # Open the atm transmission file for later use...
    if TELLURIC_CORR_file is not None :
        # Actually not possible not to have TELLURIC_CORR_file I think...
        at_hdulist=fits.open(TELLURIC_CORR_file.name)
        # Don't use /= in order to get double precision...
        if TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].header['NAXIS'] == 3 :
          # Must be an IFU image...
          for ifu_col in range(TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].shape[2]) :
            TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].data[:,:,ifu_col]=(
                copy.deepcopy(TELLURIC_CORR["SCIENCE_TELLURIC_CORR"][0].data[:,:,ifu_col])
                /
                np.transpose(np.array([at_hdulist[1].data]))
            )
        for file in files or {} :
            if file.category in ["SCIENCE_TELLURIC_CORR",] :
                propagate_keys_file=propagate_keys_file or file
    else :
        ## JPr: 2021-06-10
        ## I've turned off using molecfit_correct to correct all the spectra,
        ## instead it will just correct the one (SCIENCE_CALCTRANS)
        ## We should NEVER arrive here...
        raise Exception("Could not find TELLURIC_CORR_file in input dataset.")

    output_dir=os.path.dirname(propagate_keys_file.name or files[0].name)
    bname=os.path.basename(orig_file.name)
    hdr=None
    if propagate_keys_file is not None :
        hdr=iu.propagate_keys(
            propagate_keys_file,
            ["molecfit_model","molecfit_calctrans","molecfit_correct"],
            orig_file
        )
    for k in TELLURIC_CORR.keys() :
        new_files=new_files or []
        out_file_name="%s/%s_%s" %(output_dir,k,bname)
        if hdr is not None :
            TELLURIC_CORR[k][0].header+=hdr
        print("writing '%s'..." %out_file_name)
        TELLURIC_CORR[k][0].header['PIPEFILE']="%s_%s" %(k,bname)
        iu.HDUList(TELLURIC_CORR[k]).writeto(
            out_file_name, overwrite=True, checksum=True, output_verify='silentfix',
        )
        print("wrote '%s'." %out_file_name)
        new_files+=[
            reflex.FitsFile(
                out_file_name,
                k,
                TELLURIC_CORR[k][0].header['CHECKSUM'],
                ['UNIVERSAL',]
            )
        ]

    if TELLURIC_CORR_file is not None :
        for cfile in ifu_files or [] :
            new_files=new_files or []
            k="SCIENCE_TELLURIC_CORR"
            bname=os.path.basename(cfile.name)
            c_hdulist=fits.open(cfile.name)
            c_hdulist[0].header+=hdr
            c_hdulist[0].header['ESO PRO CATG']="%s_CUBE" %k
            # The image cube of an ARGUS mode observation is in the Primary HDU
            # The <N> image cubes of an IFU mode observation are in <N> ImageHDUs
            # with the Primary HDU "empty".
            # Both type also have an attached FIBER_SETUP binary table ()
            # so we can't just process all HDUs...
            ## ARGUS
            list_of_c_hdus=[c_hdulist[0]]
            ## IFU
            if c_hdulist[0].header.get('ESO INS MODE','').strip() in ["IFU",] :
                list_of_c_hdus=c_hdulist[1:]

            for c_hdu in list_of_c_hdus :
                if isinstance(c_hdu, (fits.ImageHDU,fits.PrimaryHDU)) :
                    # Don't use /= in order to get double precision...
                    c_hdu.data=c_hdu.data/np.transpose(np.array([[at_hdulist[1].data]]))

            out_file_name="%s/%s_%s" %(output_dir,k,bname)
            print("writing '%s'..." %out_file_name)
            iu.HDUList(c_hdulist).writeto(
                out_file_name,
                overwrite=True,
                checksum=True,
                output_verify='silentfix',
            )
            print("Wrote '%s'..." %out_file_name)
            new_files+=[reflex.FitsFile(out_file_name, k, c_hdulist[0].header['CHECKSUM'], ['UNIVERSAL',])]
    return new_files or files
# ------------------------------------------------------------------------------------------
