from edps import QC0, CALCHECKER, IDP

from .xshooter_association_rules import *
from .xshooter_fit_orders import fit_orders
from .xshooter_flat_reduction import process_flats
from .xshooter_flat_strategy import flat_strategy
from .xshooter_linearity import xshoo_linearity
from .xshooter_response import *
from .xshooter_science import xshoo_science_slit, xshoo_science_ifu
from .xshooter_spectra_combination import spectra_combination
from .xshooter_telluric_correction import xshoo_telluric_correction, model_on_standard, model_on_science
from .xshooter_wavelength import wavelength_calibration

__title__ = "XSHOOTER workflow"

# --- Processing Tasks ----------------------------------------------------------------------------

# --- Detector linearity processing tasks
detmon_opt, detmon_nir = xshoo_linearity()

# --- Bias processing task
bias = (task("bias")
        .with_recipe("xsh_mbias")
        .with_report('xshooter_rawdisp', ReportInput.RECIPE_INPUTS)
        .with_report("xshooter_master_bias", ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_main_input(raw_bias)
        .with_alternative_associated_inputs(bad_pixel_map_rp_static_calibrations)
        .with_dynamic_parameter("arm", which_arm)
        .with_job_processing(set_bias_processing)
        .with_input_filter(bias_uvb_class, bias_vis_class, bp_map_rp_uvb_class, bp_map_rp_vis_class, bp_map_nl_uvb,
                           bp_map_nl_vis)
        .with_meta_targets([QC1_CALIB])
        .build())

# --- Dark processing task
dark = (task("dark")
        .with_recipe("xsh_mdark")
        .with_report('xshooter_rawdisp', ReportInput.RECIPE_INPUTS)
        .with_report("xshooter_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_main_input(raw_dark)
        .with_associated_input(bias, [master_bias_uvb], condition=is_UVB, min_ret=1,
                               match_rules=associate_bias_cals)
        .with_associated_input(bias, [master_bias_vis], condition=is_VIS, min_ret=1,
                               match_rules=associate_bias_cals)
        .with_associated_input(bias, [master_bias_agc], condition=is_AGC, min_ret=1,
                               match_rules=associate_bias_cals_agc)
        .with_alternative_associated_inputs(bad_pixel_map_rp_static_calibrations)
        .with_dynamic_parameter("arm", which_arm)
        .with_job_processing(set_dark_processing)
        .with_input_filter(master_bias_uvb, bp_map_nl_uvb, bp_map_rp_uvb_class,
                           master_bias_vis, bp_map_nl_vis, bp_map_rp_vis_class,
                           bp_map_nl_nir, bp_map_rp_nir_class, master_bias_agc, bp_map_rp_agc_class)
        .with_meta_targets([QC1_CALIB])
        .build())

# --- Tracing order tasks
predict, orderdef, orderdef_calibrations = fit_orders(alternatives_spectral_format, bias, bp_map_rp_nir)

# -- Flat processing tasks
(sky_flat, lamp_flat, lamp_flat_hc, flat_calibrations, flat_calibrations_for_arcs, flat_calibrations_for_flex,
 flat_calibrations_for_dmap) = process_flats(raw_skyflat_agc, raw_flat, raw_flat_nir_off, raw_flat_hc, raw_flat_qth_hc,
                                             raw_flat_nir_off_hc, bias, alternatives_spectral_format,
                                             orderdef_calibrations, dark, bp_map_rp_nir)

# Task to select the flat_calibrations to be used to process standard stars.
# The selection is done by setting the workflow parameter "use_flat" to the desired value
#   use_flat: "standard". Use the flats that are closest in time to the standard star observations
#   use_flat: "science". Use the same flats used to process science observations also for standard stars.
flat_calibrations_for_standard = flat_strategy(flat_calibrations, raw_science, raw_standard)

# --- Processing tasks for arc frames, 2D distortion, and wavelength calibration
wavelength_calibration_2d, alternatives_dmap, arc = wavelength_calibration(bias, flat_calibrations_for_dmap, predict,
                                                                           alternatives_spectral_format,
                                                                           flat_calibrations_for_arcs)

# --- Task for flexure correction ----
flex = (task('flexures')
        .with_recipe('xsh_flexcomp')
        .with_report('xshooter_rawdisp', ReportInput.RECIPE_INPUTS)
        .with_main_input(raw_flex)
        .with_dynamic_parameter('arm', which_arm)
        .with_dynamic_parameter('jh', which_jh)
        .with_dynamic_parameter('ifu', which_ifu)
        .with_associated_input(bp_map_rp_nir, min_ret=1, condition=is_NIR)
        .with_alternative_associated_inputs(flat_calibrations_for_flex)
        .with_associated_input(bias, [master_bias_uvb], condition=is_UVB, min_ret=1,
                               match_rules=associate_bias_flex)
        .with_associated_input(bias, [master_bias_vis], condition=is_VIS, min_ret=1,
                               match_rules=associate_bias_flex)
        .with_associated_input(wavelength_calibration_2d, [xsh_mod_cfg_opt_2d_uvb], condition=physical_mode_UVB)
        .with_associated_input(wavelength_calibration_2d, [wave_tab_2d_uvb_class], condition=polynomial_mode_UVB)
        .with_associated_input(wavelength_calibration_2d, [xsh_mod_cfg_opt_2d_vis], condition=physical_mode_VIS)
        .with_associated_input(wavelength_calibration_2d, [wave_tab_2d_vis_class], condition=polynomial_mode_VIS)
        .with_associated_input(wavelength_calibration_2d, [xsh_mod_cfg_opt_2d_nir], condition=physical_mode_NIR)
        .with_associated_input(wavelength_calibration_2d, [wave_tab_2d_nir_class], condition=polynomial_mode_NIR)
        .with_associated_input(arc_line_list_afc)
        .with_alternative_associated_inputs(alternatives_spectral_format)
        .with_job_processing(set_flexures_parameters)
        .with_input_filter(bp_map_nl_uvb, master_bias_uvb, order_tab_edges_slit_uvb,
                           xsh_mod_cfg_opt_2d_uvb, bp_map_rp_uvb_class, arc_line_list_uvb_class,
                           arc_line_list_afc_uvb_class, wave_tab_2d_uvb_class,
                           spectral_format_tab_uvb_class, bp_map_nl_vis, master_bias_vis, order_tab_edges_slit_vis,
                           xsh_mod_cfg_opt_2d_vis, bp_map_rp_vis_class, arc_line_list_vis_class,
                           arc_line_list_afc_vis_class, wave_tab_2d_vis_class,
                           spectral_format_tab_vis_class,
                           bp_map_nl_nir, order_tab_edges_slit_nir,
                           xsh_mod_cfg_opt_2d_nir, bp_map_rp_nir_class, arc_line_list_nir_class,
                           arc_line_list_afc_nir_class, wave_tab_2d_nir_class,
                           spectral_format_tab_nir_jh_class, spectral_format_tab_nir_class, order_tab_edges_ifu_uvb,
                           order_tab_edges_ifu_vis, order_tab_edges_ifu_nir)
        .with_meta_targets([QC1_CALIB, CALCHECKER])
        .build())

# - Calibrations to correct for flexures the standard stars in offset and stare modes
flexure_calibrations_for_standard = (alternative_associated_inputs()
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_slit_uvb,
                                                                   disp_tab_afc_uvb], condition=is_UVB_slit,
                                                            match_rules=associate_flex_using_target_name, min_ret=0)
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_slit_vis,
                                                                   disp_tab_afc_vis], condition=is_VIS_slit,
                                                            match_rules=associate_flex_using_target_name, min_ret=0)
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_slit_nir,
                                                                   disp_tab_afc_nir], condition=is_NIR_slit,
                                                            match_rules=associate_flex_using_target_name, min_ret=0)
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_ifu_uvb,
                                                                   disp_tab_afc_uvb], condition=is_UVB_ifu,
                                                            match_rules=associate_flex_using_target_name, min_ret=0)
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_ifu_vis,
                                                                   disp_tab_afc_vis], condition=is_VIS_ifu,
                                                            match_rules=associate_flex_using_target_name, min_ret=0)
                                     .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_ifu_nir,
                                                                   disp_tab_afc_nir], condition=is_NIR_ifu,
                                                            match_rules=associate_flex_using_target_name, min_ret=0))
# - Calibrations to correct for flexures the standard stars in nodding mode
flexure_calibrations_for_standard_nod = (alternative_associated_inputs()
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_slit_uvb,
                                                                       disp_tab_afc_uvb], condition=is_UVB_slit,
                                                                match_rules=associate_flex_using_target_name)
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_slit_vis,
                                                                       disp_tab_afc_vis], condition=is_VIS_slit,
                                                                match_rules=associate_flex_using_target_name)
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_slit_nir,
                                                                       disp_tab_afc_nir], condition=is_NIR_slit,
                                                                match_rules=associate_flex_using_target_name)
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_ifu_uvb,
                                                                       disp_tab_afc_uvb], condition=is_UVB_ifu,
                                                                match_rules=associate_flex_using_target_name)
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_ifu_vis,
                                                                       disp_tab_afc_vis], condition=is_VIS_ifu,
                                                                match_rules=associate_flex_using_target_name)
                                         .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_ifu_nir,
                                                                       disp_tab_afc_nir], condition=is_NIR_ifu,
                                                                match_rules=associate_flex_using_target_name))

# - Calibrations to correct for flexures the science observations
flexure_calibrations_for_science = (alternative_associated_inputs()
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_slit_uvb,
                                                                  disp_tab_afc_uvb], condition=is_UVB_slit,
                                                           match_rules=associate_flex_using_container)
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_slit_vis,
                                                                  disp_tab_afc_vis], condition=is_VIS_slit,
                                                           match_rules=associate_flex_using_container)
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_slit_nir,
                                                                  disp_tab_afc_nir], condition=is_NIR_slit,
                                                           match_rules=associate_flex_using_container)
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_uvb, order_tab_afc_ifu_uvb,
                                                                  disp_tab_afc_uvb], condition=is_UVB_ifu,
                                                           match_rules=associate_flex_using_container)
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_vis, order_tab_afc_ifu_vis,
                                                                  disp_tab_afc_vis], condition=is_VIS_ifu,
                                                           match_rules=associate_flex_using_container)
                                    .with_associated_input(flex, [xsh_mod_cfg_opt_afc_nir, order_tab_afc_ifu_nir,
                                                                  disp_tab_afc_nir], condition=is_NIR_ifu,
                                                           match_rules=associate_flex_using_container))
# ---

# -- Processing tasks for the computation of the response curve
# The selection of the type of response curve calibration is done by setting the workflow parameter "response"
# Note: the (night/master) response calibration closer in time is selected, regardless of the observing mode.
# response = night. The response curve obtained from the standard star observed that night is used.
#                   If not available, then the master response is used.
# response = master. The master response curve downloaded with CalSelector is used. It is constructed by combining
#                    the information of flux standards from several nights.
#
response_slit_offset, response_slit_stare, response_slit_nod, response_calib = xshoo_response(bias,
                                                                                              flat_calibrations_for_standard,
                                                                                              alternatives_dmap,
                                                                                              flexure_calibrations_for_standard,
                                                                                              flexure_calibrations_for_standard_nod,
                                                                                              alternatives_spectral_format,
                                                                                              bad_pixel_map_rp_static_calibrations)

# ---- Subworkflows to reduce science exposures ---------------------------------------------------

# - Object exposures
science_slit_stare, science_slit_offset, science_slit_nod = xshoo_science_slit(raw_science_slit_stare,
                                                                               raw_science_slit_offset,
                                                                               raw_science_slit_nod,
                                                                               alternatives_spectral_format,
                                                                               bias, dark, flat_calibrations,
                                                                               alternatives_dmap,
                                                                               bad_pixel_map_rp_static_calibrations,
                                                                               sky_line_list, response_calib, atmos_ext,
                                                                               tell_mask,
                                                                               flexure_calibrations_for_science,
                                                                               input_type="science",
                                                                               additional_metatargets=[IDP, QC0,
                                                                                                       CALCHECKER])

# Telluric standards are usually used for telluric correction only. These tasks process them like a
# normal science observation. This is only triggered when these tasks are targets or when the metatarget
# "telluric_standard" , "qc0" or "idp" are specified.
telluric_slit_stare, telluric_slit_offset, telluric_slit_nod = xshoo_science_slit(raw_telluric_slit_stare,
                                                                                  raw_telluric_slit_offset,
                                                                                  raw_telluric_slit_nod,
                                                                                  alternatives_spectral_format,
                                                                                  bias, dark, flat_calibrations,
                                                                                  alternatives_dmap,
                                                                                  bad_pixel_map_rp_static_calibrations,
                                                                                  sky_line_list, response_calib,
                                                                                  atmos_ext, tell_mask,
                                                                                  flexure_calibrations_for_science,
                                                                                  input_type="telluric_standard",
                                                                                  additional_metatargets=[IDP,
                                                                                                          CALCHECKER,
                                                                                                          QC1_CALIB],
                                                                                  report="xshooter_telluric_std")

# Flux standards are usually used for response computation only. These tasks process them like a
# normal science observation. This is only triggered when these tasks are targets or when the metatarget
# "flux_standard" or "idp" are specified.
# Note: Flux standard are processed by the response recipe through the regular qc1_calibration, therefore
# they do not need to be processed also in the QC0 process.
flux_standard_slit_stare, flux_standard_slit_offset, flux_standard_slit_nod = xshoo_science_slit(
    raw_standard_slit_stare, raw_standard_slit_offset,
    raw_standard_slit_nod,
    alternatives_spectral_format,
    bias, dark, flat_calibrations, alternatives_dmap,
    bad_pixel_map_rp_static_calibrations,
    sky_line_list, response_calib, atmos_ext,
    tell_mask, flexure_calibrations_for_science,
    input_type="flux_standard",
    additional_metatargets=[IDP, CALCHECKER])

# - Processing tasks for IFU observations
science_ifu_stare, science_ifu_offset = xshoo_science_ifu(alternatives_spectral_format, bias, dark, flat_calibrations,
                                                          wavelength_calibration_2d,
                                                          bad_pixel_map_rp_static_calibrations,
                                                          flexure_calibrations_for_science)

# --- Tasks for Telluric correction ----------------------------------------------------------------

# The computation of the atmospheric parameters can be done on a telluric standard, or on the science frame itself.
# The choice is selected by the workflow parameter "telluric_correction". Possible values are:
# "standard": a telluric standard within the same night matching the instrument setup of the file to correct is used.
#             If not found, the correction is not performed. The preferential order for telluric standard
#             is nodding, stare, offset.
# "science":  the very same file to correct is used.
# "none":     no telluric correction

# - Computation of atmospheric parameters using telluric standards
atmospheric_model_nod = model_on_standard(telluric_slit_nod, input_mode="nod")
atmospheric_model_stare = model_on_standard(telluric_slit_stare, input_mode="stare")
atmospheric_model_offset = model_on_standard(telluric_slit_offset, input_mode="offset")

# - Computation of atmospheric parameters using directly the file to correct.
#   In these tasks, the file to correct are the scientific targets.
atmospheric_model_on_science_nod = model_on_science(science_slit_nod, input_type="science", input_mode="nod")
atmospheric_model_on_science_offset = model_on_science(science_slit_offset, input_type="science", input_mode="offset")
atmospheric_model_on_science_stare = model_on_science(science_slit_stare, input_type="science", input_mode="stare")

# - Computation of atmospheric parameters using directly the file to correct.
#   In these tasks, the file to correct are the flux standards if treated as regular science observations.
atmospheric_model_on_science_star_nod = model_on_science(flux_standard_slit_nod, input_type="flux_standard",
                                                         input_mode="nod")
atmospheric_model_on_science_star_offset = model_on_science(flux_standard_slit_offset, input_type="flux_standard",
                                                            input_mode="offset")
atmospheric_model_on_science_star_stare = model_on_science(flux_standard_slit_stare, input_type="flux_standard",
                                                           input_mode="stare")

# - Computation of atmospheric parameters using directly the file to correct.
#   In these tasks, the file to correct are the telluric standards if treated as regular science observations.
atmospheric_model_on_science_telluric_nod = model_on_science(telluric_slit_nod, input_type="telluric_standard",
                                                             input_mode="nod")
atmospheric_model_on_science_telluric_offset = model_on_science(telluric_slit_offset, input_type="telluric_standard",
                                                                input_mode="offset")
atmospheric_model_on_science_telluric_stare = model_on_science(telluric_slit_stare, input_type="telluric_standard",
                                                               input_mode="stare")

# ---- Computation and application of atmospheric transmission function. ----------------------------
#   Note: The atmospheric transmission will be computed using:
#      - the atmospheric parameters fitted in the reference spectrum (either telluric standard or
#         science frame to correct), and
#      - the scientific exposure that needs correction.


# - Correction of scientific targets
science_slit_stare_tc = xshoo_telluric_correction(science_slit_stare, atmospheric_model_nod, atmospheric_model_stare,
                                                  atmospheric_model_offset, atmospheric_model_on_science_stare,
                                                  input_type="science", input_mode="stare")

science_slit_offset_tc = xshoo_telluric_correction(science_slit_offset, atmospheric_model_nod, atmospheric_model_stare,
                                                   atmospheric_model_offset, atmospheric_model_on_science_offset,
                                                   input_type="science", input_mode="offset")

science_slit_nod_tc = xshoo_telluric_correction(science_slit_nod, atmospheric_model_nod, atmospheric_model_stare,
                                                atmospheric_model_offset, atmospheric_model_on_science_nod,
                                                input_type="science", input_mode='nod')

# - Correction of flux standards if treated as scientific targets.
flux_standard_slit_stare_tc = xshoo_telluric_correction(flux_standard_slit_stare, atmospheric_model_nod,
                                                        atmospheric_model_stare,
                                                        atmospheric_model_offset,
                                                        atmospheric_model_on_science_star_stare,
                                                        input_type="flux_standard", input_mode="stare")

flux_standard_slit_offset_tc = xshoo_telluric_correction(flux_standard_slit_offset, atmospheric_model_nod,
                                                         atmospheric_model_stare,
                                                         atmospheric_model_offset,
                                                         atmospheric_model_on_science_star_offset,
                                                         input_type="flux_standard", input_mode="offset")

flux_standard_slit_nod_tc = xshoo_telluric_correction(flux_standard_slit_nod, atmospheric_model_nod,
                                                      atmospheric_model_stare,
                                                      atmospheric_model_offset, atmospheric_model_on_science_star_nod,
                                                      input_type="flux_standard", input_mode="nod")

# - Correction of telluric standards if treated as scientific targets.
telluric_slit_stare_tc = xshoo_telluric_correction(telluric_slit_stare, atmospheric_model_nod, atmospheric_model_stare,
                                                   atmospheric_model_offset,
                                                   atmospheric_model_on_science_telluric_stare,
                                                   input_type="telluric_standard", input_mode="stare")

telluric_slit_offset_tc = xshoo_telluric_correction(telluric_slit_offset, atmospheric_model_nod,
                                                    atmospheric_model_stare, atmospheric_model_offset,
                                                    atmospheric_model_on_science_telluric_offset,
                                                    input_type="telluric_standard", input_mode="offset")
telluric_slit_nod_tc = xshoo_telluric_correction(telluric_slit_nod, atmospheric_model_nod, atmospheric_model_stare,
                                                 atmospheric_model_offset, atmospheric_model_on_science_telluric_offset,
                                                 input_type="telluric_standard", input_mode="nod")

# --- Tasks for combination of reduced spectra across Observing Blocks -----------------------------

# - Combination of science spectra. Only telluric corrected or not telluric corrected spectra
#   are combined, depending on whether the telluric correction was active or not (workflow
#   parameter "telluric_correction_mode" != none

# Combination of science exposures
science_combined, collect_science = spectra_combination(raw_science_to_combine,
                                                        science_slit_stare, science_slit_offset, science_slit_nod,
                                                        science_slit_stare_tc, science_slit_offset_tc,
                                                        science_slit_nod_tc,
                                                        input_type="science")

# Combination of flux standards if treated as science exposures
flux_standard_combined, collect_std = spectra_combination(raw_standard_to_combine,
                                                          flux_standard_slit_stare, flux_standard_slit_offset,
                                                          flux_standard_slit_nod,
                                                          flux_standard_slit_stare_tc, flux_standard_slit_offset_tc,
                                                          flux_standard_slit_nod_tc,
                                                          input_type="flux_standard")

# Combination of telluric standards if treated as science exposures
telluric_standard_combined, collect_tell = spectra_combination(raw_telluric_to_combine,
                                                               telluric_slit_stare, telluric_slit_offset,
                                                               telluric_slit_nod,
                                                               telluric_slit_stare_tc, telluric_slit_offset_tc,
                                                               telluric_slit_nod_tc,
                                                               input_type="telluric_standard")
