/*
 * This file is part of the Molecfit Pipeline
 * Copyright (C) 2001-2019 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef MF_WRAP_CONFIG_H
#define MF_WRAP_CONFIG_H

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

//#include <telluriccorr.h>
#include "mf_model.h"

#include "mf_wrap_fits.h"
#include "mf_wrap_data.h"
#include "mf_wrap_dfs.h"
#include "mf_strlst.h"
///#include <mf_wrap_utils.h>

//CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/**
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define MOLECFIT_MODEL       "molecfit_model"

/* COLUMN MAPPING PARAMETERS */
#define MOLECFIT_MAPPING_KERNEL_EXT                        "KERNEL_LIBRARY_EXT"
#define MOLECFIT_MAPPING_ATMOSPHERIC_EXT                   "ATM_PARAMETERS_EXT"
#define MOLECFIT_MAPPING_CONVOLVE_EXT                      "LBLRTM_RESULTS_EXT"
#define MOLECFIT_MAPPING_CORRECT_EXT                       "TELLURIC_CORR_EXT"


/* RECIPE PARAMETERS */
#define MOLECIFT_TELLURICCORR_PARAMETER_DEFAULT            "TELLURICCORR_PARAMETER_DEFAULT"


#define MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI              "USE_ONLY_INPUT_PRIMARY_DATA"
#define MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI_DESC         "Value=TRUE implies that only the fits primary contains the input science flux data.\n"\
                                                           " Value=FALSE implies that the fits extensions also contains input science flux data."
#define MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI_INIT         CPL_FALSE

#define MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA            "USE_DATA_EXTENSION_AS_DFLUX"
#define MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA_DESC       "Only valid if USE_ONLY_INPUT_PRIMARY_DATA=TRUE. The fits extension index that contains the\n"\
                                                           " errors of the science flux data (DFLUX). A value of 0 implies that there is no DFLUX"
#define MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA_INIT       0

#define MOLECFIT_PARAMETER_MASK_EXTENSION_DATA             "USE_DATA_EXTENSION_AS_MASK"
#define MOLECFIT_PARAMETER_MASK_EXTENSION_DATA_DESC        "Only valid if USE_ONLY_INPUT_PRIMARY_DATA=TRUE. The fits extension index that contains the\n"\
                                                           " mask associated with the science flux data. A value of 0 implies that there is no mask data."
#define MOLECFIT_PARAMETER_MASK_EXTENSION_DATA_INIT        0

#define MOLECFIT_PARAMETER_USE_INPUT_KERNEL                "USE_INPUT_KERNEL"
#define MOLECFIT_PARAMETER_USE_INPUT_KERNEL_DESC           "If TRUE, use the kernel library if it is provided."
#define MOLECFIT_PARAMETER_USE_INPUT_KERNEL_INIT           CPL_TRUE

#define MOLECFIT_PARAMETER_SUPPRESS_EXTENSION              "SUPPRESS_EXTENSION"
#define MOLECFIT_PARAMETER_SUPPRESS_EXTENSION_DESC         "Suppress arbitrary filename extension : TRUE (apply) or FALSE (don't apply)"
#define MOLECFIT_PARAMETER_SUPPRESS_EXTENSION_INIT         CPL_FALSE

#define MOLECFIT_PARAMETER_CHIP_EXTENSIONS                 "CHIP_EXTENSIONS"
#define MOLECFIT_PARAMETER_CHIP_EXTENSIONS_DESC            "Flag that determines if image extensions are to be treated as independent\n"\
                                                           " science data to be fitted for independently or as CHIP specific subranges \n"\
                                                           " of a single observation to be fitted for as a single combined spectrum.\n"\
                                                           " Value = TRUE implies to treat as CHIPS to be combined. Value = FALSE implies\n"\
                                                           " to treat as independent. [FALSE]\n"
#define MOLECFIT_PARAMETER_CHIP_EXTENSIONS_INIT            CPL_FALSE


/* Molecules table */
#define MOLECFIT_PARAMETER_LIST                            MF_COL_LIST_MOLECULES
#define MOLECFIT_PARAMETER_LIST_DESC                       "List of molecules to be included in the model. Represented as a comma separated\n"\
                                                           " string of molecule names, e.g. \"H2O,CO2,O3\".\n"\
                                                           " If set to NULL, the input TAG[MOLECULES] FITS BINTABLE values have to be provided\n"\
                                                           " where the FITS BINTABLE specified contains the three columns:\n"\
                                                           " LIST_MOLEC; FIT_MOLEC; and REL_COL."
#define MOLECFIT_PARAMETER_LIST_INIT                       MF_PARAMETERS_NULL

#define MOLECFIT_PARAMETER_FIT                             MF_COL_FIT_MOLECULES
#define MOLECFIT_PARAMETER_FIT_DESC                        "List of flags that specify which of the listed molecules are to be fitted for.\n"\
                                                           " Flag=1 implies yes. Flag=0 implies no. Represented as a string of comma separated\n"\
                                                           " integers in the same order as the listed molecules. For example: if LIST_MOLEC=\"H2O,CO2,O3\", then \n"\
                                                           " FIT_MOLEC=\"1,0,1\" implies that only H2O and O3 should be fitted for.\n"\
                                                           " If set to NULL, the input TAG[MOLECULES] FITS BINTABLE values have to be provided where the FITS \n"\
                                                           " BINTABLE specified contains the three columns: LIST_MOLEC; FIT_MOLEC; and REL_COL."
#define MOLECFIT_PARAMETER_FIT_INIT                        MF_PARAMETERS_NULL

#define MOLECFIT_PARAMETER_RELATIVE_VALUE                  MF_COL_REL_COL
#define MOLECFIT_PARAMETER_RELATIVE_VALUE_DESC             "List of the intial values of fitting of the molecular columns expressed relatively to the input\n"\
                                                           "  ATM profile columns. Represented as a comma separated list of doubles in the same order as the \n"\
                                                           " listed molecules. For example, if LIST_MOLEC=\"H2O,CO2,O3\", then REL_COL=\"1.0,1.2,0.8\"\n"\
                                                           " implies that H2O, CO2 and O3 have initial relative values of 1.0, 1.2 and 0.8 respectively.\n"\
                                                           " If set to NULL, the input TAG[MOLECULES] FITS BINTABLE values have to be provided where the FITS \n"\
                                                           " BINTABLE specified contains the three columns: LIST_MOLEC; FIT_MOLEC; and REL_COL."
#define MOLECFIT_PARAMETER_RELATIVE_VALUE_INIT             MF_PARAMETERS_NULL


/* Range tables */
#define MOLECFIT_PARAMETER_WAVE_RANGE_INCLUDE              MOLECFIT_WAVE_INCLUDE
#define MOLECFIT_PARAMETER_WAVE_RANGE_INCLUDE_DESC         "Wavelength ranges to be included. Represented as a string of comma separated doubles in pairs \n"\
                                                           " specifying the start and end wavelengths of a range. The wavelength units are always in microns.\n"\
                                                           " For example a KMOS sample data in the range of 1.11um to 1.67um may have\n"\
                                                           " WAVE_INCLUDE=\"1.773,1.78633,1.79098,1.80434,1.187691,1.189937\" to represent three inclusion regions:\n"\
                                                           " [1.773,1.78633], [1.79098,1.80434] and [1.187691,1.189937].\n"\
                                                           " If set to NULL, molecfit will check if the TAG[WAVE_INCLUDE] FITS BINTABLE values is provided where\n"\
                                                           " the FITS BINTABLE specified has the two columns: LOWER_LIMIT; and UPPER_LIMIT."
#define MOLECFIT_PARAMETER_WAVE_RANGE_INCLUDE_INIT         MF_PARAMETERS_NULL

#define MOLECFIT_PARAMETER_WAVE_RANGE_EXCLUDE              MOLECFIT_WAVE_EXCLUDE
#define MOLECFIT_PARAMETER_WAVE_RANGE_EXCLUDE_DESC         "Wavelength ranges to be excluded. Represented as a string of comma separated doubles in pairs \n"\
                                                           " specifying the start and end wavelengths of a range. The wavelength units are always in microns.\n"\
                                                           " as the input science data. For example a KMOS sample data in the range of 1.11um to 1.67um may have\n"\
                                                           " WAVE_EXCLUDE=\"1.773,1.78633,1.79098,1.80434,1.187691,1.189937\" to represent three exclusion regions:\n"\
                                                           " [1.773,1.78633], [1.79098,1.80434] and [1.187691,1.189937].\n"\
                                                           " If set to NULL, molecfit will check if the TAG[WAVE_EXCLUDE] FITS BINTABLE values is provided where\n"\
                                                           " the FITS BINTABLE specified has the two columns: LOWER_LIMIT; and UPPER_LIMIT."
#define MOLECFIT_PARAMETER_WAVE_RANGE_EXCLUDE_INIT         MF_PARAMETERS_NULL

#define MOLECFIT_PARAMETER_PIXEL_RANGE_EXCLUDE             MOLECFIT_PIXEL_EXCLUDE
#define MOLECFIT_PARAMETER_PIXEL_RANGE_EXCLUDE_DESC        "Pixel ranges to be excluded. Represented as a string of comma separated integers in pairs specifying the\n"\
                                                           " start and end pixel of a range. For example: PIXEL_EXCLUDE=\"54,128,512,514,1020,1024\" represents three\n"\
                                                           " exclusion regions: [54,128], [512,514] and [1020,1024].\n"\
                                                           " If set to NULL, molecfit will check if the TAG[PIXEL_EXCLUDE] FITS BINTABLE values is provided where the\n"\
                                                           " FITS BINTABLE specified has the two columns: LOWER_LIMIT; and UPPER_LIMIT."
#define MOLECFIT_PARAMETER_PIXEL_RANGE_EXCLUDE_INIT        MF_PARAMETERS_NULL


/*** MAPPING ***/
#define MOLECFIT_MAPPING_DEFAULT_PRIMARY                   MF_PARAMETERS_NULL


/* Mapping MODEL STD_MODEL/SCIENCE - KERNEL_EXT table */
#define MOLECFIT_PARAMETER_MODEL_MAPPING_KERNEL            MOLECFIT_MODEL_MAPPING_KERNEL
#define MOLECFIT_PARAMETER_MODEL_MAPPING_KERNEL_DESC       "Mapping '"MOLECFIT_STD_MODEL"/"MOLECFIT_SCIENCE"' - '"MOLECFIT_MODEL_KERNEL_LIBRARY"' [string with ext_number comma separated (int)] :\n"\
                                                           " If set to NULL, check if the TAG["MOLECFIT_MODEL_MAPPING_KERNEL"] FITS BINTABLE values is provided.\n"\
                                                           " The FITS BINTABLE have to one column ["MOLECFIT_MAPPING_KERNEL_EXT"]"
#define MOLECFIT_PARAMETER_MODEL_MAPPING_KERNEL_INIT       MOLECFIT_MAPPING_DEFAULT_PRIMARY

/* Mapping CALCTRANS SCIENCE - KERNEL_EXT table */
#define MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL        MOLECFIT_CALCTRANS_MAPPING_KERNEL
#define MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL_DESC   "Mapping '"MOLECFIT_SCIENCE"' - '"MOLECFIT_CALCTRANS_KERNEL_LIBRARY"' [string with ext_number comma separated (int)] :\n"\
                                                           " If set to NULL, check if the TAG["MOLECFIT_CALCTRANS_MAPPING_KERNEL"] FITS BINTABLE values is provided.\n"\
                                                           " The FITS BINTABLE have to one column ["MOLECFIT_MAPPING_KERNEL_EXT"]"
#define MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL_INIT   MOLECFIT_MAPPING_DEFAULT_PRIMARY


/* Mapping CALCTRANS SCIENCE_EXT - ATM_PARAMETERS_EXT input cpl_table */
#define MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC             MOLECFIT_MAPPING_ATMOSPHERIC
#define MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC_DESC        "Mapping '"MOLECFIT_SCIENCE"' - '"MOLECFIT_ATM_PARAMETERS"' input [string with ext_number comma separated (int)] :\n"\
                                                           " If set to NULL, check if the TAG["MOLECFIT_MAPPING_ATMOSPHERIC"] FITS BINTABLE value is provided.\n"\
                                                           " The FITS BINTABLE have to one column ["MOLECFIT_MAPPING_ATMOSPHERIC_EXT"]"
#define MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC_INIT        MOLECFIT_MAPPING_DEFAULT_PRIMARY


/* Mapping LBLRTM_RESULTS_EXT - TELLURIC_CORR output cpl_table */
#define MOLECFIT_PARAMETER_MAPPING_CONVOLVE                MOLECFIT_MAPPING_CONVOLVE
#define MOLECFIT_PARAMETER_MAPPING_CONVOLVE_DESC           "Mapping '"MOLECFIT_LBLRTM_RESULTS"' - '"MOLECFIT_TELLURIC_CORR"' output [string with ext_number comma separated (int)] :\n"\
                                                           " If set to NULL, check if the TAG["MOLECFIT_MAPPING_CONVOLVE"] FITS BINTABLE value is provided.\n"\
                                                           " The FITS BINTABLE have to one column ["MOLECFIT_MAPPING_CONVOLVE_EXT"]"
#define MOLECFIT_PARAMETER_MAPPING_CONVOLVE_INIT           MOLECFIT_MAPPING_DEFAULT_PRIMARY


/* Mapping CORRECT SCIENCE_EXT - TELLURIC_CORR_EXT table */
#define MOLECFIT_PARAMETER_MAPPING_CORRECT                 MOLECFIT_MAPPING_CORRECT
#define MOLECFIT_PARAMETER_MAPPING_CORRECT_DESC            "Mapping '"MOLECFIT_SCIENCE"' - '"MOLECFIT_TELLURIC_CORR"' [string with ext_number comma separated (int)] :\n"\
                                                           " If set to NULL, check if the TAG["MOLECFIT_MAPPING_CORRECT"] FITS BINTABLE value is provided.\n"\
                                                           " The FITS BINTABLE have to one column ["MOLECFIT_MAPPING_CORRECT_EXT"]"
#define MOLECFIT_PARAMETER_MAPPING_CORRECT_INIT            MOLECFIT_MAPPING_DEFAULT_PRIMARY

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structs and enum types
 */
/*----------------------------------------------------------------------------*/

typedef struct {

  cpl_boolean                      use_only_input_pri_ext;       /* If the user want to use only the STD_MODEL/SCIENCE_CALCTRANS/SCIENCE input FITS primary extension                          */
  int                              dflux_extension_data;         /* If use_only_input_pri_ext == CPL_TRUE, you can provide a extension as DFLUX (error flux)                                   */
  int                              mask_extension_data;          /* If use_only_input_pri_ext == CPL_TRUE, you can provide a extension as MASK                                                 */

  cpl_boolean                      use_input_kernel;             /* If the user provided, use or not the user kernel                                                                           */

  const char                       *list_molec;                  /* list_molec,           i.e. {H2O,CO,CO2,CH4,O2} string                                                                      */
  const char                       *fit_molec;                   /* fit_molec,            i.e. {1,  0, 1,  1,  0 } flag                                                                        */
  const char                       *rel_col;                     /* relative density,     i.e. {1.,1.,1.06,1.,1. } double                                                                      */
  cpl_table                        *molecules_table;             /* Molecules to fit. Contains 3 columns [MF_COL_LIST_MOLECULES, MF_COL_FIT_MOLECULES, MF_COL_REL_COL]                         */

  const char                       *wave_range_include;          /* wave_ranges_include,  i.e. {ini1,end1,...,iniN,endN} double                                                                */
  cpl_table                        *wave_ranges_include_table;   /* Wavelength ranges include to fit. Contains 2 columns [MF_COL_WAVE_RANGE_LOWER, MF_COL_WAVE_RANGE_UPPER]                    */

  const char                       *wave_range_exclude;          /* wave_ranges_exclude,  i.e. {ini1,end1,...,iniN,endN} double                                                                */
  cpl_table                        *wave_ranges_exclude_table;   /* Wavelength ranges exclude to fit. Contains 2 columns [MF_COL_WAVE_RANGE_LOWER, MF_COL_WAVE_RANGE_UPPER]                    */

  const char                       *pixel_range_exclude;         /* pixel_ranges_exclude, i.e. {ini1,end1,...,iniN,endN} int                                                                   */
  cpl_table                        *pixel_ranges_exclude_table;  /* Pixel ranges exclude      to fit. Contains 2 columns [MF_COL_WAVE_RANGE_LOWER, MF_COL_WAVE_RANGE_UPPER]                    */

  const char                       *mapping_kernel;              /* Mapping extensions : STD_MODEL/SCIENCE-MODEL_KERNEL_LIBRARY; i.e. {Data_ext1-Kernel_lib_ext, ..., Data_extN-Kernel_lib_ext} */
  cpl_table                        *mapping_kernel_table;        /* Mapping kernel extensions cpl_table. Contains 1 columns [KERNEL_LIBRARY_EXT]                                               */

  mf_configuration                 *mf_config;                   /* Molecfit configuration parameter                                                                                           */

  cpl_boolean                      chip_extensions;              /* If the user want to combine the input FITS extensions                                                                      */

  cpl_propertylist                 *pl;                          /* All input parameters to save in the output files                                                                           */

} mf_wrap_model_parameter;



/* Enumeration for the different types of Molecfit string list parameter types */
typedef enum {
  MF_SLIST,                  /* String list of strings                */
  MF_DLIST,                  /* String list of doubles                */
  MF_BLIST,                  /* String list of Booleans               */
  MF_NLIST,                  /* String list of positive integers > 0  */
  MF_N0LIST,                 /* String list of positive integers >=0  */
  MF_DRANGE,                 /* Range of doubles                      */
  MF_NRANGE,                 /* Range of positive integers > 0        */
  MF_N0RANGE,                 /* Range of positive integers >=0        */
  MF_MF_BLIST_OLD               /* B LIST that could be a single value "FALSE" or "TRUE" */
} mf_wrap_strlst_type;

/*enum MF_WRAP_STRLST_TYPE {
                     MF_WRAP_STRLST_NULL,
                     MF_WRAP_STRLST_STRING,
                     MF_WRAP_STRLST_DOUBLE,
                     MF_WRAP_STRLST_MF_DRANGE,
                     MF_WRAP_STRLST_INT,
                     MF_WRAP_STRLST_BOOL
};*/


/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

//cpl_error_code mf_wrap_config_model_input_filenames(const char* suffix);
cpl_error_code mf_wrap_config_model_output_filenames(cpl_frameset *frameset, cpl_parameterlist* ilist, const char* suffix);

cpl_error_code mf_wrap_preconfig_check(cpl_frameset *frameset,
		const cpl_parameterlist  *parlist,
		cpl_boolean *science_data_is_table);

//mf_wrap_model_parameter * mf_wrap_model_parameters(
//    const cpl_parameterlist            *list,
//    const cpl_propertylist             *raw_head_pri);



mf_wrap_model_parameter * wrap_config_update(cpl_frameset *frameset,
		const cpl_parameterlist  *parlist);

mf_wrap_model_parameter * mf_wrap_config_model_init(const cpl_parameterlist  *parlist,const cpl_propertylist* data_primary_header);
					//cpl_frameset *frameset --> was this needed?

//mf_wrap_model_parameter * init_wrap_check_tags(); //FLESH OUT!

cpl_error_code mf_wrap_merge_parameterlists( cpl_parameterlist* ilist,
		const cpl_parameterlist* parlist,
        cpl_parameterlist* result);

cpl_error_code init_wrap_merge_with_header( mf_parameters_config  *config,
		const cpl_propertylist   *header);

cpl_boolean  mf_wrap_config_chk_science_is_table(cpl_frame *input_frame);

/* Checks that molecule string lists are self compatible */
cpl_error_code mf_wrap_config_molecule_strs_chk(const char* str1,const char* str4,const char* str3);

/* Check that input string parameter is a valid list*/
cpl_error_code  mf_wrap_config_strlstchk( const cpl_parameterlist* parlist,
    const char*              parname,
    mf_wrap_strlst_type strlst_type);


/* Checks that mappings are valid*/
cpl_error_code  mf_wrap_config_mappingchk( const cpl_frameset* frameset,
   const cpl_parameterlist* parlist, const char*  library, const char*  lib_map,
   const char*  ext_column, const char*  parname, const char*  target_tag);

/* Check that keyword exists*/
cpl_error_code mf_wrap_config_check_keyword(
    const mf_wrap_fits* fits_data,
    const cpl_parameterlist* parlist,
    const char* keyword_par);

cpl_error_code mf_wrap_config_chk_column(
    const mf_wrap_fits* fits_data,
    const char* col_name);

cpl_error_code mf_config_chk_ref_atmos(
    const char* dirname,
    const char* filename,
    const cpl_table* molecules);

cpl_boolean  mf_wrap_config_str_isin_strlst(
    const char* inp_str,
    const char* options_strlst);

cpl_error_code mf_config_chk_gdas_prof(const char* file);

cpl_error_code mf_config_chk_lnfl_line_db( const char* dirname,
		const char* db);

cpl_error_code  mf_wrap_config_stroptchk(
    const cpl_parameterlist* parlist,
    const char* parname,
    const char* options_strlst);

cpl_table * mf_wrap_config_table_mapping(
    const char              *mapping_str,
    const char              *column_name);

cpl_table * mf_wrap_config_table_molecules(
    const char               *list_molec_str,
    const char               *fit_molec_str,
    const char               *rel_col_str);

cpl_table * mf_wrap_config_table_ranges(
    const char              *range_str,
    cpl_type                type);

/* Get molecfit config parameters from the input recipe parameter list */
mf_configuration * mf_wrap_config_get_parameters(
    const cpl_parameterlist  *list,
    const cpl_propertylist   *data_primary_header,
    cpl_propertylist         *pl);




void mf_wrap_model_parameter_delete(
    mf_wrap_model_parameter *parameters);

#endif /* MF_WRAP_CONFIG_H */
