/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_wrap.h"
#include "mf_wrap_config_calc.h"


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

// LOAD TAG FUCNTIONS

cpl_error_code load_tag_science(mf_wrap_fits *data, cpl_frame * input_frame){
	  /* Load TAG = SCIENCE_CALCTRANS/SCIENCE */

	  cpl_error_code err= CPL_ERROR_NONE;

	  //mf_wrap_fits *data = NULL;
	  if (!err) {
	      const char *science_file = cpl_frame_get_filename(input_frame);

	      data = mf_wrap_fits_load(science_file, CPL_FALSE);

	      if (!data) err = CPL_ERROR_ILLEGAL_INPUT;
	      else       err = cpl_error_get_code();
	  }
	  return err;
}






/*cpl_error_code mf_wrap_config_calc_setup_input_filenames(cpl_parameterlist* ilist, const char* suffix, const char* input_name){

		cpl_error_code err= CPL_ERROR_NONE;

        const char* suf;
        if(!suffix) suf = "";
        else suf = "_%s" % (suffix);
        const char* type;
        if(strcmp(input_name,"SCI")){
            type ="_SCI";
        } else if(strcmp(input_name,"STD") || strcmp(input_name,"TELL")){
            type ="_STD";
        } else {
            type = "";
        }

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("ATM_PARAMETERS%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_ATM_PARAMETERS));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("BEST_FIT_PARAMETERS%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_BEST_FIT_PARAMETERS));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("BEST_FIT_MODEL%s%s",type,suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_BEST_FIT_MODEL));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("MODEL_MOLECULES%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_MODEL_MOLECULES));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("KERNEL_LIBRARY%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_KERNEL_LIBRARY));

        return CPL_ERROR_NONE;
    }

cpl_error_code mf_wrap_config_calc_setup_output_filenames(cpl_parameterlist* ilist, const char* suffix){

		cpl_error_code err = CPL_ERROR_NONE;

		const char* suf;
        if(!suffix) suf = "";
        else suf = "_%s" % (suffix);

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("TELLURIC_CORR%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_TELLURIC_CORR));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("CALCTRANS_KERNEL_LIBRARY%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_CALCTRANS_KERNEL_LIBRARY));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(cpl_sprintf("LBLRTM_RESULTS%s",suf),CPL_TYPE_STRING,NULL,NULL,MOLECFIT_LBLRTM_RESULTS));

        return CPL_ERROR_NONE;
}
*/









/**@}*/




/*----------------------------------------------------------------------------*/
/**
 * @brief  Generate the internal configuration file for the recipe and check values
 *
 * @param  list   parameterlist with the parameters
 *
 * @return configuration file or NULL if exist an error
 *
 */
/*----------------------------------------------------------------------------*/


molecfit_calctrans_parameter * molecfit_calctrans_parameters(
    const cpl_frameset              *frameset,
    const cpl_parameterlist         *list,
    const cpl_propertylist          *scientific_head_pri,
    const cpl_size                  n_ext)
{
  /* Check input */
  cpl_error_ensure(frameset && list, CPL_ERROR_NULL_INPUT,
                   return NULL, "list input is NULL!");

  /* Get preState */
  cpl_errorstate preState = cpl_errorstate_get();
  const cpl_parameter *p;

  /* Create the configuration parameter */
  molecfit_calctrans_parameter *parameters = cpl_malloc(sizeof(molecfit_calctrans_parameter));
  parameters->mapping_kernel_table               = NULL;
  parameters->mapping_atmospheric_table          = NULL;
  parameters->mapping_convolve_table             = NULL;
  parameters->atm_parameters                     = NULL;
  parameters->best_fit_parameters                = NULL;
  parameters->telluriccorr_head                      = NULL;
  parameters->telluriccorr_data                      = NULL;
  parameters->best_fit_parameters_table          = NULL;
  parameters->results_lblrtm                     = NULL;
  parameters->results_convolution                = NULL;
  parameters->mf_config                          = NULL;
  parameters->pl                                 = cpl_propertylist_new();


  /* Use only primary extension in the input FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI);
  parameters->use_only_input_pri_ext = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI, parameters->use_only_input_pri_ext);

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a error flux extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA);
  parameters->dflux_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA, parameters->dflux_extension_data);

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a mask extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MASK_EXTENSION_DATA);
  parameters->mask_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MASK_EXTENSION_DATA, parameters->mask_extension_data);


  /* User input kernel ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_INPUT_KERNEL);
  parameters->use_input_kernel = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_USE_INPUT_KERNEL, parameters->use_input_kernel);

  /* Mapping kernel */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL);
  parameters->mapping_kernel = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL, parameters->mapping_kernel);
  /* Create mapping kernel cpl_table */
  if (strcmp(parameters->mapping_kernel, MF_PARAMETERS_NULL)) {
      parameters->mapping_kernel_table = mf_wrap_config_table_mapping(parameters->mapping_kernel, MOLECFIT_MAPPING_KERNEL_EXT);
  }


  /* Mapping ATMOSPHERIC : SCIENCE_CALCTRANS/SCIENCE - ATM_PARAMETERS */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC);
  parameters->mapping_atmospheric = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC, parameters->mapping_atmospheric);
  /* Create mapping atmospheric cpl_table */
  if (strcmp(parameters->mapping_atmospheric, MF_PARAMETERS_NULL)) {
      parameters->mapping_atmospheric_table = mf_wrap_config_table_mapping(parameters->mapping_atmospheric, MOLECFIT_MAPPING_ATMOSPHERIC_EXT);
  }


  /* Mapping CONVOLVE : LBLRTM_RESULTS - TELLURIC_CORR */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_CONVOLVE);
  parameters->mapping_convolve = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MAPPING_CONVOLVE, parameters->mapping_convolve);
  /* Create mapping convolve cpl_table */
  if (strcmp(parameters->mapping_convolve, MF_PARAMETERS_NULL)) {
      parameters->mapping_convolve_table = mf_wrap_config_table_mapping(parameters->mapping_convolve, MOLECFIT_MAPPING_CONVOLVE_EXT);
  }


  /* Load TAG = MOLECFIT_ATM_PARAMETERS */
  cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)", MOLECFIT_ATM_PARAMETERS);
  const cpl_frame *frm_atm_parameters = cpl_frameset_find_const(frameset, MOLECFIT_ATM_PARAMETERS);
  if (frm_atm_parameters) {
      const char *filename = cpl_frame_get_filename(frm_atm_parameters);
      parameters->atm_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
  }

  /* Load TAG = MOLECFIT_BEST_FIT_PARAMETERS */
  cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)", MOLECFIT_BEST_FIT_PARAMETERS);
  const cpl_frame *frm_best_fit_parameters = cpl_frameset_find_const(frameset, MOLECFIT_BEST_FIT_PARAMETERS);
  if (frm_best_fit_parameters) {
      const char *filename = cpl_frame_get_filename(frm_best_fit_parameters);
      parameters->best_fit_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
  }

  /* Check if MOLECFIT_ATM_PARAMETERS and MOLECFIT_BEST_FIT_PARAMETERS were loaded */
  if (   !(parameters->atm_parameters       ) || !(parameters->best_fit_parameters       )
      ||   parameters->atm_parameters->n_ext  !=   parameters->best_fit_parameters->n_ext) {
      molecfit_calctrans_parameter_delete(parameters);
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                            "molecfit_model output files not correct or not provided!");
      return NULL;
  }

  /* Reserve memory for the output results */
  parameters->n_ext                              = n_ext;
  parameters->telluriccorr_head                  = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->telluriccorr_data                  = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->best_fit_parameters_table          = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->results_lblrtm                     = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_lblrtm_results      *));
  parameters->results_convolution                = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_convolution_results *));
  for (cpl_size ext = 0; ext < n_ext; ext++) {
      parameters->telluriccorr_head[ext]         = NULL;
      parameters->telluriccorr_data[ext]         = NULL;
      parameters->best_fit_parameters_table[ext] = NULL;
      parameters->results_lblrtm[ext]            = NULL;
      parameters->results_convolution[ext]       = NULL;
  }


  /* Get recipe parameters in the recipe molecfit_model --> Best_fit_parameters */
  parameters->mf_config = molecfit_config_load_header_parameters(parameters->best_fit_parameters->v_ext[0].header, scientific_head_pri, parameters->pl);


  /* Combine extensions in the FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CHIP_EXTENSIONS);
  parameters->chip_extensions = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_CHIP_EXTENSIONS, parameters->chip_extensions);


  /* Check status */
  if (!cpl_errorstate_is_equal(preState) || !(parameters->mf_config)) {
      /* Configuration failed */
      molecfit_calctrans_parameter_delete(parameters);
      return NULL;
  }

  return parameters;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Deallocate the given parameter configuration object and its contents
 *
 * @param    parameters       The parameter configuration variable in the recipe.
 */
/*----------------------------------------------------------------------------*/

void molecfit_calctrans_parameter_delete(
    molecfit_calctrans_parameter *parameters)
{
  if (parameters) {

      if (parameters->mapping_kernel_table)                   cpl_table_delete(                        parameters->mapping_kernel_table);

      if (parameters->mapping_atmospheric_table)              cpl_table_delete(                        parameters->mapping_atmospheric_table);

      if (parameters->mapping_convolve_table)                 cpl_table_delete(                        parameters->mapping_convolve_table);

      if (parameters->atm_parameters)                         mf_wrap_fits_delete(                  parameters->atm_parameters);
      if (parameters->best_fit_parameters)                    mf_wrap_fits_delete(                  parameters->best_fit_parameters);

      if (parameters->telluriccorr_head) {
          for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
              if (parameters->telluriccorr_head[ext])             cpl_propertylist_delete(                 parameters->telluriccorr_head[ext]);
          }
          cpl_free(parameters->telluriccorr_head);
      }

      if (parameters->telluriccorr_data) {
          for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
              if (parameters->telluriccorr_data[ext])             cpl_table_delete(                        parameters->telluriccorr_data[ext]);
          }
          cpl_free(parameters->telluriccorr_data);
      }

      if (parameters->best_fit_parameters_table) {
          for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
              if (parameters->best_fit_parameters_table[ext]) cpl_table_delete(                        parameters->best_fit_parameters_table[ext]);
          }
          cpl_free(parameters->best_fit_parameters_table);
      }

      if (parameters->results_lblrtm) {
          for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
              if (parameters->results_lblrtm[ext])            mf_calctrans_lblrtm_results_delete(      parameters->results_lblrtm[ext]);
          }
          cpl_free(parameters->results_lblrtm);
      }

      if (parameters->results_convolution) {
          for (cpl_size ext = 0; ext < parameters->n_ext; ext++) {
              if (parameters->results_convolution[ext])       mf_calctrans_convolution_results_delete( parameters->results_convolution[ext]);
          }
          cpl_free(parameters->results_convolution);
      }

      if (parameters->mf_config)                              mf_configuration_delete(                              parameters->mf_config);

      if (parameters->pl)                                     cpl_propertylist_delete(                 parameters->pl);

      cpl_free(parameters);
  }
}


/*----------------------------------------------------------------------------*/
/**
 * @brief  Generate the internal configuration file required for CALCTRANS .... mf_model and check values
 *
 * @param  parlist   parameterlist with the parameters
 *
 * @return mf_wrap_model_parameter parameters structure with instrument-independent defaults
 *
 */
/*----------------------------------------------------------------------------*/
molecfit_calctrans_parameter * mf_wrap_config_calc_init(cpl_frameset * frameset,const cpl_parameterlist  *list,cpl_propertylist *scientific_head_pri,cpl_size n_ext, const char* combined_suffix)
							// cpl_frameset *frameset --> was this needed ?
{
    /* copy of molecfit_calctrans_parameters from molecfit_calctrans.c */
    cpl_msg_info(cpl_func,"mf_wrap_config_calc_init");

  cpl_frameset_dump(frameset,stdout);
  /* Check input */
  cpl_error_ensure(frameset && list, CPL_ERROR_NULL_INPUT,
                   return NULL, "list input is NULL!");

  /* Get preState */
  cpl_errorstate preState = cpl_errorstate_get();
  const cpl_parameter *p;

  /* Create the configuration parameter */
  // molecfit_calctrans_parameter is defined in mf_wrap_config_calc.h
  molecfit_calctrans_parameter *parameters = cpl_malloc(sizeof(molecfit_calctrans_parameter));
  parameters->mapping_kernel_table               = NULL;
  parameters->mapping_atmospheric_table          = NULL;
  parameters->mapping_convolve_table             = NULL;
  parameters->atm_parameters                     = NULL;
  parameters->best_fit_parameters                = NULL;
  parameters->telluriccorr_head                      = NULL;
  parameters->telluriccorr_data                      = NULL;
  parameters->best_fit_parameters_table          = NULL;
  parameters->results_lblrtm                     = NULL;
  parameters->results_convolution                = NULL;
  parameters->mf_config                          = NULL;
  parameters->pl                                 = cpl_propertylist_new();

  cpl_msg_info(cpl_func,"molecfit_calctrans_parameter parameters mallocd %s",cpl_error_get_message());

  /* Use only primary extension in the input FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI);
  parameters->use_only_input_pri_ext = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI, parameters->use_only_input_pri_ext);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI %s",cpl_error_get_message());

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a error flux extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA);
  parameters->dflux_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA, parameters->dflux_extension_data);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA %s",cpl_error_get_message());

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a mask extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MASK_EXTENSION_DATA);
  parameters->mask_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MASK_EXTENSION_DATA, parameters->mask_extension_data);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_MASK_EXTENSION_DATA %s",cpl_error_get_message());

  /* User input kernel ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_INPUT_KERNEL);
  parameters->use_input_kernel = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_USE_INPUT_KERNEL, parameters->use_input_kernel);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_USE_INPUT_KERNEL %s",cpl_error_get_message());
  /* Mapping kernel */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL);
  parameters->mapping_kernel = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL, parameters->mapping_kernel);
  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_USE_INPUT_KERNEL 2 %s",cpl_error_get_message());
  /* Create mapping kernel cpl_table */
  if (strcmp(parameters->mapping_kernel, MF_PARAMETERS_NULL)) {
  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_USE_INPUT_KERNEL 3 %s",cpl_error_get_message());
      parameters->mapping_kernel_table = mf_wrap_config_table_mapping(parameters->mapping_kernel, MOLECFIT_MAPPING_KERNEL_EXT);
  }
  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_CALCTRANS_MAPPING_KERNEL %s",cpl_error_get_message());


  /* Mapping ATMOSPHERIC : SCIENCE_CALCTRANS/SCIENCE - ATM_PARAMETERS */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC);
  parameters->mapping_atmospheric = cpl_parameter_get_string(p);
  cpl_msg_info(cpl_func,"parameters->mapping_atmospheric == %s",parameters->mapping_atmospheric);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC, parameters->mapping_atmospheric);
  /* Create mapping atmospheric cpl_table */
  if (strcmp(parameters->mapping_atmospheric, MF_PARAMETERS_NULL)) {
      cpl_msg_info(cpl_func,"parameters->mapping_atmospheric != NULL");
      parameters->mapping_atmospheric_table = mf_wrap_config_table_mapping(parameters->mapping_atmospheric, MOLECFIT_MAPPING_ATMOSPHERIC_EXT);
  }
  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_MAPPING_ATMOSPHERIC %s",cpl_error_get_message());


  /* Mapping CONVOLVE : LBLRTM_RESULTS - TELLURIC_CORR */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_CONVOLVE);
  parameters->mapping_convolve = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MAPPING_CONVOLVE, parameters->mapping_convolve);
  /* Create mapping convolve cpl_table */
  if (strcmp(parameters->mapping_convolve, MF_PARAMETERS_NULL)) {
      parameters->mapping_convolve_table = mf_wrap_config_table_mapping(parameters->mapping_convolve, MOLECFIT_MAPPING_CONVOLVE_EXT);
  }
  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_MAPPING_CONVOLVE %s",cpl_error_get_message());


  /* Load TAG = MOLECFIT_ATM_PARAMETERS */
  const char* input_tag = mf_wrap_tag_suffix(MOLECFIT_ATM_PARAMETERS,combined_suffix,CPL_FALSE);
  cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)",input_tag);
  const cpl_frame *frm_atm_parameters = cpl_frameset_find_const(frameset, input_tag);
  if (frm_atm_parameters) {
      const char *filename = cpl_frame_get_filename(frm_atm_parameters);
      cpl_msg_info(cpl_func, "Loading %s from %s from molecfit_model(...)", filename,input_tag);
      parameters->atm_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
  }
  cpl_msg_info(cpl_func,"MOLECFIT_ATM_PARAMETERS %s",cpl_error_get_message());

  /* Load TAG = MOLECFIT_BEST_FIT_PARAMETERS */
  input_tag = mf_wrap_tag_suffix(MOLECFIT_BEST_FIT_PARAMETERS,combined_suffix,CPL_FALSE);
  cpl_msg_info(cpl_func, "Load %s from molecfit_model(...)", input_tag);
  const cpl_frame *frm_best_fit_parameters = cpl_frameset_find_const(frameset, input_tag);
  if (frm_best_fit_parameters) {
      const char *filename = cpl_frame_get_filename(frm_best_fit_parameters);
      cpl_msg_info(cpl_func, "Loading %s from %s from molecfit_model(...)", filename,input_tag);
      parameters->best_fit_parameters = mf_wrap_fits_load(filename, CPL_FALSE);
  }
  cpl_msg_info(cpl_func,"MOLECFIT_BEST_FIT_PARAMETERS %s",cpl_error_get_message());

  /* Check if MOLECFIT_ATM_PARAMETERS and MOLECFIT_BEST_FIT_PARAMETERS were loaded */
  if (   !(parameters->atm_parameters       ) || !(parameters->best_fit_parameters       )
      ||   parameters->atm_parameters->n_ext  !=   parameters->best_fit_parameters->n_ext) {
      cpl_msg_info(cpl_func,"MOLECFIT_BEST_FIT_PARAMETERS 2 %s",cpl_error_get_message());
      molecfit_calctrans_parameter_delete(parameters);
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                            "molecfit_model output files not correct or not provided!");
      return NULL;
  }
  //cpl_size n_ext = cpl_fits_count_extensions(science_fname);

  /* Reserve memory for the output results */
  parameters->n_ext                              = n_ext;
  parameters->telluriccorr_head                  = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->telluriccorr_data                  = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->best_fit_parameters_table          = cpl_calloc(parameters->n_ext, sizeof(cpl_table                        *));
  parameters->results_lblrtm                     = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_lblrtm_results      *));
  parameters->results_convolution                = cpl_calloc(parameters->n_ext, sizeof(mf_calctrans_convolution_results *));
  for (cpl_size ext = 0; ext < n_ext; ext++) {
      parameters->telluriccorr_head[ext]         = NULL;
      parameters->telluriccorr_data[ext]         = NULL;
      parameters->best_fit_parameters_table[ext] = NULL;
      parameters->results_lblrtm[ext]            = NULL;
      parameters->results_convolution[ext]       = NULL;
  }
  cpl_msg_info(cpl_func,"setup memory for output results %s",cpl_error_get_message());

  //IS THIS CORRECT?
  //cpl_propertylist* scientific_head_pri = cpl_propertylist_load(science_fname,0);//parameters->best_fit_parameters->v_ext[0].header;
  //scientific_head_pri

  cpl_msg_info(cpl_func,"scientific_head_pri %s",cpl_error_get_message());
  /* Get recipe parameters in the recipe molecfit_model --> Best_fit_parameters */
  parameters->mf_config = molecfit_config_load_header_parameters(parameters->best_fit_parameters->v_ext[0].header, scientific_head_pri, parameters->pl);

  cpl_msg_info(cpl_func,"molecfit_config_load_header_parameters %s",cpl_error_get_message());

  /* Combine extensions in the FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CHIP_EXTENSIONS);
  parameters->chip_extensions = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_CHIP_EXTENSIONS, parameters->chip_extensions);


  /* Check status */
  if (!cpl_errorstate_is_equal(preState) || !(parameters->mf_config)) {
      /* Configuration failed */
      molecfit_calctrans_parameter_delete(parameters);
      return NULL;
  }

  return parameters;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief  delete the memory allocations that are allocated
 *         from calling molecfit_config_load_header_parameters.
 *         Note: this is primarily used for unit tests
 *
 * @param  mf_configuration     mf_configuration object created from a call
 *                              to molecfit_config_load_header_parameters
 *
 * @return void
 */
/*----------------------------------------------------------------------------*/
void molecfit_config_delete_configuration(mf_configuration* cfg)
{
    mf_configuration_delete(cfg);
}



/*----------------------------------------------------------------------------*/
/**
 * @brief  Parse output header and add to the mf_parameters_config and the header propertylist
 *
 * @param  parameters_header      Header with input parameters.
 * @param  data_header            Header with the data parameters.
 * @param  pl                     Output header.
 *
 * @return mf_configuration       Molecfit configuration
 */
/*----------------------------------------------------------------------------*/

mf_configuration * molecfit_config_load_header_parameters(
    const cpl_propertylist   *parameters_header,
    const cpl_propertylist   *data_header,
    cpl_propertylist         *pl)
{
  /* Check input */
  cpl_error_ensure(parameters_header && data_header && pl, CPL_ERROR_NULL_INPUT,
                   return NULL, "Input is NULL!");

  /* Get preState */
  cpl_errorstate pre_state = cpl_errorstate_get();
  const char *p;


  /*** Get the molecfit parameters from the BEST_FIT_PARAMETERS primary header (cpl_propertylist *) ***/
  cpl_msg_info(cpl_func, "Loading input parameters saved in a output FITS header ...");

  /*** Molecfit default configuration ***/
  cpl_msg_info(cpl_func, "Load Molecfit default configuration ...");
  mf_configuration *mf_config = mf_configuration_create();
  if (!mf_config) {
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                            "Error loading Molecfit default configuration");
      return NULL;
  }

  /*** Set Molecfit configuration with recipe parameters ***/
  mf_parameters_directories  *directories  = &(mf_config->parameters->directories );
  mf_parameters_inputs       *inputs       = &(mf_config->parameters->inputs      );
  mf_parameters_fitting      *fitting      = &(mf_config->parameters->fitting     );
  mf_parameters_ambient      *ambient      = &(mf_config->parameters->ambient     );
  mf_parameters_instrumental *instrumental = &(mf_config->parameters->instrumental);
  mf_parameters_atmospheric  *atmospheric  = &(mf_config->parameters->atmospheric );


  /*** Directories ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELLURICCORR_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->telluric_path) cpl_free(directories->telluric_path);
      directories->telluric_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->telluric_path);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELLURICCORR_DATA_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->telluriccorr_data_path) cpl_free(directories->telluriccorr_data_path);
      directories->telluriccorr_data_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->telluriccorr_data_path);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TMP_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->tmp_path) cpl_free(directories->tmp_path);
      directories->tmp_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->tmp_path);
  }

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_PATH;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_path) cpl_free(directories->output_path);
      directories->output_path = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_path);
  }*/

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OUTPUT_NAME;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (directories->output_name) cpl_free(directories->output_name);
      directories->output_name = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, directories->output_name);
  }*/


  /*** Input parameters ***/
  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OMP_NUM_THREADS;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->omp_num_threads = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, inputs->omp_num_threads);
  }*/

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SILENT_EXTERNAL_BINS;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->silent_external_bins = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->silent_external_bins);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TRANSMISSION;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->transmission = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->transmission);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_LAMBDA;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_lambda) cpl_free(inputs->column_lambda);
      inputs->column_lambda = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_lambda);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_FLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_flux) cpl_free(inputs->column_flux);
      inputs->column_flux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_flux);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_DFLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_dflux) cpl_free(inputs->column_dflux);
      inputs->column_dflux = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_dflux);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_MASK;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->column_mask) cpl_free(inputs->column_mask);
      inputs->column_mask = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->column_mask);
  }


  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_DEFAULT_ERROR;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->default_error = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->default_error);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLG_TO_MICRON;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->wlg_to_micron = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->wlg_to_micron);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WAVELENGTH_FRAME;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->wavelengths_frame) cpl_free(inputs->wavelengths_frame);
      inputs->wavelengths_frame = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->wavelengths_frame);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_ERF_RV_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (inputs->observing_erv_rv.key) cpl_free(inputs->observing_erv_rv.key);
      inputs->observing_erv_rv.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, inputs->observing_erv_rv.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_ERF_RV_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->observing_erv_rv.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, inputs->observing_erv_rv.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CLEAN_MODEL_FLUX;
  if (cpl_propertylist_has(parameters_header, p)) {
      inputs->clean_flux = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, inputs->clean_flux);
  }



  /*** Convergence criterion ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FTOL;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->ftol = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->ftol);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_XTOL;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->xtol = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->xtol);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FLUX_UNIT;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->flux_unit = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->flux_unit);
  }


  /*** Telescope background ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_TELESCOPE_BACK;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_telescope_background.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_telescope_background.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_BACK_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_telescope_background.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_telescope_background.const_val);
  }


  /*** Continuum ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_CONTINUUM;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_continuum.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_N;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.n = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_continuum.n);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_continuum.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_continuum.const_val);
  }

  /*p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVATORY_BARY_RV;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->obs_bary_rv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->obs_bary_rv);
  }
  */


  /*** Wavelength solution fit/adjustment ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_WLC;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_wavelenght.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_N;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.n = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, fitting->fit_wavelenght.n);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_CONST;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_wavelenght.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_wavelenght.const_val);
  }

  /* p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_REF;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (fitting->wlc_ref) cpl_free(fitting->wlc_ref);
      fitting->wlc_ref = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, fitting->wlc_ref);
  } */


  /*** Default kernel: Boxcar kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_RES_BOX;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_res_box.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_res_box.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_BOX;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_res_box.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_res_box.const_val);
  }


  /*** Default kernel: Gaussian kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_GAUSS;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_gauss.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_gauss.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_GAUSS;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_gauss.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_gauss.const_val);
  }


  /*** Default kernel: Lorentz kernel ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_LORENTZ;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_lorentz.fit = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->fit_lorentz.fit);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_LORENTZ;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->fit_lorentz.const_val = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->fit_lorentz.const_val);
  }


  /*** Default kernels: Generic parameters ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_MODE;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->kern_mode = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->kern_mode);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_FAC;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->kern_fac = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, fitting->kern_fac);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_VAR_KERN;
  if (cpl_propertylist_has(parameters_header, p)) {
      fitting->var_kern = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, fitting->var_kern);
  }


  /*** Ambient ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVING_DATE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->observing_date.key) cpl_free(ambient->observing_date.key);
      ambient->observing_date.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->observing_date.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_OBSERVING_DATE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->observing_date.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->observing_date.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_UTC_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->utc.key) cpl_free(ambient->utc.key);
      ambient->utc.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->utc.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_UTC_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->utc.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->utc.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_ANGLE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->telescope_angle.key) cpl_free(ambient->telescope_angle.key);
      ambient->telescope_angle.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->telescope_angle.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TELESCOPE_ANGLE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->telescope_angle.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->telescope_angle.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RELATIVE_HUMIDITY_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->relative_humidity.key) cpl_free(ambient->relative_humidity.key);
      ambient->relative_humidity.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->relative_humidity.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RELATIVE_HUMIDITY_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->relative_humidity.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->relative_humidity.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PRESSURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->pressure.key) cpl_free(ambient->pressure.key);
      ambient->pressure.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->pressure.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PRESSURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->pressure.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->pressure.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TEMPERATURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->temperature.key) cpl_free(ambient->temperature.key);
      ambient->temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->temperature.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_TEMPERATURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->temperature.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->temperature.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_MIRROR_TEMPERATURE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->mirror_temperature.key) cpl_free(ambient->mirror_temperature.key);
      ambient->mirror_temperature.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->mirror_temperature.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_MIRROR_TEMPERATURE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->mirror_temperature.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->mirror_temperature.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->elevation.key) cpl_free(ambient->elevation.key);
      ambient->elevation.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->elevation.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->elevation.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->elevation.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LONGITUDE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->longitude.key) cpl_free(ambient->longitude.key);
      ambient->longitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->longitude.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LONGITUDE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->longitude.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->longitude.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (ambient->latitude.key) cpl_free(ambient->latitude.key);
      ambient->latitude.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, ambient->latitude.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      ambient->latitude.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, ambient->latitude.value);
  }


  /*** Instrumental ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SLIT_WIDTH_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (instrumental->slit_width.key) cpl_free(instrumental->slit_width.key);
      instrumental->slit_width.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, instrumental->slit_width.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_SLIT_WIDTH_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      instrumental->slit_width.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, instrumental->slit_width.value);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PIXEL_SCALE_KEYWORD;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (instrumental->pixel_scale.key) cpl_free(instrumental->pixel_scale.key);
      instrumental->pixel_scale.key = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, instrumental->pixel_scale.key);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PIXEL_SCALE_VALUE;
  if (cpl_propertylist_has(parameters_header, p)) {
      instrumental->pixel_scale.value = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, instrumental->pixel_scale.value);
  }


  /*** Atmospheric ***/
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_REFERENCE_ATMOSPHERIC;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (atmospheric->ref_atm) cpl_free(atmospheric->ref_atm);
      atmospheric->ref_atm = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, atmospheric->ref_atm);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_GDAS_PROFILE;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (atmospheric->gdas_prof) cpl_free(atmospheric->gdas_prof);
      atmospheric->gdas_prof = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, atmospheric->gdas_prof);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LAYERS;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->layers = cpl_propertylist_get_bool(parameters_header, p);
      cpl_propertylist_update_bool(pl, p, atmospheric->layers);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_EMIX;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->emix = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, atmospheric->emix);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_PWV;
  if (cpl_propertylist_has(parameters_header, p)) {
      atmospheric->pwv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, atmospheric->pwv);
  }


  /*** LNFL ***/
  mf_io_lnfl_config *lnfl = mf_config->lnfl;

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LNFL_LINE_DB;
  if (cpl_propertylist_has(parameters_header, p)) {
      if (lnfl->line_db) cpl_free(lnfl->line_db);
      lnfl->line_db = cpl_strdup(cpl_propertylist_get_string(parameters_header, p));
      cpl_propertylist_update_string(pl, p, lnfl->line_db);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LNFL_LINE_DB_FMT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lnfl->line_db_fmt = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lnfl->line_db_fmt);
  }


  /*** LBLRTM ***/
  mf_io_lblrtm_config *lblrtm = mf_config->lblrtm;

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ICNTNM;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->icntnm = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->icntnm);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_IAERSL;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->iaersl = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->iaersl);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_MPTS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->mpts = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->mpts);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NPTS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->npts = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->npts);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_V1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->v[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->v[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_V2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->v[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->v[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SAMPLE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sample = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->sample);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALFAL0;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->alfal0 = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->alfal0);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_AVMASS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->avmass = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->avmass);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DPTMIN;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->dptmin = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->dptmin);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DPTFAC;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->dptfac = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->dptfac);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TBOUND;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tbound = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tbound);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[1]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SREMIS3;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->sremis[2] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->sremis[2]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[1]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_SRREFL3;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->srrefl[2] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->srrefl[2]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_MODEL;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->model = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->model);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ITYPE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->itype = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->itype);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NOZERO;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->nozero = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->nozero);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_NOPRNT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->noprnt = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->noprnt);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_IPUNCH;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->ipunch = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->ipunch);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_RE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->re = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->re);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_HSPACE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->hspace = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->hspace);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_LATITUDE_VALUE; // MF_LBLRTM_REF_LAT;   // MF_PARAMETERS_LATITUDE_VALUE
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->ref_lat = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->ref_lat);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_ELEVATION_VALUE; // MF_LBLRTM_H1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->h[0] =  ambient->elevation.value /1000.00 ; // cpl_propertylist_get_double(parameters_header, p);
      /*cpl_propertylist_update_double(pl, p, lblrtm->h[0]);*/
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_H2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->h[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->h[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_RANGE;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->range = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->range);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_BETA;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->beta = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->beta);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_LEN;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->len = cpl_propertylist_get_int(parameters_header, p);
      cpl_propertylist_update_int(pl, p, lblrtm->len);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_HOBS;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->hobs = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->hobs);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_AVTRAT;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->avtrat = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->avtrat);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TDIFF1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tdiff[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tdiff[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_TDIFF2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->tdiff[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->tdiff[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALTD1;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->altd[0] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->altd[0]);
  }
  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_ALTD2;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->altd[1] = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->altd[1]);
  }

  p = MF_PARAMETERS_CONTEX_DEFAULT" "MF_LBLRTM_DELV;
  if (cpl_propertylist_has(parameters_header, p)) {
      lblrtm->delv = cpl_propertylist_get_double(parameters_header, p);
      cpl_propertylist_update_double(pl, p, lblrtm->delv);
  }


  /*** Update the Molecfit configuration with the header ***/
  cpl_error_code e = mf_parameters_config_update_with_header_keywords(mf_config->parameters, data_header);


  cpl_msg_info(cpl_func, "Columns : wave[%s], flux[%s], dflux[%s], mask[%s]", inputs->column_lambda, inputs->column_flux, inputs->column_dflux, inputs->column_mask);


  /* Check possible errors */
  if (!cpl_errorstate_is_equal(pre_state) || e != CPL_ERROR_NONE) {
      mf_configuration_delete(mf_config);
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                            "Error loading Molecfit default configuration");
      return NULL;
  }

  return mf_config;
}


