/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

//#include "telluriccorr.h"

#include "mf_wrap_utils.h"

/*#include "molecfit_dfs.h"
#include "molecfit_config.h"
#include "molecfit_data.h"*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_wrap_utils  Functions used by mf_wrap_* for molecfit model, calctrans and correct recipes.
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* -------------------- START: Common Functions -------------------- */

/*----------------------------------------------------------------------------*/
/**
 * @brief   check the entries in the recipe and classify the frameset with the tags
 *
 * @param   frameset      input set of frames
 *
 * @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_check_and_set_groups(
  cpl_frameset *frameset)
{
  /* Check size of frameset for to know if the sof file is not empty */
  cpl_size nframes = cpl_frameset_get_size(frameset);
  if(nframes <= 0){

      /* Empty sof file */
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                 "There aren't frames in the frameset");
  } else {
      for (cpl_size i = 0; i < nframes; i++) {

          cpl_frame  *frame    = cpl_frameset_get_position(frameset, i);
          const char *filename = cpl_frame_get_filename(frame);

          /* Check if the FITS file exist and have correct data,
           * return 0 if the fits file is valid without extensions */
          if (cpl_fits_count_extensions(filename) < 0){

              return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                         "Problem with the file '%s' (%s --> Code %d)",
                         filename, cpl_error_get_message(), cpl_error_get_code());
          }
      }
  }

  /* Identify the RAW, CONF and CALIB frames in the input frameset */
  if (mf_wrap_dfs_set_groups(frameset)) {

      /* Error classify frames */
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                 "Cannot classify RAW and/or CALIB frames");
  } else {

      /* Check classification */
      for (cpl_size i = 0; i < nframes; i++) {

          cpl_frame       *frame = cpl_frameset_get_position(frameset, i);
          const char      *tag   = cpl_frame_get_tag(frame);
          cpl_frame_group group  = cpl_frame_get_group(frame);

          /* The tag is invalid */
          if (group == CPL_FRAME_GROUP_NONE) {
              return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Frame:%lld with tag:%s is invalid", i, tag);
          }
      }
  }

  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief     Load kernel tag
 *
 * @param frameset           Set of input cpl_frame's
 * @param tag_kernel         Tag of kernel cpl_frame
 * @param use_input_kernel   Is it use the kernel?
 * @param parameters         Molecfit parameters configuration
 *
 * @return  mf_wrap_fits loaded or NULL if error.
 *
 */
/*----------------------------------------------------------------------------*/
mf_wrap_fits * mf_wrap_load_kernel_tag(
    cpl_frameset             *frameset,
    const char               *tag_kernel,
    cpl_boolean              use_input_kernel,
    mf_parameters_config     *parameters,
    cpl_boolean              verbose)
{
    /* Check inputs */
    if (!frameset) return NULL;

    mf_wrap_fits *kernel_data = NULL;


    cpl_errorstate pre_state = cpl_errorstate_get();
    const cpl_frame *input_kernel = cpl_frameset_find(frameset, tag_kernel);
    if (!input_kernel) {
        cpl_errorstate_set(pre_state);
        input_kernel = cpl_frameset_find(frameset, MOLECFIT_KERNEL_LIBRARY);
        if (input_kernel && verbose) cpl_msg_info(cpl_func, "Input KERNEL FITS file provided with the general tag = %s", MOLECFIT_KERNEL_LIBRARY);
    }

    if (input_kernel && use_input_kernel) {

        if (verbose) cpl_msg_info(cpl_func, "Load user kernel ...");
        const char *kernel_file = cpl_frame_get_filename(input_kernel);

        kernel_data = mf_wrap_fits_load_verbose(kernel_file, CPL_TRUE,verbose);
        if (!kernel_data) cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Load user kernel failed!");

    } else {

        if (     input_kernel    ) cpl_msg_warning(cpl_func, "User kernel is    provided but 'use_input_kernel' = false!");
        else if (use_input_kernel) cpl_msg_warning(cpl_func, "User kernel isn't provided but 'use_input_kernel' = true !");

        if (verbose) {
	    cpl_msg_info(cpl_func, "Using the synthetic kernels ...");
            cpl_msg_info(cpl_func, "Fit resolution by Boxcar   (--%s) = %d", MF_PARAMETERS_FIT_RES_BOX, parameters->fitting.fit_res_box.fit);
            cpl_msg_info(cpl_func, "Fit resolution by Gaussian (--%s) = %d", MF_PARAMETERS_FIT_GAUSS,   parameters->fitting.fit_gauss.fit  );
            cpl_msg_info(cpl_func, "Fit resolution by Lorentz  (--%s) = %d", MF_PARAMETERS_FIT_LORENTZ, parameters->fitting.fit_lorentz.fit);
        }
    }

    return kernel_data;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Convert input data to internal spectrum
 *
 * @param data                   .
 * @param use_only_pri_ext       .
 * @param dflux_extension_data   .
 * @param mask_extension_data    .
 * @param column_lambda          .
 * @param column_flux            .
 * @param column_dflux           .
 * @param column_mask            .
 *
 * @return  CPL_ERROR_NONE or code error if it fails.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_convert_to_table(
    mf_wrap_fits            *data,
    cpl_boolean              chip_extensions,
    cpl_boolean              use_only_pri_ext,
    int                      dflux_extension_data,
    int                      mask_extension_data,
    const char               *column_lambda,
    const char               *column_flux,
    const char               *column_dflux,
    const char               *column_mask)
{
    cpl_msg_info(cpl_func,"mf_wrap_data_convert_to_table");
    cpl_error_code err = CPL_ERROR_NONE;

    cpl_size n_spectrum = 0;

    /* Execution extensions */
    cpl_size n_ext = use_only_pri_ext ? 1 : data->n_ext;

    cpl_msg_info(cpl_func,"mf_wrap_data_convert_to_table 2");
    /* Create spectrum header */
    for (cpl_size ext = 0; ext < n_ext && !err; ext++) {
        data->v_ext[ext].spectrum_head = cpl_propertylist_duplicate(data->v_ext[ext].header);
    }
    cpl_msg_info(cpl_func,"mf_wrap_data_convert_to_table 3");

    /* Check if the raw_data is a FITS BINTABLE : If not, convert to input molecfit_data cpl_table. No throw error if some extension don't have data. */
    if (data->format == mf_wrap_fits_file_table) {

        for (cpl_size ext = 0; ext < n_ext && !err; ext++) {
            if (data->v_ext[ext].table) {
                int chip = chip_extensions ? CPL_MAX(ext, 1) : 1;
                cpl_msg_info(cpl_func, "File[%s], ext[%lld] - Extract cpl_table from cpl_table (chip=%d)", data->filename, ext, chip);
                data->v_ext[ext].spectrum_data = mf_wrap_data_extract_spectrum_from_cpl_table(data->v_ext[ext].spectrum_head, data->v_ext[ext].table,
                                                                                               chip, column_lambda, column_flux, column_dflux, column_mask);
                n_spectrum++;
           }
        }

    } else if (data->format == mf_wrap_fits_file_image_1D) {

        cpl_vector *dflux_in = NULL;
        cpl_vector *mask_in  = NULL;
        if (use_only_pri_ext && dflux_extension_data <= data->n_ext && mask_extension_data  <= data->n_ext) {
            dflux_in = data->v_ext[dflux_extension_data].vector;
            mask_in  = data->v_ext[mask_extension_data ].vector;
        }

        for (cpl_size ext = 0; ext < n_ext && !err; ext++) {
            if (data->v_ext[ext].vector) {
                int chip = chip_extensions ? CPL_MAX(ext, 1) : 1;
                cpl_msg_info(cpl_func, "File[%s], ext[%lld] - Extract cpl_table from cpl_vector (chip=%d)", data->filename, ext, chip);
                data->v_ext[ext].spectrum_data = mf_wrap_data_extract_spectrum_from_cpl_vector(data->v_ext[ext].spectrum_head, data->v_ext[ext].vector, dflux_in, mask_in,
                                                                                                chip, column_lambda, column_flux, column_dflux, column_mask);
                err = cpl_error_get_code();
                n_spectrum++;
            }
        }

    } else if (data->format == mf_wrap_fits_file_image_2D) {

        for (cpl_size ext = 0; ext < n_ext && !err; ext++) {
            if (data->v_ext[ext].image) {
                int chip = chip_extensions ? CPL_MAX(ext, 1) : 1;
                cpl_msg_info(cpl_func, "File[%s], ext[%lld] - Extract cpl_table from cpl_image (chip=%d)", data->filename, ext, chip);
                data->v_ext[ext].spectrum_data = mf_wrap_data_extract_spectrum_from_cpl_image(data->v_ext[ext].spectrum_head, data->v_ext[ext].image, NULL, NULL,
                                                                                               chip, column_lambda, column_flux, column_dflux, column_mask);
                err = cpl_error_get_code();
                n_spectrum++;
            }
        }

    } else if (data->format == mf_wrap_fits_file_image_3D) {

        for (cpl_size ext = 0; ext < n_ext && !err; ext++) {
            if (data->v_ext[ext].cube) {
                cpl_size size = cpl_imagelist_get_size(data->v_ext[ext].cube);
                int chip = chip_extensions ? CPL_MAX(ext, 1) : 1;
                cpl_msg_info(cpl_func, "File[%s], ext[%lld] - Extract cpl_table from cpl_imagelist - n_images = %lld (chip=%d)", data->filename, ext, size, chip);
                data->v_ext[ext].spectrum_data  = mf_wrap_data_extract_spectrum_from_cpl_imagelist(data->v_ext[ext].spectrum_head, data->v_ext[ext].cube, NULL, NULL,
                                                                                                    chip, column_lambda, column_flux, column_dflux, column_mask);
                err = cpl_error_get_code();
                n_spectrum++;
            }
        }
    }


    /* Combine molecfit_data */
    cpl_size index_combined = 0;
    if (chip_extensions && (data->n_ext > 1)) {
        index_combined = data->v_ext[0].spectrum_data ? 0 : 1;
        for (cpl_size ext = index_combined + 1; ext < data->n_ext && !err; ext++) {
            if (data->v_ext[ext].spectrum_data) {
                cpl_msg_info(cpl_func, "Combine data->molecfit_spectrum[ext=%lld] into first extension (data->molecfit_spectrum[%lld])", ext, index_combined);
                err = cpl_table_insert(data->v_ext[index_combined].spectrum_data, data->v_ext[ext].spectrum_data, cpl_table_get_nrow(data->v_ext[index_combined].spectrum_data));
                if (err != CPL_ERROR_NONE) cpl_msg_error(cpl_func, "Error insert molecfit_spectrum in first extension");
            }
        }
    }


    /* Check if the valid input have, at least, one extension with data : Show structure of input tables */
    if (!err) {
        if (n_spectrum == 0) {
            err = cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                        "Incompatible format -> It wasn't possible convert data to table : 0 spectrums converted");
        } else if (chip_extensions) {
            cpl_msg_info(cpl_func, "DATA for molecfit :  Combined extension[%lld] (number original spectrums = %lld)", index_combined, n_spectrum);
            if (data->v_ext[index_combined].spectrum_data) {
                cpl_msg_info(cpl_func, "DATA for molecfit :  Show spectrum cpl_table combined structure extension[%lld]", index_combined);
                cpl_table_dump_structure(data->v_ext[index_combined].spectrum_data, NULL);
            }
        } else {
            cpl_msg_info(cpl_func, "DATA for molecfit :  Number of extensions with data = %lld", n_spectrum);
            for (cpl_size ext = 0; ext < n_ext; ext++) {
                if (data->v_ext[ext].spectrum_data) {
                    cpl_msg_info(cpl_func, "DATA for molecfit :  Show spectrum cpl_table structure extension[%lld]", ext);
                    cpl_table_dump_structure(data->v_ext[ext].spectrum_data, NULL);
                }
            }
        }
    }
    return err;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load table in a input FITS BINTABLE with one only extension
 *
 * @param    frameset        Input frameset
 * @param    tag             TAG
 *
 * @return   cpl_table
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table * mf_wrap_load_unique_table(
    const cpl_frameset       *frameset,
    const char               *tag)
{
  /* Check inputs */
  cpl_error_ensure(frameset && tag, CPL_ERROR_NULL_INPUT, return NULL, "NULL input");

  /* Get frame */
  const cpl_frame *frame = cpl_frameset_find_const(frameset, tag);

  /* Load table */
  if (frame) {
      return mf_wrap_load_table(frame, 1);
  } else {
      cpl_msg_info(cpl_func, "TAG = %s : No 'cpl_frame *' provided in the input 'cpl_frameset *'", tag);
      return NULL;
  }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load table in a input FITS BINTABLE of the extension 'ext'
 *
 * @param    frame          Input frame
 * @param    ext            Extension to load
 *
 * @return   cpl_table
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table * mf_wrap_load_table(
    const cpl_frame          *frame,
    const cpl_size           ext)
{
  /* Check inputs */
  cpl_error_ensure(frame, CPL_ERROR_NULL_INPUT, return NULL, "NULL input");

  cpl_table *table = NULL;

  const char *tag      = cpl_frame_get_tag(frame);
  const char *filename = cpl_frame_get_filename(frame);
  if (filename) {

      cpl_size n_ext = cpl_frame_get_nextensions(frame);
      if (n_ext < ext) {

          cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO,
                                "Tag = %s, File = %s, n_ext = %lld : Cannot load extension = %lld", tag, filename, n_ext, ext);

      } else {

          table = cpl_table_load(filename, ext, 0);
          if (table) {
              cpl_msg_info(cpl_func, "Tag = %s, File = %s : Loaded 'cpl_table *' in extension %lld ...", tag, filename, ext);
              cpl_table_dump(table, 0, cpl_table_get_nrow(table), NULL);
          }
      }
  }

  return table;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Save to disk a new *.fits output file
 *
 * @param  all_frameset   Input frameset       in the recipe.
 * @param  used_frameset  frameset with the frame affected in the saving function.
 * @param  parlist        Input parameter list in the recipe.
 * @param  recipe         Name of the recipe
 * @param  list           Propertylist with the recipe input parameters
 * @param  tag            Tag in the ESO PRO CATG property.
 * @param  filename       Name of the output *.fits file, if NULL the function compose the name with the tag
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_save(
    cpl_frameset             *all_frameset,
    cpl_frameset             *used_frameset,
    const cpl_parameterlist  *parlist,
    const char               *recipe,
    cpl_propertylist         *list,
    const char               *tag,
    const char               *filename)
{
  /* Check inputs */
  cpl_error_ensure(all_frameset && used_frameset && parlist &&	strcmp(recipe, "") != 0 && list	&& strcmp(tag, "") != 0,
                   CPL_ERROR_NULL_INPUT, return CPL_ERROR_NULL_INPUT, "Null inputs in save function");

  /*** Save the base files ***/
  cpl_errorstate preState = cpl_errorstate_get();

  /* Set applist with the input recipe parameters */
  cpl_propertylist *applist = cpl_propertylist_duplicate(list);
  cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, tag);

  /*cpl_frameset_dump(all_frameset,stdout);
  cpl_propertylist_dump(applist,stdout);
  cpl_parameterlist_dump(parlist,stdout);
  */

  /* Get filename: If filename not NULL get this, if NULL create with tag */
  char *tag_fits;
  if (filename) tag_fits = cpl_sprintf("%s",      filename);
  else          tag_fits = cpl_sprintf("%s.fits", tag     );

  //cpl_msg_info(cpl_func,"Before msg_info Writing FITS...fname=%s %s; PACKAGE/PACKAGE_VERSION: %s/%s",tag_fits,cpl_error_get_message(),PACKAGE,PACKAGE_VERSION);
  /* Save to disk the fits file */
  //header == NULL
  //cpl_frame inherit  == NULL
  cpl_dfs_save_propertylist(  all_frameset, NULL, parlist, used_frameset, NULL,
                              recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                              tag_fits);

  //cpl_msg_info(cpl_func,"After msg_info Writing FITS... %s",cpl_error_get_message());
  cpl_free(tag_fits);

  /* Cleanup */
  cpl_propertylist_delete(applist);

  /* Check possible errors */
  if (!cpl_errorstate_is_equal(preState)) {
      return cpl_error_get_code();
  }

  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Save output cpl_table data to disk: Only add data to extension
 *
 * @param    header_data       Header of data
 * @param    name              Name of the output file
 * @param    addFitsExtension  If yes, the function add '.fits' to the name
 * @param    matrix            cpl_matrix kernel data to save (only one table or spec, if exist both only the table).
 * @param    table             cpl_table data to save (only one table or spec, if exist both only the table).
 * @param    vec               1D vector spectrum data to save (only one table or spec, if exist both only the table).
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_save_mf_results(
    cpl_propertylist         *header_data,
    const char               *name,
    cpl_boolean              addFitsExtension,
    cpl_matrix               *matrix,
    cpl_table                *table,
    cpl_vector               *vec)
{
  /* Check inputs */
  cpl_error_ensure(header_data && name,
                   CPL_ERROR_NULL_INPUT, return CPL_ERROR_NULL_INPUT, "Null inputs in molecfit save data execution");

  /*** Save the extension files ***/
  cpl_errorstate preState = cpl_errorstate_get();

  /* Get filename: If filename not NULL get this, if NULL create with tag */
  char *filename_fits = addFitsExtension ? cpl_sprintf("%s.fits", name) : cpl_strdup(name);

  /* Save to disk */
  if (matrix) {

      cpl_image *img = cpl_image_wrap_double(cpl_matrix_get_ncol(matrix),
                                             cpl_matrix_get_nrow(matrix),
                                             cpl_matrix_get_data(matrix));
      cpl_image_save(img, filename_fits, CPL_TYPE_DOUBLE, header_data, CPL_IO_EXTEND);
      cpl_image_unwrap(img);

  } else if (table) {

      cpl_table_save(table, NULL, header_data, filename_fits, CPL_IO_EXTEND);

  } else if (vec) {

      cpl_vector_save(vec, filename_fits, CPL_TYPE_DOUBLE, header_data, CPL_IO_EXTEND);

  } else {

      cpl_propertylist_save(header_data, filename_fits, CPL_IO_EXTEND);
  }

  cpl_free(filename_fits);

  /* Check possible errors */
  if (!cpl_errorstate_is_equal(preState)) {
      return cpl_error_get_code();
  }

  return CPL_ERROR_NONE;
}

//function to handle suffixes to tags (required for xshooter pipeline; suffix can be NULL - in which case, do not add suffix)
const char* mf_wrap_tag_suffix(const char* tag, const char* suffix,cpl_boolean fits){
    const char* result = cpl_sprintf("%s",tag);
    if(suffix){
        result = cpl_sprintf("%s_%s",result,suffix);
    }
    if(fits){
        result = cpl_sprintf("%s.fits",result);
    }
    return result;
}
/* -------------------- END: Common Functions -------------------- */

/* -------------------- START: DFS Functions -------------------- */



/* -------------------- END: DFS Functions ---------------------- */



/** @cond PRIVATE */


/** @endcond */


/**@}*/
