/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_parameters.h"

#include <cpl_test.h>

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_parameters   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int main(void)
{
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

  /* Initialize */
  cpl_errorstate errstate = cpl_errorstate_get();

  /* Get time */
  double cs = cpl_test_get_walltime();

  /* Create configuration */
  mf_parameters_config *config_parameters = mf_parameters_config_create();
  cpl_test_error(CPL_ERROR_NONE);

  if (config_parameters->inputs.column_lambda) cpl_free(config_parameters->inputs.column_lambda      );
  if (config_parameters->inputs.column_flux  ) cpl_free(config_parameters->inputs.column_flux        );
  if (config_parameters->inputs.column_dflux ) cpl_free(config_parameters->inputs.column_dflux       );
  if (config_parameters->inputs.column_mask  ) cpl_free(config_parameters->inputs.column_mask        );

  //if (config_parameters->directories.output_name) cpl_free(config_parameters->directories.output_name);

  config_parameters->inputs.column_lambda                 = cpl_sprintf("%s", "Wavelength");
  config_parameters->inputs.column_flux                   = cpl_sprintf("%s", "Extracted_OPT");
  config_parameters->inputs.column_dflux                  = cpl_sprintf("%s", "Error_OPT");
  config_parameters->inputs.column_mask                   = cpl_sprintf("%s", "NULL");

  config_parameters->inputs.wlg_to_micron                 = 1e-3;
//  config_parameters->directories.output_name              = cpl_sprintf("%s", "telluriccorr_test_crires");
  config_parameters->fitting.ftol                         = 1e-2;
  config_parameters->fitting.xtol                         = 1e-2;
  config_parameters->fitting.fit_telescope_background.fit = CPL_FALSE;
  config_parameters->fitting.fit_continuum.n              = 3;
  config_parameters->fitting.fit_wavelenght.n             = 3;
  config_parameters->fitting.fit_res_box.fit              = CPL_FALSE;
  config_parameters->fitting.fit_res_box.const_val        = 0.;
  config_parameters->fitting.fit_lorentz.fit              = CPL_FALSE;
  config_parameters->fitting.fit_lorentz.const_val        = 0.5;
  config_parameters->fitting.kern_fac                     = 300.;

  /* Initialize parameters */
  mf_parameters *params = mf_parameters_initialize(config_parameters, NULL);
  cpl_test_error(CPL_ERROR_NONE);
  cpl_test_nonnull(params);

  /* Set all fit flags to 1 */
  int nmolec = 3;

  cpl_array *molec_flags = cpl_array_new(nmolec, CPL_TYPE_INT);
  cpl_array *cont_flags  = cpl_array_new(1,      CPL_TYPE_INT);
  cpl_array *wlc_flags   = cpl_array_new(1,      CPL_TYPE_INT);
  cpl_array *res_flags   = cpl_array_new(3,      CPL_TYPE_INT);
  cpl_test_error(CPL_ERROR_NONE);

  cpl_array_fill_window_int(molec_flags, 0, nmolec, 1);
  cpl_array_fill_window_int(cont_flags,  0, 1,      1);
  cpl_array_fill_window_int(wlc_flags,   0, 1,      1);
  cpl_array_fill_window_int(res_flags,   0, 3,      1);
  cpl_test_error(CPL_ERROR_NONE);


  //cpl_table *prange_exc = cpl_table_load(TELLURICCORR_SOURCEDIR"/input/exclude_crires.fits", 1, 0);


  // TODO


  //cpl_table_delete(prange_exc);


  /* Cleanup */
  cpl_array_delete(molec_flags);
  cpl_array_delete(cont_flags );
  cpl_array_delete(wlc_flags  );
  cpl_array_delete(res_flags  );
  mf_parameters_delete(params);
  mf_parameters_config_delete(config_parameters);
  cpl_test_error(CPL_ERROR_NONE);

  /* Show time */
  double ce = cpl_test_get_walltime();
  cpl_msg_info(cpl_func, "test_mf_parameters() -> Run time: %g min\n", (ce - cs) / 60.);

  /* Show errors and return */
  cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
  return cpl_test_end(0);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/
