# SPDX-License-Identifier: BSD-3-Clause
import abc
from adari_core.plots.plot import Plot
from ..plots.combined import CombinedPlot
from adari_core.plots.images import ImagePlot
from adari_core.plots.points import (
    LinePlot,
    ScatterPlot,
    MARKERS,
    DEFAULT_MARKER,
    DEFAULT_MARKER_SIZE,
    DEFAULT_LINESTYLE,
    DEFAULT_LINEWIDTH,
)
from adari_core.plots.collapse import CollapsePlot
from adari_core.plots.histogram import HistogramPlot
from adari_core.plots.cut import CutPlot
from adari_core.plots.text import TextPlot


class BaseRenderer(metaclass=abc.ABCMeta):
    """
    The base class for the rendering of Plots and Panels.

    This class should be inherited by each backend module to
    provide definitions of the necessary methods.
    """

    MARKER_RENDER_COMMANDS = {m: {} for m in MARKERS}

    @staticmethod
    def _compute_marker_properties(plotobj, datadict):
        """
        Compute marker size and type given a ScatterPlot and data dict.
        """
        # Work out the marker and markersize for these data
        markersize = DEFAULT_MARKER_SIZE
        marker = DEFAULT_MARKER
        if datadict.get("markersize") is not None:
            markersize = datadict.get("markersize")
        elif plotobj.markersize is not None:
            markersize = plotobj.markersize
        if datadict.get("marker") is not None:
            marker = datadict.get("marker")
        elif plotobj.marker is not None:
            marker = plotobj.marker
        return markersize, marker

    @staticmethod
    def _compute_linestyle_properties(plotobj, datadict):
        """
        Compute linestyle for a LinePlot and data dict.
        """
        # Work out the linestyle for these data
        linestyle = DEFAULT_LINESTYLE
        if datadict.get("linestyle") is not None:
            linestyle = datadict.get("linestyle")
        elif plotobj.linestyle is not None:
            linestyle = plotobj.linestyle
        return linestyle

    @staticmethod
    def _compute_linewidth_properties(plotobj, datadict):
        """
        Compute linewidth for a LinePlot and data dict.
        """
        # Work out the linewidth for these data
        linewidth = DEFAULT_LINEWIDTH
        if datadict.get("linewidth") is not None:
            linewidth = datadict.get("linewidth")
        elif plotobj.linewidth is not None:
            linewidth = plotobj.linewidth
        return linewidth

    @abc.abstractmethod
    def __init__(self, *args, **kwargs):
        self._backend = "None"
        # Bind plot render functions to dict, with class name as key.
        # Will use the method defined in the superclass rather than this
        # base class for each
        self._plot_render_functions = {
            ImagePlot.__name__: self.plot_image,
            LinePlot.__name__: self.plot_line,
            HistogramPlot.__name__: self.plot_histogram,
            ScatterPlot.__name__: self.plot_scatter,
            CollapsePlot.__name__: self.plot_collapse,
            CombinedPlot.__name__: self.plot_combined,
            CutPlot.__name__: self.plot_cut,
            TextPlot.__name__: self.plot_text,
        }

        # List of artifacts created by this renderer
        self.artifacts = []

    @abc.abstractmethod
    def render_panels(self, panels_dict):
        """Render all the Panels.

        This function renders all the panels. In principle
        it should be just a loop that calls render_panel for each
        panel. However for some backends there is prepatory work
        to do common to all panels.
        See the documentation for individual backends for more details.

        Parameters
        ----------
        pnl : :any:`adari_core.plots.panel.Panel`
            The Panel to be rendered.
        pnl_name : str
            The Panel name
        pnl_description: str
            The Panel description that will be added to the metadata
        pnl_tags : iterable of str
            A list of tags that are added to the metadata
        """
        raise NotImplementedError("This must be implemented in a child class.")

    @abc.abstractmethod
    def render_panel(self, pnl, pnl_name, pnl_description, pnl_tags):
        """Render the given Panel.

        This function renders a given panel using the specific
        functionality of the backend being invoked. See the documentation
        for individual backends for more details.

        Parameters
        ----------
        pnl : :any:`adari_core.plots.panel.Panel`
            The Panel to be rendered.
        pnl_name : str
            The Panel name
        pnl_description: str
            The Panel description that will be added to the metadata
        pnl_tags : iterable of str
            A list of tags that are added to the metadata
        """
        raise NotImplementedError("This must be implemented in a child class.")

    def get_artifacts(self):
        """Return the created artifacts"""
        return self.artifacts

    def get_render_function(self, plotobj):
        """Get the class method that will render the given Plot.

        Parameters
        ----------
        plotobj : child of :any:`adari_core.plots.plot.Plot` object
            The Plot object to be rendered.

        Returns
        -------
        plot_method : class method
            The class method that will render the given Plot.

        Raises
        ------
        NotImplementedError
            If the given Plot type has no render function.
        """

        # Why is this code block bugged out? A: Plot wasn't being imported.
        self.check_plot_obj_class(plotobj, Plot)

        # Get the function required, or raise an error
        try:
            return self._plot_render_functions[plotobj.__str__()]
        except KeyError:
            raise NotImplementedError(
                "Backend %s has no function for " "rendering %s",
                self._backend,
                plotobj.__str__(),
            )

    @staticmethod
    def check_plot_obj_class(plotobj, c):
        """
        Check if the given object in a member of a given class.
        Parameters
        ----------
        plotobj : object
            The object to test.
        c : class
            The class to test against.

        Returns
        -------
        bool
            Whether plotobj is an instance of c (True) or not (False).
        """
        if not isinstance(plotobj, c):
            raise ValueError("The given object is not an instance of %s", c().__str__())

    @abc.abstractmethod
    def plot_image(self, plotobj, ax):
        """
        Render a representation of an ImagePlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.images.ImagePlot`
            The ImagePlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Image Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_line(self, plotobj, ax):
        """
        Render a representation of an LinePlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.points.LinePlot`
            The ImagePlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Line Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_scatter(self, plotobj, ax):
        """
        Render a representation of an ScatterPlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.points.ScatterPlot`
            The ScatterPlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Scatter Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_histogram(self, plotobj, ax):
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Histogram Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_collapse(self, plotobj, ax):
        """
        Render a representation of an CollapsePlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.collapse.CollapsePlot`
            The CollapsePlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering CollapsePlot",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_combined(self, plotobj, ax):
        """
        Render a representation of an CombinedPlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.combined.CombinedPlot`
            The ScatterPlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Combined Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_cut(self, plotobj, ax):
        """
        Render a representation of an CutPlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.cut.CutPlot`
            The ScatterPlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering Cut Plots",
            self._backend,
            plotobj.__str__(),
        )

    @abc.abstractmethod
    def plot_text(self, plotobj, ax):
        """
        Render a representation of a TextPlot object.

        Parameters
        ----------
        plotobj : :any:`adari_core.plots.text.TextPlot`
            The ScatterPlot object to be rendered.
        ax : plotting-like object
            The plotting-like object to render the Plot with/onto.
            Will vary between backends.
        """
        raise NotImplementedError(
            "Backend %s has no function for " "rendering TextPlots",
            self._backend,
            plotobj.__str__(),
        )
