# SPDX-License-Identifier: BSD-3-Clause

import argparse
import os
import logging
import logging.config
import re
import sys
import signal
import requests_unixsocket

from adari_core.client.client import AdariClient
from adari_core.server.server_ctl import AdariServerController
from adari_core import __version__

requests_unixsocket.monkeypatch()

LOGGING_LEVELS = {
    "notset": logging.NOTSET,
    "debug": logging.DEBUG,
    "info": logging.INFO,
    "warning": logging.WARNING,
    "error": logging.ERROR,
    "critical": logging.CRITICAL,
}

logger = logging.getLogger(__name__)

server = None
driver = None


def main():

    global server
    global driver

    # Parse arguments
    parser = argparse.ArgumentParser(description="genreport", exit_on_error=False)
    parser.add_argument(
        "-i", "--inputs", help="input files or directories", type=str, required=True
    )
    parser.add_argument(
        "-o", "--outputs", help="output artifacts", type=str, required=True
    )
    parser.add_argument(
        "-d",
        "--driver",
        help="driver (png, pdf, json, [not yet supported: desktop, bokeh])",
        type=str,
        required=True,
        choices=["png", "pdf", "json"],  # "desktop", "bokeh"],
    )
    parser.add_argument("-l", "--log-file", help="logfile", type=str)
    parser.add_argument(
        "--log-level",
        help="logging level (notset, debug, info, warning, " "error, critical",
        type=str,
    )
    parser.add_argument("report", help="name of report")
    parser.add_argument(
        "-V",
        "--version",
        action="version",
        version="%(prog)s {version}".format(version=__version__),
    )
    try:
        args = parser.parse_args()
    except argparse.ArgumentError as exc:
        if re.search("--driver$", exc.argument_name):
            parser.print_usage()
            logger.error(
                "genreport: error: argument {}: {}".format(
                    exc.argument_name, exc.message
                )
            )
            sys.exit(3)
        else:
            logger.error(
                "genreport: error: argument {}: {}".format(
                    exc.argument_name, exc.message
                )
            )
            raise exc
    driver = args.driver
    logger.debug(f"genreport args: {args}")

    try:
        # Set up the logging by manipulating the master logger for the module
        if args.log_level is not None:
            for arg, level in LOGGING_LEVELS.items():
                if args.log_level.casefold() == arg.casefold():
                    logging.getLogger("adari_core").setLevel(level)
                    logger.debug(
                        f"Set ADARI logger level "
                        f"to {level} == "
                        f"{logging.getLogger('adari_core').getEffectiveLevel()} "
                        f"({logging.getLevelName(logging.getLogger('adari_core').getEffectiveLevel())})"
                    )
        else:
            logging.getLogger("adari_core").setLevel(logging.INFO)

        # Try connecting to the Adari server or start it if not yet running
        if os.environ.get("XDG_RUNTIME_DIR") is not None:  # linux systemd as user
            adari_socket = os.environ["XDG_RUNTIME_DIR"]
        elif os.environ.get("RUNTIME_DIRECTORY") is not None:  # linux systemd as root
            adari_socket = os.environ["RUNTIME_DIRECTORY"]
        else:
            if sys.platform.startswith("linux"):  # linux
                if not os.path.isdir("/run/user/" + str(os.getuid())):
                    os.makedirs("/run/user/" + str(os.getuid()), mode=0o700)
                adari_socket = "/run/user/" + str(os.getuid())
            elif sys.platform.startswith("darwin"):  # macOS
                if not os.path.isdir("/private/tmp/adari_" + str(os.getuid())):
                    os.makedirs("/private/tmp/adari_" + str(os.getuid()), mode=0o700)
                adari_socket = "/private/tmp/adari_" + str(os.getuid())
            else:  # Try a generic option.
                # Other startswith() options:
                # 'freebsd', 'aix', 'emscripten', 'wasi', 'win32', 'cygwin',
                # 'msys', 'os2', 'os2emx', 'riscos', 'atheos', 'openbsd', etc.
                if not os.path.isdir("/var/run/user/" + str(os.getuid())):
                    os.makedirs("/var/run/user/" + str(os.getuid()), mode=0o700)
                adari_socket = "/var/run/user/" + str(os.getuid())
        adari_socket += "/adari_server_" + str(os.getpid()) + ".socket"

        # Requests for unixsocket requires a subtitution of slashes
        adari_url = "http+unix://" + adari_socket.replace("/", "%2F")
        logger.info(f"adari_url: {adari_url}")

        if driver == "bokeh":
            # adari_url = adari_socket + '/adari_server.socket'
            adari_url = "http://localhost:8000"

        # Create an Adari client
        # adari_client = AdariClient(adari_url)

        # try:
        # Try to connect with the server
        # adari_client.connect()
        # except requests.ConnectionError:
        # Start Adari server if connection was not successful
        logging.debug("Starting Adari server with driver " + driver)
        config = {"pwd": os.getcwd()}
        server = AdariServerController(driver, config, uds=adari_socket)
        server.start()

        # Install a signal handler for SIGTERM, SIGINT
        # This is done after the server has been started since uvicorn
        # child process inherits the signal and since version 0.29
        # it does not behave properly if previous signals where installed
        signal.signal(signal.SIGTERM, signal_handler)
        signal.signal(signal.SIGINT, signal_handler)

        # Create client
        adari_client = AdariClient(adari_url)

        # Confirm that now the server is active with a timeout of 10 sec.
        adari_client.connect(timeout=10)

        # Check whether the server supports the driver
        if not adari_client.server_supports_driver(driver):
            logging.error(
                "The current running Adari Server"
                + " does not support the driver "
                + driver
            )
            sys.exit(3)

        # Make the request to load the report
        retval = adari_client.loadreport(args.report)
        # will return None or exit code
        if retval:
            sys.exit(retval)

        # Make the request to run the report
        retval = adari_client.runreport(args.report, args.inputs, args.outputs)
        if retval:
            sys.exit(retval)

    finally:
        server_stop()


def server_stop():
    if server is not None and driver != "bokeh":
        server.stop()


def signal_handler(sig, frame):
    logging.warning("Termination requested. Stopping...")
    server_stop()
    sys.exit(128 + sig)


if __name__ == "__main__":
    main()
