# SPDX-License-Identifier: BSD-3-Clause
"""Colormaps for ADARI Core plotting.

This module should explicitly define colormaps (i.e,. break points and
corresponding colours) for use. This allows for absolutely consistency
between different plotting backends.
"""

import numpy as np
from collections.abc import Iterable
from adari_core.utils.clipping import ClippingMixin
import warnings
import re
try:
    import numpy.exceptions as np_exc # For numpy >= 1.25
except:
    import numpy as np_exc

HEX_REGEX = re.compile("^#?([a-f0-9]{6}|[a-f0-9]{3})$")


class ColorbarMixin(ClippingMixin, object):
    """
    Mixin for Plot objects to support colorbar operations.

    Includes colormaps for use in ADARI.
    """

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._colorbar_visible = False
        # Show colormap by default
        self.colorbar_visible = kwargs.get("show_colorbar", True)

        self._colormap = None
        # As per requirements 'heat' is the default colormap.
        # In matplotlib this is equivalent to 'hot'
        self.colormap = kwargs.get("colormap", "hot")

        # set default to be sigma with nsigma=1
        self._v_clip = kwargs.get("v_clip", "sigma")
        self._v_clip_kwargs = kwargs.get("v_clip_kwargs", {})
        if self._v_clip == "sigma" and not self._v_clip_kwargs:
            self._v_clip_kwargs = {"nsigma": 1}

    @property
    def colormap(self):
        """
        str : The colormap definition

        Colormaps have different structures internally between different
        backends. For example, matplotlib colorbars are an N x 4 array, where
        each row of the array defines the RGBA color at that breakpoint, while
        Bokeh 'palettes', by contrast, are simply lists of hex colour strings.

        However, matplotlib provides compatibility to use hex strings as an
        input for creating custom colormaps via
        `matplotlib.colors.LinearSegmentedColormap.from_list`; hence, to ensure
        compatibility use of both backends, ADARI uses an iterable of hex
        strings to represent a color map.

        The colormap can be set in one of two ways:

        - Pass the name of a pre-defined ADARI colormap. This will copy
          a representation of the colormap into the colormap property.
          Available colormaps are specified in
          :any:`adari_core.plots.colormaps.__colormaps__`.
        - Specify a custom iterable of hex strings, representing a custom
          colormap.
        """
        return self._colormap

    @colormap.setter
    def colormap(self, map_data):
        if isinstance(map_data, str):
            # map_data is a string and thus
            # should be set from the pre-defined maps in __colormaps__
            map_name = map_data
            # The next line raises ValueError on failure
            self._colormap = list(self.get_colormap(map_name))
            # NOTE: It is important the above line makes a copy, as it
            # currently does
        elif isinstance(map_data, Iterable):

            # This code block catches the 'ragged array' input
            # warning as an error instead
            warnings.filterwarnings("error")
            try:
                tmp_arr = np.array(map_data)
            except np_exc.VisibleDeprecationWarning as w:
                raise ValueError(str(w))
            warnings.filterwarnings("default")

            # --- RGB INPUT ---
            if tmp_arr.ndim == 2 and tmp_arr.shape[1] == 3:
                # Is in rgb format: 3xN tuple
                try:
                    assert not np.any(np.isnan(tmp_arr)), (
                        "Passed RGB values " "contain at least " "one NaN"
                    )
                    assert np.all(0 <= tmp_arr), (
                        "RGB values must be " "in the range [0, 255]"
                    )
                    assert np.all(tmp_arr <= 255), (
                        "RGB values must be " "in the range [0, 255]"
                    )
                except TypeError:  # Happens when tmp_arr has non-numeric values
                    raise ValueError(
                        "Your input RGB structure contained at "
                        "least one non-numeric value."
                    )
                except AssertionError as e:
                    raise ValueError(str(e))
                # Convert rgb data to hex strings
                # Use numpy's optimised iterator
                hex_list = ["#%02x%02x%02x" % tuple(rgb_vals) for rgb_vals in tmp_arr]
                try:
                    assert np.all([_validate_hex_string(_) for _ in hex_list]), (
                        "Failure in " "converting " "RGB sequence " "to hex sequence"
                    )
                except AssertionError as e:
                    raise RuntimeError(str(e))
                self._colormap = hex_list

            # --- Hex INPUT ---
            elif tmp_arr.ndim == 1 and tmp_arr.dtype == "<U7":
                # Hex strings are 7 chars long
                # Already in correct format, just set directly
                try:
                    assert np.all([_validate_hex_string(_) for _ in tmp_arr]), (
                        "Invalid hex string " "detected in input"
                    )
                except AssertionError as e:
                    raise ValueError(str(e))
                self._colormap = list(tmp_arr)
            else:
                # Incompatible iterator passed
                raise __invalid_colormap_error__
        else:
            # Incorrect type passed
            raise __invalid_colormap_error__

    @property
    def colorbar_visible(self):
        """bool : Determines if the Plot should display a colorbar."""
        return self._colorbar_visible

    @colorbar_visible.setter
    def colorbar_visible(self, t):
        self._colorbar_visible = bool(t)

    @staticmethod
    def get_colormap(map_name):
        """
        Get an ADARI colormap.

        Parameters
        ----------
        map_name : str
            The name of the colormap to return.

        Returns
        -------
        list of str
            The list of hex strings defining the requested colormap.
        """
        try:
            return __colormaps__[map_name]
        except KeyError:
            raise ValueError(
                "{} is not an ADARI colormap - please choose one "
                "of: {}".format(str(map_name), ", ".join(__colormaps__.keys()))
            )


__colormaps__ = {
    "jet": (
        "#30123b",
        "#311542",
        "#32184a",
        "#341b51",
        "#351e58",
        "#36215f",
        "#372365",
        "#38266c",
        "#392972",
        "#3a2c79",
        "#3b2f7f",
        "#3c3285",
        "#3c358b",
        "#3d3791",
        "#3e3a96",
        "#3f3d9c",
        "#4040a1",
        "#4043a6",
        "#4145ab",
        "#4148b0",
        "#424bb5",
        "#434eba",
        "#4350be",
        "#4353c2",
        "#4456c7",
        "#4458cb",
        "#455bce",
        "#455ed2",
        "#4560d6",
        "#4563d9",
        "#4666dd",
        "#4668e0",
        "#466be3",
        "#466de6",
        "#4670e8",
        "#4673eb",
        "#4675ed",
        "#4678f0",
        "#467af2",
        "#467df4",
        "#467ff6",
        "#4682f8",
        "#4584f9",
        "#4587fb",
        "#4589fc",
        "#448cfd",
        "#438efd",
        "#4291fe",
        "#4193fe",
        "#4096fe",
        "#3f98fe",
        "#3e9bfe",
        "#3c9dfd",
        "#3ba0fc",
        "#39a2fc",
        "#38a5fb",
        "#36a8f9",
        "#34aaf8",
        "#33acf6",
        "#31aff5",
        "#2fb1f3",
        "#2db4f1",
        "#2bb6ef",
        "#2ab9ed",
        "#28bbeb",
        "#26bde9",
        "#25c0e6",
        "#23c2e4",
        "#21c4e1",
        "#20c6df",
        "#1ec9dc",
        "#1dcbda",
        "#1ccdd7",
        "#1bcfd4",
        "#1ad1d2",
        "#19d3cf",
        "#18d5cc",
        "#18d7ca",
        "#17d9c7",
        "#17dac4",
        "#17dcc2",
        "#17debf",
        "#18e0bd",
        "#18e1ba",
        "#19e3b8",
        "#1ae4b6",
        "#1be5b4",
        "#1de7b1",
        "#1ee8af",
        "#20e9ac",
        "#22eba9",
        "#24eca6",
        "#27eda3",
        "#29eea0",
        "#2cef9d",
        "#2ff09a",
        "#32f197",
        "#35f394",
        "#38f491",
        "#3bf48d",
        "#3ff58a",
        "#42f687",
        "#46f783",
        "#4af880",
        "#4df97c",
        "#51f979",
        "#55fa76",
        "#59fb72",
        "#5dfb6f",
        "#61fc6c",
        "#65fc68",
        "#69fd65",
        "#6dfd62",
        "#71fd5f",
        "#74fe5c",
        "#78fe59",
        "#7cfe56",
        "#80fe53",
        "#84fe50",
        "#87fe4d",
        "#8bfe4b",
        "#8efe48",
        "#92fe46",
        "#95fe44",
        "#98fe42",
        "#9bfd40",
        "#9efd3e",
        "#a1fc3d",
        "#a4fc3b",
        "#a6fb3a",
        "#a9fb39",
        "#acfa37",
        "#aef937",
        "#b1f836",
        "#b3f835",
        "#b6f735",
        "#b9f534",
        "#bbf434",
        "#bef334",
        "#c0f233",
        "#c3f133",
        "#c5ef33",
        "#c8ee33",
        "#caed33",
        "#cdeb34",
        "#cfea34",
        "#d1e834",
        "#d4e735",
        "#d6e535",
        "#d8e335",
        "#dae236",
        "#dde036",
        "#dfde36",
        "#e1dc37",
        "#e3da37",
        "#e5d838",
        "#e7d738",
        "#e8d538",
        "#ead339",
        "#ecd139",
        "#edcf39",
        "#efcd39",
        "#f0cb3a",
        "#f2c83a",
        "#f3c63a",
        "#f4c43a",
        "#f6c23a",
        "#f7c039",
        "#f8be39",
        "#f9bc39",
        "#f9ba38",
        "#fab737",
        "#fbb537",
        "#fbb336",
        "#fcb035",
        "#fcae34",
        "#fdab33",
        "#fda932",
        "#fda631",
        "#fda330",
        "#fea12f",
        "#fe9e2e",
        "#fe9b2d",
        "#fe982c",
        "#fd952b",
        "#fd9229",
        "#fd8f28",
        "#fd8c27",
        "#fc8926",
        "#fc8624",
        "#fb8323",
        "#fb8022",
        "#fa7d20",
        "#fa7a1f",
        "#f9771e",
        "#f8741c",
        "#f7711b",
        "#f76e1a",
        "#f66b18",
        "#f56817",
        "#f46516",
        "#f36315",
        "#f26014",
        "#f15d13",
        "#ef5a11",
        "#ee5810",
        "#ed550f",
        "#ec520e",
        "#ea500d",
        "#e94d0d",
        "#e84b0c",
        "#e6490b",
        "#e5460a",
        "#e3440a",
        "#e24209",
        "#e04008",
        "#de3e08",
        "#dd3c07",
        "#db3a07",
        "#d93806",
        "#d73606",
        "#d63405",
        "#d43205",
        "#d23005",
        "#d02f04",
        "#ce2d04",
        "#cb2b03",
        "#c92903",
        "#c72803",
        "#c52602",
        "#c32402",
        "#c02302",
        "#be2102",
        "#bb1f01",
        "#b91e01",
        "#b61c01",
        "#b41b01",
        "#b11901",
        "#ae1801",
        "#ac1601",
        "#a91501",
        "#a61401",
        "#a31201",
        "#a01101",
        "#9d1001",
        "#9a0e01",
        "#970d01",
        "#940c01",
        "#910b01",
        "#8e0a01",
        "#8b0901",
        "#870801",
        "#840701",
        "#810602",
        "#7d0502",
        "#7a0402",
    ),
    "hot": (
        "#800026",
        "#bd0026",
        "#e31a1c",
        "#fc4e2a",
        "#fd8d3c",
        "#feb24c",
        "#fed976",
        "#ffeda0",
        "#ffffcc",
    ),
    "grey": (
        "#000000",
        "#010101",
        "#020202",
        "#030303",
        "#040404",
        "#050505",
        "#060606",
        "#070707",
        "#080808",
        "#090909",
        "#0a0a0a",
        "#0b0b0b",
        "#0c0c0c",
        "#0d0d0d",
        "#0e0e0e",
        "#0f0f0f",
        "#101010",
        "#111111",
        "#121212",
        "#131313",
        "#141414",
        "#151515",
        "#161616",
        "#171717",
        "#181818",
        "#191919",
        "#1a1a1a",
        "#1b1b1b",
        "#1c1c1c",
        "#1d1d1d",
        "#1e1e1e",
        "#1f1f1f",
        "#202020",
        "#212121",
        "#222222",
        "#232323",
        "#242424",
        "#252525",
        "#262626",
        "#272727",
        "#282828",
        "#292929",
        "#2a2a2a",
        "#2b2b2b",
        "#2c2c2c",
        "#2d2d2d",
        "#2e2e2e",
        "#2f2f2f",
        "#303030",
        "#313131",
        "#323232",
        "#333333",
        "#343434",
        "#353535",
        "#363636",
        "#373737",
        "#383838",
        "#393939",
        "#3a3a3a",
        "#3b3b3b",
        "#3c3c3c",
        "#3d3d3d",
        "#3e3e3e",
        "#3f3f3f",
        "#404040",
        "#414141",
        "#424242",
        "#434343",
        "#444444",
        "#454545",
        "#464646",
        "#474747",
        "#484848",
        "#494949",
        "#4a4a4a",
        "#4b4b4b",
        "#4c4c4c",
        "#4d4d4d",
        "#4e4e4e",
        "#4f4f4f",
        "#505050",
        "#515151",
        "#525252",
        "#535353",
        "#545454",
        "#555555",
        "#565656",
        "#575757",
        "#585858",
        "#595959",
        "#5a5a5a",
        "#5b5b5b",
        "#5c5c5c",
        "#5d5d5d",
        "#5e5e5e",
        "#5f5f5f",
        "#606060",
        "#616161",
        "#626262",
        "#636363",
        "#646464",
        "#656565",
        "#666666",
        "#676767",
        "#686868",
        "#696969",
        "#6a6a6a",
        "#6b6b6b",
        "#6c6c6c",
        "#6d6d6d",
        "#6e6e6e",
        "#6f6f6f",
        "#707070",
        "#717171",
        "#727272",
        "#737373",
        "#747474",
        "#757575",
        "#767676",
        "#777777",
        "#787878",
        "#797979",
        "#7a7a7a",
        "#7b7b7b",
        "#7c7c7c",
        "#7d7d7d",
        "#7e7e7e",
        "#7f7f7f",
        "#808080",
        "#818181",
        "#828282",
        "#838383",
        "#848484",
        "#858585",
        "#868686",
        "#878787",
        "#888888",
        "#898989",
        "#8a8a8a",
        "#8b8b8b",
        "#8c8c8c",
        "#8d8d8d",
        "#8e8e8e",
        "#8f8f8f",
        "#909090",
        "#919191",
        "#929292",
        "#939393",
        "#949494",
        "#959595",
        "#969696",
        "#979797",
        "#989898",
        "#999999",
        "#9a9a9a",
        "#9b9b9b",
        "#9c9c9c",
        "#9d9d9d",
        "#9e9e9e",
        "#9f9f9f",
        "#a0a0a0",
        "#a1a1a1",
        "#a2a2a2",
        "#a3a3a3",
        "#a4a4a4",
        "#a5a5a5",
        "#a6a6a6",
        "#a7a7a7",
        "#a8a8a8",
        "#a9a9a9",
        "#aaaaaa",
        "#ababab",
        "#acacac",
        "#adadad",
        "#aeaeae",
        "#afafaf",
        "#b0b0b0",
        "#b1b1b1",
        "#b2b2b2",
        "#b3b3b3",
        "#b4b4b4",
        "#b5b5b5",
        "#b6b6b6",
        "#b7b7b7",
        "#b8b8b8",
        "#b9b9b9",
        "#bababa",
        "#bbbbbb",
        "#bcbcbc",
        "#bdbdbd",
        "#bebebe",
        "#bfbfbf",
        "#c0c0c0",
        "#c1c1c1",
        "#c2c2c2",
        "#c3c3c3",
        "#c4c4c4",
        "#c5c5c5",
        "#c6c6c6",
        "#c7c7c7",
        "#c8c8c8",
        "#c9c9c9",
        "#cacaca",
        "#cbcbcb",
        "#cccccc",
        "#cdcdcd",
        "#cecece",
        "#cfcfcf",
        "#d0d0d0",
        "#d1d1d1",
        "#d2d2d2",
        "#d3d3d3",
        "#d4d4d4",
        "#d5d5d5",
        "#d6d6d6",
        "#d7d7d7",
        "#d8d8d8",
        "#d9d9d9",
        "#dadada",
        "#dbdbdb",
        "#dcdcdc",
        "#dddddd",
        "#dedede",
        "#dfdfdf",
        "#e0e0e0",
        "#e1e1e1",
        "#e2e2e2",
        "#e3e3e3",
        "#e4e4e4",
        "#e5e5e5",
        "#e6e6e6",
        "#e7e7e7",
        "#e8e8e8",
        "#e9e9e9",
        "#eaeaea",
        "#ebebeb",
        "#ececec",
        "#ededed",
        "#eeeeee",
        "#efefef",
        "#f0f0f0",
        "#f1f1f1",
        "#f2f2f2",
        "#f3f3f3",
        "#f4f4f4",
        "#f5f5f5",
        "#f6f6f6",
        "#f7f7f7",
        "#f8f8f8",
        "#f9f9f9",
        "#fafafa",
        "#fbfbfb",
        "#fcfcfc",
        "#fdfdfd",
        "#fefefe",
        "#ffffff",
    ),
}
"""
The pre-defined ADARI colormaps.

The maps defined are:

jet
    A 'rainbow' style map from cold (blue/purple) to hot (orange/red).
hot
    A 'heat' style map from cold (brown) to hot (yellow).
grey
    A greyscale map from low (black) to high (white).

Requests to add additional colormaps to the pre-defined set should be
directed towards the ADARI team.
"""

__invalid_colormap_error__ = ValueError(
    "'colormap' must be either a set of "
    "list colors in (r,g,b) tuple "
    "format, as hex strings, or a single "
    "string referring to "
    "an ADARI colormap ({})".format(",".join(__colormaps__.keys()))
)


def _validate_hex_string(s):
    """
    Validate that the passed string is a valid hex code.

    Parameters
    ----------
    s : str
       String to test.

    Returns
    -------
    bool
        True if s is a valid hex string, False otherwise.
    """
    try:
        match = re.search(HEX_REGEX, s)
        if match:
            return True
        return False
    except (ValueError, TypeError):
        return False
