# SPDX-License-Identifier: BSD-3-Clause
"""This module specifies the CutPlot type for ADARI.

Cut Plots inherit most functionaility from the :any:`LinePlot` type from
:any:`adari_core.plots.points`, but only accepts an :any:`ImagePlot` object
as data.
"""
from .points import LinePlot
from .images import ImagePlot

CUT_PLOT_DESCRIPTION_TRUNCATION_LIMIT = 15


class CutPlot(LinePlot):
    """
    Shows a plot of data values along a set line through an image.

    Cut plots show the pixel values along one column (cutting the x axis) or
    row (cutting the y axis) of a 2D image. By convention, they are always
    displayed together with the image in the same :any:`Panel`,
    and are connected to that :any:`ImagePlot`.

    Parameters
    ----------
    cut_ax : str, "x" or "y"
        The axis to cut in the image ('x' or 'y')
    data : :any:`ImagePlot` object, optional
        :any:`ImagePlot` from which to make the CutPlot. Defaults to None,
        at which point data can be added later using :any:`add_data`.
    cut_pos : int, optional
        Point along the `cut_ax` axis to cut to extract values.
        Optional, but must be provided if `data` is provided.
    """

    def __str__(self):
        return "CutPlot"

    def __init__(self, cut_ax, *args, data=None, cut_pos=None, **kwargs):
        super().__init__(*args, **kwargs)
        self.cut_ax = cut_ax
        if cut_ax == "x":
            self.x_label = "y"  # Y values down column at pos X
        elif cut_ax == "y":
            self.x_label = "x"  # X values along row at pos Y
        # if the x_label or y_label are overridden in the constructor
        # (e.g. if we have rotated data), make sure they override the defaults
        self.x_label = kwargs.get("x_label", self.x_label)
        self.y_label = kwargs.get("y_label", self.y_label)

        self._data = {}
        if data is not None:
            if cut_pos is None:
                raise ValueError(
                    "If data is given, cut_pos should also be \
                                given"
                )
            self.add_data(data, cut_pos)

    @staticmethod
    def _generate_description(input_data=None, ax=None, pos=None):
        """
        Generate a description string for use in labels, titles etc.

        Any of the input parameters can be set to 'None'. This denotes that
        there are multiple possible values for this parameter in the
        description we are attempting to generate, so a special value will be
        used instead.

        The descriptions returned are of the following format, depending on
        which parameters are set to non-None values:

        - input_data: "<input_data.title>"
        - ax: "<row/col>"
        - pos: "@ <pos>"
        - input_data, ax: "<input_data> <row/col>
        - input_data, pos: "<input_data.title> @ <pos>"
        - ax, pos: "<row/col> @ <x/y>=<pos>"
        - input_data, ax, pos: "<input_data.title> <row/column> @ <x/y>=<pos>"


        Parameters
        ----------
        input_data : ImagePlot, or child of
            The input ImagePlot (or child of) being used
        ax : str, "x" or "y"
            The axis the cut is to be made along.
        pos : float
            The *data* coordinate where the cut is to be made.

        Returns
        -------
        description : str
            The description requested. The format of the description will
            be different depending on which of input_data, ax and pos are/are
            not provided.
        """

        if input_data is None and pos is None and ax is None:
            raise ValueError("Cannot have all inputs be None!")
        if ax not in ["x", "y", None]:
            raise ValueError("Invalid axis value {}".format(ax))

        ret_str = ""
        if input_data is not None:
            input_title_words = input_data.title.rsplit(" ")
            input_title_trunc = input_title_words[0]
            i = 1
            while len(
                input_title_trunc
            ) <= CUT_PLOT_DESCRIPTION_TRUNCATION_LIMIT and i < len(input_title_words):
                input_title_trunc += " " + input_title_words[i]
                i += 1
            ret_str += f"{input_title_trunc}"
        if ax is not None:
            ret_str += f" {'col' if ax=='x' else 'row'}"
        if pos is not None:
            if ax is not None:
                ret_str += f" @ {ax}={pos}"
            else:
                ret_str += f" @ {pos}"

        return ret_str.strip()

    def add_data(self, data, cut_pos, *args, label=None, color=None, **kwargs):
        """
        Add an :any:`ImagePlot` source to this CutPlot.

        data : :any:`ImagePlot` object
            :any:`ImagePlot` from which to make the CutPlot.
        cut_pos : int
            Point along the `cut_ax` axis to cut to extract values. Note that
            this position is given in *data* coordinates.
        color : str, optional
            Color specification for plotting the line in. Defaults to None, at
            which point a color will be randomly assigned.
        """
        # FIXME check if cutpos is valid
        # TODO add special words for cut_pos, e.g. 'center'
        self._is_data_valid(data, fail_on_none=True, error_on_fail=True)
        if label is None:
            label = self._generate_description(data, self.cut_ax, cut_pos)
        self._data[label] = {"data": data, "color": color, "cut_pos": cut_pos}

    def _is_data_valid(self, data, fail_on_none=True, error_on_fail=True):
        try:
            # Ensure data is an image plot. Currently no other data types
            # are to be supported and it is always shown along with the image
            # plot
            assert isinstance(data, ImagePlot), (
                "CutPlot requires an " "ImagePlot " "object as its data source"
            )
            return True
        except AssertionError as e:
            # If we get here, we didn't return True, so return False
            if error_on_fail:
                raise ValueError(str(e))
            return False

    def get_labels(self):
        """
        Get all currently-existing data labels for this CutPlot.

        Returns
        -------
        labels: list of str
            The existing data labels on this CutPlot object. This list is
            not ordered in any particular way.
        """
        return self._data.keys()

    def update_cut_pos(self, label, cut_pos):
        """
        Alter the cut_pos location of a particular data set within this CutPlot.

        Parameters
        ----------
        label : str
            The label of the data set to update
        cut_pos : float, or similar
            The updated cut position.

        Raises
        -------
        ValueError
            If the given `label` does not exist in this CutPlot.
        """
        try:
            self.data[label]["cut_pos"] = cut_pos
        except KeyError:
            raise ValueError(f"The label {label} does " f"not exist in this CutPlot.")

    def retrieve_data(self, label):
        """
        Returns a reference to the data set of a particular label.

        Parameters
        ----------
        label : str
            The label of the data set for which to return the underlying data.

        Returns
        -------
        data
            The underlying data set. Normally a :any:`numpy` array or similar.

        Raises
        ------
        ValueError
            Raised if the label does not exist in this CutPlot
        """
        try:
            return self.data[label]["data"]
        except KeyError:
            raise ValueError(f"Label {label} does not exist in this CutPlot")

    @property
    def cut_ax(self):
        """str : the axis to cut in the image ('x' or 'y')"""
        return self._cut_ax

    @cut_ax.setter
    def cut_ax(self, cut_ax):
        if cut_ax not in ("x", "y"):
            raise ValueError("cut_ax must be 'x' or 'y'")
        self._cut_ax = cut_ax

    # Override the default title getter for semi-automated behaviour
    @property
    def title(self):
        if self._title is not None:
            return self._title

        # There are three cases:
        # - No data attached
        # - One data attached
        # - Multiple data attached
        # We can borrow/adopt the various labels to make a title
        if len(self._data) == 0:
            return self._generate_description(None, self.cut_ax, None)
        elif len(self._data) == 1:
            return self._generate_description(
                list(self._data.values())[0]["data"],
                self.cut_ax,
                list(self._data.values())[0]["cut_pos"],
            )
        else:
            # See if all input data are the same ImagePlot
            if len(set([_["data"] for _ in self._data.values()])) == 1:
                return self._generate_description(
                    list(self._data.values())[0]["data"], self.cut_ax, None
                )  # Assume multiple posns
            else:
                return self._generate_description(None, self.cut_ax, None)

    @title.setter
    def title(self, t):
        self._title = str(t)
