# SPDX-License-Identifier: BSD-3-Clause
import abc


class Plot(metaclass=abc.ABCMeta):
    """The root Plot class.

    A Plot is a single graphical representation of astronomical data. It can
    take many forms (e.g. image, line plot, histogram etc), which should
    be defined as children of the parent Plot class.

    Most of the work of this class will be done by the child classes.

    If given, ADARI will attempt to load the first positional argument
    as data to the Plot.
    """

    def __init__(self, *args, **kwargs):
        # Init private attributes
        self._data = None
        self._title = None
        if len(args) > 0:
            self.add_data(args[0])
        if kwargs.get("title") is not None:
            self.title = kwargs.get("title")
        self._legend = kwargs.get("legend", True)

    def __str__(self):
        return "Plot"

    @property
    def legend(self):
        """
        bool: Show legend
        """
        return self._legend

    @legend.setter
    def legend(self, legend):
        self._legend = legend

    @abc.abstractmethod
    def _is_data_valid(self, data, fail_on_none=True, error_on_fail=True):
        """Determines if the data provided is valid or not for this Plot.

        Parameters
        ----------
        data : assorted
            The data for this Plot. The required structure is set by the
            plot type.
        fail_on_none : bool
            Denotes is the function should fail if None is passed as data.
            Defaults to True.
        raise_none_error : bool
            Raise a ValueError if data in None; defaults to True.

        Returns
        -------
        bool
            Denotes if function succeeded (True) or failed (False). If
            `raise_none_error` is set to `True`, a ValueError will be thrown
            instead of False being returned.

        :param data: The data for the Plot.
        :param raise_none_error: Raise an Error if data is None
        :param error_on_fail: Raise an Error if validation fails; otherwise, return False
        :return: True is data is valid, or False if not (and if error_on_fail is False)
        """
        # Provide the most basic check so we can just super it - child
        # classes are still required to define this method themselves
        if data is None:
            if fail_on_none:
                if error_on_fail:
                    raise ValueError("Data must not be None")
                return False
            return True
        return None  # None denotes the child version should continue

    @property
    def data(self):
        return self._data

    @data.setter
    def data(self, d):
        raise NotImplementedError(
            "In general, you may not set the data "
            "attribute of a Plot directly. Use the "
            "add_data helper function instead."
        )

    @property
    def title(self):
        if self._title is None:
            return ""
        return self._title

    @title.setter
    def title(self, t):
        self._title = str(t)

    @abc.abstractmethod
    def add_data(self, *args, **kwargs):
        """
        Add data to this Plot object.

        Different Plot objects will accept different data structures; some
        may accept multiple data structures. Refer to the documentation for
        each specific Plot object for what is required.
        """
        raise NotImplementedError("The child Plot must define this function.")
