# SPDX-License-Identifier: BSD-3-Clause
import os
import re
import sys
import shutil
from pathlib import Path
import uvicorn
import inspect
import logging
import __main__
import importlib
from fastapi import FastAPI
from pydantic import BaseModel
from adari_core.report import AdariReportBase

import json

logger = logging.getLogger("adari_core")


class ReportModel(BaseModel):
    report_name: str
    sof: str
    output: str


class AdariServerBase:
    """
    Base class to be used by the different ADARI servers
    """

    # This needs to be defined outside of __init__,
    # since FastAPI scans for members of the class when
    # run() is called.
    adariserver = FastAPI(title="adariserver")

    def __init__(self, uds=None):
        # A dictionary of the reports known to this server
        # map of str: AdariReportBase
        self.report_dict = {}
        # List of drivers supported by this server
        self.drivers_list = []
        # Renderer
        self.renderer = None
        self.renderer_config = None
        self.uds = uds

    def loadreport(self, report_name: str):
        """
        Loads a report given the name
        """

        report_mod = None
        install_reports_dir = None
        report_path_method = None
        # Add  a directory where the reports can be expected
        if os.environ.get("ADARI_REPORTS_DIR") is not None:
            install_reports_dir = os.environ.get("ADARI_REPORTS_DIR")
            report_path_method = "ADARI_REPORTS_DIR environment variable"

        # This assumes that esorex and the adari reports, which are installed
        # by the pipelines, are installed under the same "prefix".
        # And that the reports directory installation
        # is under {prefix}/share/esopipes/reports.
        # This would need to be changed if that assumtion does not hold
        elif shutil.which("esorex"):
            esorex_path = Path(shutil.which("esorex"))
            install_reports_dir = os.path.join(
                os.path.dirname(esorex_path), "..", "share", "esopipes", "reports"
            )
            report_path_method = "esorex path"

        # This assumes that adari and the adari reports are installed
        # with the same "prefix". And that the reports directory installation
        # is under {prefix}/share/esopipes/reports.
        # This would need to be changed if that assumtion does not hold
        # __main__.__file__ refers to the location of genreport
        # (Under {prefix}bin/)
        elif hasattr(__main__, "__file__"):
            install_reports_dir = os.path.join(
                os.path.dirname(__main__.__file__), "..", "share", "esopipes", "reports"
            )
            report_path_method = "genreport path"

        # Explore install_reports_dir for available reports
        if install_reports_dir is not None:
            for root, _, files in os.walk(
                install_reports_dir, topdown=False, followlinks=True
            ):
                for name in files:
                    if re.match(report_name + ".py", name):
                        if report_name in self.report_dict:
                            self.report_dict[report_name]["root"].append(root)
                        else:
                            self.report_dict[report_name] = {"root": [root]}
            # Only proceed if at least one instance of report found
            if report_name in self.report_dict:
                # Raise error if multiple copies of report installed
                if len(self.report_dict[report_name]["root"]) > 1:
                    raise LookupError(
                        "Multiple instances of {0} found: {1}".format(
                            report_name,
                            ", ".join(
                                str(p) for p in self.report_dict[report_name]["root"]
                            ),
                        )
                    )
                # Load the module if only one report installed
                elif len(self.report_dict[report_name]["root"]) == 1:
                    init_file = os.path.join(
                        self.report_dict[report_name]["root"][0], "__init__.py"
                    )

                    spec = importlib.util.spec_from_file_location(
                        "report_mod", init_file
                    )
                    report_mod = importlib.util.module_from_spec(spec)
                    sys.modules[spec.name] = report_mod
                    spec.loader.exec_module(report_mod)

        if report_mod is not None:
            for member in inspect.getmembers(report_mod):
                if member[0] == report_name:
                    for report_member in inspect.getmembers(member[1]):
                        obj = report_member[1]
                        if isinstance(obj, AdariReportBase):
                            self.report_dict[report_name]["report"] = obj

        if report_name not in self.report_dict:
            err_msg = (
                " No report found with name {0} within parent folder {1}\n"
                "(HINT: parent determined by {2})".format(
                    report_name, install_reports_dir, report_path_method
                )
            )
            logging.error(err_msg)
            return 4  # Error code for report names that cannot be found

        if "report" not in self.report_dict[report_name]:
            err_msg = (
                " The report {0} has been found but is not part of a module\n"
                "(HINT: maybe it should be added to {1}/{2})".format(
                    report_name, self.report_dict[report_name]["root"][0], "__init__.py"
                )
            )
            logging.error(err_msg)
            return 4

    def runreport(self, report_name: str, sof: str, output: str):
        """
        Runs a report given the name and the input file with the SOF
        """

        # Configure the report with the runtime information
        self.report_dict[report_name]["report"].set_renderer(self.renderer)
        if self.report_dict[report_name]["report"].set_sof(sof) is not None:
            err_msg = " Run report failed"
            logging.error(err_msg)
            return 4

        # Run the report
        self.report_dict[report_name]["report"].run()

        # Write the output JSON file with the artifacts information
        artifacts = self.report_dict[report_name]["report"].get_artifacts()
        with open(output, "w") as f:
            json.dump(artifacts, f, indent=2)
        return 0

    def initialise_main_interface(self):
        """
        Gives access to use self in FastAPI callbacks
        """

        @self.adariserver.get("/ping")
        async def ping():
            return {"ping": "OK"}

        @self.adariserver.get("/driver_support/{driver}")
        async def driver_support(driver: str):
            return driver in self.drivers_list

        @self.adariserver.get("/loadreport/{report_name}")
        async def loadreport(report_name: str):
            logger.info("Loading report " + report_name)
            retval = self.loadreport(report_name)
            # will return None or exit code
            if retval:
                return retval

        @self.adariserver.post("/runreport/{report_name}/{sof}/{output}")
        async def runreport(report_name: str, sof: str, output: str):
            if report_name in self.report_dict:
                logger.info("Executing report " + report_name)
                retval = self.runreport(report_name, sof, output, False)
                if retval:
                    return retval

        # an additional runreport to accept json payloads
        @self.adariserver.post("/runreport")
        async def runreport(rdata: ReportModel):
            if rdata.report_name in self.report_dict:
                logging.info("Executing report: " + rdata.report_name)
                retval = self.runreport(rdata.report_name, rdata.sof, rdata.output)
                if retval:
                    return retval

    def run(self, **kwargs):
        """
        Run the server
        """
        adari_logger = logging.getLogger("adari_core")
        if kwargs.get("log_level") is not None:
            adari_logger.setLevel(kwargs.get("log_level"))

        # This gives access to utilise 'self' within the callbacks
        self.initialise_main_interface()
        if self.uds is not None:
            unix_socket = self.uds
        else:
            if os.environ.get("XDG_RUNTIME_DIR") is not None:
                unix_socket = os.environ["XDG_RUNTIME_DIR"]
            else:
                if sys.platform.startswith("linux"):  # linux
                    unix_socket = "/run/user/" + str(os.getuid())
                elif sys.platform.startswith("darwin"):  # macOS
                    if not os.path.isdir("/private/tmp/adari_" + str(os.getuid())):
                        os.makedirs(
                            "/private/tmp/adari_" + str(os.getuid()), mode=0o700
                        )
                    unix_socket = "/private/tmp/adari_" + str(os.getuid())
                else:  # Try a generic option. Other startswith() options:
                    # 'freebsd', 'aix', 'emscripten', 'wasi', 'win32', 'cygwin',
                    # 'msys', 'os2', 'os2emx', 'riscos', 'atheos', 'openbsd', etc.
                    if not os.path.isdir("/var/run/user/" + str(os.getuid())):
                        os.makedirs("/var/run/user/" + str(os.getuid()), mode=0o700)
                    unix_socket = "/var/run/user/" + str(os.getuid())
            unix_socket += "/adari_server.socket"

        logger.info("Running FastAPI server on" + unix_socket)
        uvicorn.run(
            self.adariserver,
            uds=unix_socket,
            use_colors=False,
            log_level=(
                kwargs.get("log_level")
                if kwargs.get("log_level") is not None
                else "info"
            ),
        )
