# SPDX-License-Identifier: BSD-3-Clause
import os
import uvicorn
import inspect
import __main__
from importlib.machinery import SourceFileLoader
from fastapi import FastAPI
from pydantic import BaseModel
from multiprocessing import Process
from adari_core.report import AdariReportBase
import json
from adari_core.server.server_base import AdariServerBase

import logging
import socket
from threading import Thread

from bokeh.plotting import figure, show
from tornado.ioloop import IOLoop
from bokeh.models import ColumnDataSource, Select, LogColorMapper
from bokeh.models import (
    BoxSelectTool,
    SaveTool,
    PanTool,
    BoxZoomTool,
    WheelZoomTool,
    HoverTool,
    ResetTool,
)
from bokeh.models import ColorBar
from bokeh.events import Press, PressUp, Tap, MenuItemClick
from bokeh.client import pull_session
from bokeh.embed import server_session
from bokeh.layouts import gridplot, layout, grid
from bokeh.server.server import Server

import numpy as np

import time

from fastapi.templating import Jinja2Templates
from fastapi.staticfiles import StaticFiles
from fastapi import Request

from adari_core.backends.bokeh import BokehRenderer

logger = logging.getLogger(__name__)


class ReportModel(BaseModel):
    report_name: str
    sof: str
    output: str


class BokehServer:
    # basic template for bokeh usage
    embed_string = """
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <title>Title</title>
    </head>
    <body>
        <div id="app1">
        {{ script1 | safe }}
      </div>
    </body>
    </html>
    """

    templates = Jinja2Templates(directory=str(os.environ.get("ADARI_TEMPLATES_DIR")))

    def __init__(self, server=None, port=0):
        self.parent_port = 0
        self.bokeh_port = self.get_free_tcp_port()
        self.bokeh_url = f"http://127.0.0.1:{self.bokeh_port}/qc_plot/"
        self.bokeh_server = None
        self.bokeh_thread = Thread(target=self.bokeh_worker)
        self.reports = []

    def start_bokeh(self):
        self.bokeh_thread.start()

    @staticmethod
    def get_free_tcp_port():
        tcp = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        tcp.bind(("", 0))
        addr, port = tcp.getsockname()
        tcp.close()
        return port

    def bokeh_worker(self):
        logger.info("Starting bokeh server..")
        try:
            main_app = {"/qc_plot": self.bokeh_app}
            self.bokeh_server = Server(
                main_app,
                io_loop=IOLoop(),
                allow_websocket_origin=[
                    f"127.0.0.1:{self.parent_port}",
                    f"localhost:{self.parent_port}",
                    f"127.0.0.1:{self.bokeh_port}",
                ],
                port=self.bokeh_port,
            )
            logger.info(f"Bokeh Server port = {self.bokeh_server.port}")
            self.bokeh_server.start()
            self.bokeh_server.io_loop.start()

            self.bokeh_server.stop()
            self.bokeh_server = None

            logger.info("Bokeh Server has Stopped")
        except Exception as e:
            logger.debug(e.with_traceback())

    def render_fastapi(self, request: Request):
        return self.templates.TemplateResponse(
            "bokeh.html", {"request": request, "bokeh_page_url": self.bokeh_url}
        )

    def add_reports(self, docs):
        if docs is not None:
            for d in docs:
                self.reports.append(d)

    def bokeh_app(self, doc):
        doc.clear()
        logger.debug("bokeh_app: self.reports = ", self.reports)
        for d in self.reports:
            doc.add_root(d)
        # x_array = np.array([10,20,30,40,50,60])
        # y_array = np.array([50,60,70,80,90,100])
        ## Create a line plot
        # plot = figure()
        # plot.line(x_array, y_array)
        # doc.add_root(plot)
        return doc
        # self.doc = plot
        # self._doc_builder.set_doc(doc)
        # return self._doc_builder.get_doc()


class AdariServerBokeh(AdariServerBase):
    """
    Base class to be used by the different ADARI servers
    """

    # This needs to be defined outside of __init__,
    # since FastAPI scans for members of the class when
    # run() is called.

    bokeh_server = BokehServer()

    def __init__(self, renderer_config: map, uds=None):
        # A list of the reports known to this server
        # map of str: AdariReportBase
        self.uds = None
        super().__init__(self.uds)
        self.drivers_list.append("bokeh")
        self.renderer_config = renderer_config
        self.renderer = BokehRenderer()

    def runreport(self, report_name: str, sof: str, output: str):
        """
        Runs a report given the name and the input file with the SOF
        """

        # Configure the report with the runtime information
        self.report_list[report_name].set_renderer(self.renderer)
        self.report_list[report_name].set_sof(sof)

        # Run the report
        docs = self.report_list[report_name].run()

        # Once the report is ready, hand it over to Bokeh to add it to
        # the set of docs to create
        logger.debug("run report: docs= ", docs)
        self.bokeh_server.add_reports(docs)

        # Write the output JSON file with the artifacts information
        # artifacts = self.report_list[report_name].get_artifacts()
        # with open(output, "w") as f:
        #    json.dump(artifacts, f, indent=2)

    def initialise_main_interface(self):
        """
        Gives access to use self in FastAPI callbacks
        """

        self.adariserver.mount(
            "/static", StaticFiles(directory=os.environ.get("ADARI_STATIC_DIR"))
        )
        super().initialise_main_interface()

        @self.adariserver.get("/bokeh")
        async def redirect_bokeh(request: Request):
            return self.bokeh_server.render_fastapi(request)

    def run(self):
        """
        Run the server
        """
        # This gives access to utilise 'self' within the callbacks
        self.initialise_main_interface()

        # start Bokeh
        self.bokeh_server.start_bokeh()

        logger.debug("Running FastAPI server on localhost:8000")
        logger.debug("Access Bokeh from http://localhost:8000/bokeh")
        uvicorn.run(self.adariserver, host="localhost", port=8000)
