import pytest

from adari_core.report import AdariReportBase

import inst_reports.espresso as espresso
import inst_reports.kmos as kmos
import inst_reports.muse as muse
import inst_reports.uves as uves
import inst_reports.hawki as hawki
import inst_reports.fors as fors



def data_reader_plain(fake_filename):
    return "string_thing"


def data_reader_list(*args):
    return "list_thing"


def data_reader_dict(**kwargs):
    return "dict_thing"


class DummyReport(AdariReportBase):
    def __init__(self):
        super().__init__("dummy_report")
        self.data_readers = {
            "string": data_reader_plain,
            "list": data_reader_list,
            "tuple": data_reader_list,
            "dict": data_reader_dict,
        }

    def generate_panels(self, **kwargs):
        return

    def parse_sof(self):
        return [
            {"string": "string_name",
             "list": ["string_name1", "string_name2"],
             "tuple": ("string_name1", "string_name2", ),
             "dict": {"first_file": "string_name1",
                      "second_file": "string_name2"}},
        ]


class TestReports:
    def test_bare_report_init(self):
        with pytest.raises(TypeError):  # Cannot create abstract class
            r = AdariReportBase()

    def test_bare_report_version(self):
        class DummyReport(AdariReportBase):
            def __init__(self):
                super().__init__("dummy_report")

            def generate_panels(self, **kwargs):
                return

            def parse_sof(self):
                return

        rep = DummyReport()
        with pytest.raises(AttributeError):  # Cannot leave version un-set
            _ = rep.version()

    @pytest.mark.parametrize("module", [espresso, kmos, muse, uves, hawki, fors])
    def test_report_modules_all(self, module):
        """Make sure that each report module has an all with at least one
        member"""
        try:
            assert len(module.__all__) > 0, (
                f"Module {module} __all__ has no " f"members!"
            )
        except (AttributeError, TypeError):
            raise AssertionError(
                f"Module {module} has no __all__ attribute, "
                f"or its length could not be measured"
            )

    @pytest.mark.parametrize(
        "report",
        espresso.__all__
        + uves.__all__
        + kmos.__all__
        + muse.__all__
        + hawki.__all__
        + fors.__all__,
    )
    class TestReportBasics:
        def test_report_init(self, report):
            r = report.rep
            assert isinstance(r, AdariReportBase), (
                f"{r.name} is not " f"an instance " f"of AdariReportBase"
            )

        def test_report_version(self, report):
            r = report.rep
            assert r.version, f"Failed to get version of {report.name}"

    def test_report_data_reader_args(self):
        d = DummyReport()

        # "Parse" the SOF, and "open" the resulting HDUs
        d.set_hdus(d.parse_sof())

        # Check we have the expected result
        assert len(d.hdus) == 1, f"self.hdus has length {len(d.hdus)}!"
        assert d.hdus[0]["string"] == "string_thing", "String opener failed"
        assert d.hdus[0]["list"] == "list_thing", "List opener failed"
        assert d.hdus[0]["tuple"] == "list_thing", "Tuple opener failed"
        assert d.hdus[0]["dict"] == "dict_thing", "Dict opener failed"
