from adari_core.plots.collapse import CollapsePlot
from adari_core.plots.images import ImagePlot
from adari_core.plots.panel import Panel
from adari_core.plots.points import LinePlot
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase

import os


class MasterCalibratorReport(AdariReportBase):
    def __init__(self, name: str):
        super().__init__(name)

    def parse_sof(self):
        raise NotImplementedError(
            "MasterWaveReport is a template only, "
            "the child Report is responsible for "
            "defining parse_sof"
        )

    def generate_panels(self, descr="", collapse_y_label="", **kwargs):
        panels = {}
        vspace = 0.3
        ext = "DATA"

        for filedict in self.hdus:
            mean = filedict["mean"]
            cube = filedict["cube"]
            spec = filedict["spec"]

            panel = Panel(3, 4, height_ratios=[1, 4, 4, 4])

            # Text Plot
            ext0 = "PRIMARY"
            fname = os.path.basename(str(mean.filename()))
            instrume = str(mean[ext0].header.get("INSTRUME"))
            procatg = str(mean[ext0].header.get("HIERARCH ESO PRO CATG"))

            t1 = TextPlot(columns=1, v_space=vspace)

            col1 = (
                instrume,
                "EXTNAME: " + ext,
                "PRO CATG: " + procatg,
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(mean[ext0].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
            )
            t1.add_data(col1)
            panel.assign_plot(t1, 0, 0, xext=1)

            # Data Plot: white-light image
            plot_data = ImagePlot(
                mean[ext].data,
                title=procatg,
                v_clip="percentile",
                v_clip_kwargs={"percentile": 95.0},
            )
            panel.assign_plot(plot_data, 0, 1)

            # Collapse:
            collapse = CollapsePlot(
                mean[ext].data,
                "x",
                title=procatg,
                color="black",
                x_label="pixels",
                y_label=collapse_y_label,
            )
            collapse.add_data(mean[ext].data, "y", color="red")
            panel.assign_plot(collapse, 1, 1)

            # Difference in planes:
            pl1 = 200
            pl2 = cube[ext].header.get("NAXIS3") - pl1
            title = (
                str(cube[ext0].header.get("HIERARCH ESO PRO CATG"))
                + ": \n"
                + str(pl1)
                + "-"
                + str(pl2)
            )
            plot_data = ImagePlot(
                cube[ext].data[pl1] - cube[ext].data[pl2],
                title=title,
                v_clip="percentile",
                v_clip_kwargs={"percentile": 95.0},
            )
            panel.assign_plot(plot_data, 2, 1)

            # Spectrum
            spectrum = LinePlot(
                title="Spectrum",
            )
            spectrum.add_data(
                [
                    spec[1].data["WAVE"][0],
                    spec[1].data["FLUX"][0],
                ],
                label="Flux",
            )
            spectrum.x_label = "Wavelength (um)"
            spectrum.y_label = "Flux"
            panel.assign_plot(spectrum, 0, 2, xext=3)

            input_files = [mean.filename(), cube.filename(), spec.filename()]

            if "eff" in list(filedict.keys()):
                eff = filedict["eff"]

                # Efficiency
                efficiency = LinePlot(
                    title="Efficiency",
                )
                efficiency.add_data(
                    [
                        eff[1].data["WAVE"],
                        eff[1].data["EFF"],
                    ],
                    label="Efficiency",
                )
                efficiency.x_label = "Wavelength (um)"
                efficiency.y_label = "Efficiency"
                panel.assign_plot(efficiency, 0, 3, xext=3)
                input_files.append(eff.filename())

            panels[panel] = {
                "report_name": "{}_{}".format(instrume, procatg.lower()),
                "report_description": "ERIS {} panel - {}".format(descr, fname),
                "report_tags": [],
                "input_files": input_files,
            }
        return panels
