# SPDX-License-Identifier: BSD-3-Clause

from adari_core.plots.histogram import HistogramPlot
from adari_core.plots.images import ImagePlot

from .master_spec_science import MasterSpecScienceReport

import numpy as np


class MasterIfuSpecScienceReport(MasterSpecScienceReport):
    """Master report class for display of plots for single science ifu spectrum

    Description
    -----------
    Generate plots for a single IFU science data
    """

    def __init__(self, name: str):
        super().__init__(name)
        self.im_scaling = {
            "v_clip": "percentile",
            "v_clip_kwargs": {"percentile": 95.0},
        }
        self.hist_scaling = {"nsigma": 3}

    def parse_sof(self):
        raise NotImplementedError(
            "MasterIfuSpecScienceReport is a template only, "
            "the child Report is responsible for "
            "defining parse_sof"
        )

    def generate_image_plot(self, im):
        """
        Creates a plot with white-light image.

        Parameters:
        -----------
        im : 2D iterable of numeric values
            The image array

        Returns
        -------
        fullimage : ImagePlot
            A plot that can be added to a panel
        """

        fullimage = ImagePlot(
            title="White-light image",
            **self.im_scaling,
        )
        fullimage.add_data(im)
        return fullimage

    def generate_exposure_plot(self, expmap):
        """
        Creates a plot with exposure map.

        Parameters:
        -----------
        expmap : 2D iterable of numeric values
            The exposure map array

        Returns
        -------
        expimage : ImagePlot
            A plot that can be added to a panel
        """
        expimage = ImagePlot(title="Exposure map")
        expimage.add_data(expmap)
        scaling = {"low": 0, "high": np.max(expmap)}
        expimage.set_v_clip_method("val", **scaling)
        return expimage

    def generate_flux_histogram(self, flux, flux_label="F"):
        """
        Creates a plot with histogram of the data cube around 0 flux.

        Parameters:
        -----------
        flux : 2D iterable of numeric values
            The data cube with flux

        Returns
        -------
        fluxhist : HistogramPlot
            A plot that can be added to a panel
        """
        fluxhist = HistogramPlot(
            title="Flux histogram",
            bins=50,
            legend=False,
            x_label=flux_label,
            v_clip_kwargs=self.hist_scaling,
        )
        fluxhist.add_data(flux, label="data counts")
        return fluxhist

    def generate_panels(self, **kwargs):
        panels = {}
        return panels
