# SPDX-License-Identifier: BSD-3-Clause
from adari_core.report import AdariReportBase
from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot, CentralImagePlot
from adari_core.plots.cut import CutPlot
from adari_core.plots.histogram import HistogramPlot
from adari_core.plots.text import TextPlot
from adari_core.plots.collapse import CollapsePlot

from adari_core.utils.utils import fetch_kw_or_default, fetch_kw_or_error
import adari_core.utils.clipping as clipping

import logging
import numpy as np

logger = logging.getLogger(__name__)


class MasterRawCutsReport(
    AdariReportBase,
    # clipping.ClippingMixin
):
    def __init__(self, name: str):
        super().__init__(name)
        self.center_size = 200
        self.hist_bins_max = 50

    def remove_raw_scan(self, im_hdu, **kwargs):
        """
        Remove the pre/overscan regions from a raw image as required.

        As defined in :any:`MasterRawCutsReport`, the default action is no-op.

        Parameters
        ----------
        im_hdu : ImageHDU
            The input ImageHDU.

        Returns
        -------
        stripped_hdu : ImageHDU
            The image data from im_hdu, with the pre/overscan regions removed.

        """
        # Default option: no-op
        return im_hdu

    def parse_sof(self):
        """
        Convert the input SOF into a list of filenames to open.

        See the documentation of :any:`AdariReportBase.parse_sof` for full
        details.

        Returns
        -------
        list of dicts
            Each dict in the list specifies the inputs required to create
            a single :any:`Panel`. Children reports of MasterRawCutsReport
            require the following inputs to be defined for each panel:

            - ``"master_im"`` - a processed input image
            - ``"raw_im"`` *(optional)* - the raw image that "master_im" was
              derived from
        """
        raise NotImplementedError(
            "MasterRawCutsReport is a template only, "
            "the child Report is responsible for "
            "defining parse_sof"
        )

    def generate_panels(
        self,
        master_im_ext=0,
        raw_im_ext=0,
        raw_im_rotation={},
        raw_title="Raw image",
        master_title="Master image",
        master_center_title=None,
        im_clipping="sigma",
        im_n_clipping=3,
        im_zoom_clipping=None,
        im_zoom_n_clipping=None,
        hist_clipping="sigma",
        hist_n_clipping=4,
        master_im_clipping="percentile",
        master_im_n_clipping=50,
        master_im_zoom_clipping=None,
        master_im_zoom_n_clipping=None,
        cut_clipping=None,
        cut_n_clipping=None,
        cut_highest_min=None,
        cut_lowest_max=None,
        cut_min_span=None,
        cut_cent_clipping=None,
        cut_cent_n_clipping=None,
        cut_cent_highest_min=None,
        cut_cent_lowest_max=None,
        cut_cent_min_span=None,
        collapse_clipping=None,
        collapse_n_clipping=None,
        collapse_dir_1="x",
        collapse_dir_2="y",
        collapse_title=None,
        interpolation=None,
        **kwargs,
    ):
        """
        Master report for showing the images, histograms, cut plots and
        collapse plots for basic bias & dark images.

        This report displays attributes of a processed 'master' image,
        and optionally, one of the 'raw' images that the 'master' was
        derived from.

        If supplied, raw image data will be subjected to
        :any:`MasterRawCutsReport.remove_raw_scan` prior to plotting.

        Each :any:`Panel` generated by this report has the following
        :any:`Plot` arrangement in a 5x3 grid (or 4x3 if raw images are
        not supplied):

        - First row: left empty (this is designed to allow child reports
          to insert :any:`TextPlot` metadata)
        - Second row:
            - First column: :any:`ImagePlot` showing full raw image (only if
              raw image supplied; otherwise column does not exist)
            - Second column: :any:`ImagePlot` showing full master image
            - Third column: :any:`CutPlot` through the master (and possibly raw)
              image in the first cut direction (see Parameters)
            - Fourth column: :any:`CutPlot` through the master (and possibly
              raw) image in the second cut direction (see Parameters)
            - Fifth column: :any:`HistogramPlot` of master (and possibly raw)
              image counts
        - Third row:
            - First through fourth columns: As for the second row, but
              displaying/using :any:`CentralImagePlot` linked to the relevant
              :any:`ImagePlot` from the second row
            - Fifth column: :any:`CollapsePlot` of the *full* master (and
              possibly raw) image, in the first cut direction (see
              Parameters).

        Parameters
        ----------
        master_im_ext : int or str, optional
            The extension of the ``"master_im"`` file to examine.
        raw_im_ext : int or str, optional
            The extension of the ``"raw_im"`` file to examine.
        raw_im_rotation : dict or None, optional
            Image rotation arguments to be applied to the ``"raw_im"``
            image display. Takes the form of::

                rotation_kwargs = {"rotate": <rotation angle>,
                                   "flip": <"x"|"y">}

            Leave out a particular option (or the whole dict) to no-op.
        raw_title : str or None, optional
            Title for the raw image display.
        master_title : str or None, optional
            Title for the master image display.
        master_center_title : str or None, optional
            Title for the master image central region display.
        im_clipping : str, or None, optional
            The image clipping mode to use for the raw image. Allowed
            values are:

            - ``"auto"`` - percentile-based "auto" clipping
              (:any:`adari_core.utils.clipping.clipping_auto`)
            - ``"sigma"`` - standard deviation clipping
              (:any:`adari_core.utils.clipping.clipping_std`)
            - ``"mad"`` - median absolute deviation clipping
              (:any:`adari_core.utils.clipping.clipping_mad`)
            - ``"minmax"`` - minimum-maximum clipping (i.e., whole data range)
              (:any:`adari_core.utils.clipping.clipping_minmax`)
            - ``"percentile"`` - percentile clipping
              (:any:`adari_core.utils.clipping.clipping_percentile`)
            - ``"val"`` - explicit start and end values to clipping
              (:any:`adari_core.utils.clipping.clipping_val`)
        im_n_clipping : numeric (single, two-tuple, or three-tuple), optional
            The argument(s) to be supplied to the clipping function selected
            by ``im_clipping``.
        im_zoom_clipping, im_zoom_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            central region display of the raw image.
        hist_clipping, hist_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            histogram display of the raw image.
        master_im_clipping, master_im_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            display of the master image.
        master_im_zoom_clipping, master_im_zoom_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            central region display of the master image.
        cut_clipping, cut_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            data used to generate the cut plots.
        cut_highest_min, cut_lowest_max : numeric, optional
            If ``cut_clipping`` is not ``None``, these values represent
            minimum and maximum clipping values that will be imposed if the
            automatically-generated values do not reach them. E.g., if
            ``cut_highest_min=1``, and the automatically-generated minimum
            cut value is 2, the CutPlot will use a lower clipping value of 1.
        cut_min_span : numeric, optional
            Sets a guaranteed minimum span of the data range in the CutPlot.
            Will over-ride the auto-computed data range if necessary.
        cut_cent_clipping, cut_cent_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            data used to generate the central region cut plots.
        cut_cent_highest_min, cut_cent_lowest_max : optional
            As for ``cut_highest_min, cut_lowest_max``, but applied to the
            data used to generate the central region cut plots.
        cut_cent_min_span : numeric, optional
            As for ``cut_min_span``, but applied to the
            data used to generate the central region cut plots.
        collapse_clipping, collapse_n_clipping : optional
            As for ``im_clipping`` and ``im_n_clipping``, but applied to the
            data used to generate the collapse plot.
        collapse_dir_1 : <"x", "y">, optional
            Collapse direction for the first CollapsePlot.
        collapse_dir_2 : <"x", "y">, optional
            Collapse direction for the first CollapsePlot.
        collapse_title : str or None, optional
            Title for the CollapsePlot.

        Returns
        -------
        dict
            A dictionary, with :any:`Panel` objects as the keys and a
            metadata dictionary as the values. One key-value pair will be
            generated for each output element of :any:`parse_sof`.

            In addition to the standard returns in the metadata dictionary
            (see :any:`AdariReportBase`), this report returns the following
            additional information:

            - ``"master_im"`` : The master image filename
            - ``"master_im_ext"`` : The extension of the master image used
            - ``"master_procatg"`` : The ESO PRO CATG header value of the
              master image
        """
        panels = {}

        # Ensure backwards compatability
        if not im_zoom_clipping:
            im_zoom_clipping = im_clipping
            im_zoom_n_clipping = im_n_clipping
        if not master_im_zoom_clipping:
            master_im_zoom_clipping = master_im_clipping
            master_im_zoom_n_clipping = master_im_n_clipping

        # Parse the scaling arguments
        clip_dict = {}
        clipscalings = [
            "im_clipping",
            "im_zoom_clipping",
            "master_im_clipping",
            "master_im_zoom_clipping",
            "cut_clipping",
            "cut_cent_clipping",
            "hist_clipping",
            "collapse_clipping",
        ]
        clipmodes = [
            im_clipping,
            im_zoom_clipping,
            master_im_clipping,
            master_im_zoom_clipping,
            cut_clipping,
            cut_cent_clipping,
            hist_clipping,
            collapse_clipping,
        ]
        clipargs = [
            im_n_clipping,
            im_zoom_n_clipping,
            master_im_n_clipping,
            master_im_zoom_n_clipping,
            cut_n_clipping,
            cut_cent_n_clipping,
            hist_n_clipping,
            collapse_n_clipping,
        ]
        for this_scaling, this_mode, this_argument in zip(
            clipscalings, clipmodes, clipargs
        ):
            if this_mode is None:
                this_param = this_argument = None
            elif this_mode == "auto":
                this_param = "percentiles"
            elif this_mode == "sigma":
                this_param = "nsigma"
            elif this_mode == "mad":
                this_param = "nmad"
            elif this_mode == "minmax":
                this_param = this_argument = None
            elif this_mode == "percentile":
                this_param = "percentile"
            elif this_mode == "val":
                this_param = ["low", "high"]
            else:
                raise NotImplementedError(
                    "Clipping method {0} for {1} is not implemented".format(
                        this_mode, this_scaling
                    )
                )
            if isinstance(this_param, list):
                for this_p, this_a in zip(this_param, this_argument):
                    if this_scaling in clip_dict:
                        clip_dict[this_scaling][this_p] = this_a
                    else:
                        clip_dict[this_scaling] = {this_p: this_a}
            elif this_param is None:
                clip_dict[this_scaling] = {}
            else:
                clip_dict[this_scaling] = {this_param: this_argument}

        # Raw im ext - allow for single or multiple values
        if type(raw_im_ext) is str or type(raw_im_ext) is int:
            raw_im_ext = [
                raw_im_ext,
            ] * len(self.hdus)
        try:
            assert len(raw_im_ext) >= len(self.hdus), (
                "Not enough raw_im_exts " "values to match to all " "file lists"
            )
        except AssertionError as e:
            raise ValueError(str(e))

        if type(raw_im_rotation) is dict:
            raw_im_rotation = [
                raw_im_rotation,
            ] * len(self.hdus)
        try:
            assert len(raw_im_rotation) >= len(self.hdus), (
                "Not enough raw_im_rotation " "dicts to match to all file " "lists"
            )
        except AssertionError as e:
            raise ValueError(str(e))

        for i, filedict in enumerate(self.hdus):
            master_im = filedict["master_im"]
            _ = master_im[master_im_ext]
            try:
                raw_im = filedict["raw_im"]
                raw_im_ext_i = raw_im[raw_im_ext[i]]
                raw_im_space = 1
                master_procatg = fetch_kw_or_error(
                    master_im["PRIMARY"], "HIERARCH ESO PRO CATG"
                )
            except KeyError:
                # This must be an instance where we don't use the raw_im -
                # set it to None
                raw_im = None
                raw_im_space = 0
            # No need for try/except - FileNotFound and IndexError are the
            # correct errors to raise here if there's an issue

            if raw_im is not None:
                p = Panel(5, 3, height_ratios=[1, 4, 4])
            else:
                p = Panel(4, 3, height_ratios=[1, 4, 4])

            master_procatg = fetch_kw_or_error(master_im[0], "HIERARCH ESO PRO CATG")

            if raw_im is not None:
                raw_data = self.remove_raw_scan(raw_im_ext_i)
                logger.debug(f"Raw data shape: {raw_data.shape}")
                raw_plot = ImagePlot(
                    raw_data.data,
                    title=raw_title,
                    v_clip=im_clipping,
                    v_clip_kwargs=clip_dict["im_clipping"],
                    **raw_im_rotation[i],
                )
                raw_plot.interp = interpolation
                p.assign_plot(raw_plot, 0, 1)

                raw_center = CentralImagePlot(
                    raw_plot,
                    extent=self.center_size,
                    title=f"{raw_title} - central {self.center_size}px",
                    v_clip=im_zoom_clipping,
                    v_clip_kwargs=clip_dict["im_zoom_clipping"],
                )
                p.assign_plot(raw_center, 0, 2)

            logger.debug(f"Master data shape: {master_im[master_im_ext].data.shape}")
            master_plot = ImagePlot(
                master_im[master_im_ext].data,
                title=f"{master_procatg}",
                v_clip=master_im_clipping,
                v_clip_kwargs=clip_dict["master_im_clipping"],
            )
            master_plot.interp = interpolation
            p.assign_plot(master_plot, 0 + raw_im_space, 1)

            this_master_center_title = (
                f"{master_procatg} - central {self.center_size}px"
                if master_center_title is None
                else master_center_title
            )

            master_center = CentralImagePlot(
                master_plot,
                extent=self.center_size,
                title=this_master_center_title,
                v_clip=master_im_zoom_clipping,
                v_clip_kwargs=clip_dict["master_im_zoom_clipping"],
            )
            master_center.interp = interpolation
            p.assign_plot(master_center, 0 + raw_im_space, 2)

            # Plot cuts
            # Get mid line
            # Cut plot X
            cutX = CutPlot("x", title="Central column", x_label="y")

            if raw_im is not None:
                cutX.y_label = fetch_kw_or_default(raw_im_ext_i, "BUNIT", default="ADU")
                cutX.add_data(
                    raw_plot,
                    raw_plot.get_data_coord(raw_plot.data.shape[1] // 2, "x"),
                    label="raw",
                    color="black",
                )
            else:
                cutX.y_label = fetch_kw_or_default(master_im[0], "BUNIT", default="ADU")
            cutX.add_data(
                master_plot,
                master_plot.get_data_coord(master_plot.data.shape[1] // 2, "x"),
                label="master",
                color="red",
            )
            if cut_clipping:
                this_clip = getattr(clipping, "clipping_" + cut_clipping)
                this_cut = master_plot.get_axis_cut(
                    cutX.cut_ax, cutX._data["master"]["cut_pos"]
                )
                cutX.y_min, cutX.y_max = this_clip(
                    this_cut, **clip_dict["cut_clipping"]
                )
                # This expands by a factor of 2 the range of the percentile
                cutX.y_min = -cutX.y_max / 2 + 3.0 / 2.0 * cutX.y_min
                cutX.y_max = -cutX.y_min / 2 + 3.0 / 2.0 * cutX.y_max
                if cut_highest_min:
                    cutX.y_min = min(cut_highest_min, cutX.y_min)
                if cut_lowest_max:
                    cutX.y_max = max(cut_lowest_max, cutX.y_max)
                if cut_min_span:
                    if raw_im is not None:
                        median_data = np.median(cutX.data["raw"]["data"].data)
                    else:
                        median_data = np.median(cutX.data["master"]["data"].data)
                    cutX.y_min = min(
                        median_data - cut_min_span / 2.0,
                        cutX.y_min,
                    )
                    cutX.y_max = max(
                        median_data + cut_min_span / 2.0,
                        cutX.y_max,
                    )

            p.assign_plot(cutX, 1 + raw_im_space, 1)

            # Cut plot Y
            cutY = CutPlot("y", title="Central row", x_label="x")
            if raw_im is not None:
                cutY.y_label = fetch_kw_or_default(
                    raw_im[raw_im_ext_i], "BUNIT", default="ADU"
                )
                cutY.add_data(
                    raw_plot,
                    raw_plot.get_data_coord(raw_plot.data.shape[0] // 2, "y"),
                    label="raw",
                    color="black",
                )
            else:
                cutY.y_label = fetch_kw_or_default(master_im[0], "BUNIT", default="ADU")
            cutY.add_data(
                master_plot,
                master_plot.get_data_coord(master_plot.data.shape[0] // 2, "y"),
                label="master",
                color="red",
            )
            if cut_clipping:
                this_cut = master_plot.get_axis_cut(
                    cutY.cut_ax, cutY._data["master"]["cut_pos"]
                )
                cutY.y_min, cutY.y_max = this_clip(
                    this_cut, **clip_dict["cut_clipping"]
                )
                # This expands by a factor of 2 the range of the percentile
                cutY.y_min = -cutY.y_max / 2 + 3.0 / 2.0 * cutY.y_min
                cutY.y_max = -cutY.y_min / 2 + 3.0 / 2.0 * cutY.y_max
                if cut_highest_min:
                    cutY.y_min = min(cut_highest_min, cutY.y_min)
                if cut_lowest_max:
                    cutY.y_max = max(cut_lowest_max, cutY.y_max)
                if cut_min_span:
                    if raw_im is not None:
                        median_data = np.median(cutY.data["raw"]["data"].data)
                    else:
                        median_data = np.median(cutY.data["master"]["data"].data)
                    cutY.y_min = min(
                        median_data - cut_min_span / 2.0,
                        cutY.y_min,
                    )
                    cutY.y_max = max(
                        median_data + cut_min_span / 2.0,
                        cutY.y_max,
                    )

            p.assign_plot(cutY, 2 + raw_im_space, 1)

            # Do same for the central region
            # Cut plot X
            cutX_cent = CutPlot(
                "x", title="Central region - central column", x_label="y"
            )
            if raw_im is not None:
                cutX_cent.y_label = fetch_kw_or_default(
                    raw_im[raw_im_ext_i], "BUNIT", default="ADU"
                )
                cutX_cent.add_data(
                    raw_center,
                    raw_center.get_data_coord(raw_center.data.shape[1] // 2, "x"),
                    label="raw",
                    color="black",
                )
            else:
                cutX_cent.y_label = fetch_kw_or_default(
                    master_im[0], "BUNIT", default="ADU"
                )
            cutX_cent.add_data(
                master_center,
                master_center.get_data_coord(master_center.data.shape[1] // 2, "x"),
                label="master",
                color="red",
            )
            if cut_cent_clipping:
                this_clip = getattr(clipping, "clipping_" + cut_cent_clipping)
                this_cut = master_center.get_axis_cut(
                    cutX_cent.cut_ax, self.center_size // 2
                )
                cutX_cent.y_min, cutX_cent.y_max = this_clip(
                    this_cut, **clip_dict["cut_cent_clipping"]
                )
                # This expands by a factor of 2 the range of the percentile
                cutX_cent.y_min = -cutX_cent.y_max / 2 + 3.0 / 2.0 * cutX_cent.y_min
                cutX_cent.y_max = -cutX_cent.y_min / 2 + 3.0 / 2.0 * cutX_cent.y_max
                if cut_cent_highest_min:
                    cutX_cent.y_min = min(cut_cent_highest_min, cutX_cent.y_min)
                if cut_cent_lowest_max:
                    cutX_cent.y_max = max(cut_cent_lowest_max, cutX_cent.y_max)
                if cut_cent_min_span:
                    if raw_im is not None:
                        median_data = np.median(cutX_cent.data["raw"]["data"].data)
                    else:
                        median_data = np.median(cutX_cent.data["master"]["data"].data)
                    cutX_cent.y_min = min(
                        median_data - cut_cent_min_span / 2.0,
                        cutX_cent.y_min,
                    )
                    cutX_cent.y_max = max(
                        median_data + cut_cent_min_span / 2.0,
                        cutX_cent.y_max,
                    )

            p.assign_plot(cutX_cent, 1 + raw_im_space, 2)

            # Cut plot Y
            cutY_cent = CutPlot("y", title="Central region - central row", x_label="x")
            if raw_im is not None:
                cutY_cent.y_label = fetch_kw_or_default(
                    raw_im[raw_im_ext_i], "BUNIT", default="ADU"
                )
                cutY_cent.add_data(
                    raw_center,
                    raw_center.get_data_coord(raw_center.data.shape[0] // 2, "y"),
                    label="raw",
                    color="black",
                )
            else:
                cutY_cent.y_label = fetch_kw_or_default(
                    master_im[0], "BUNIT", default="ADU"
                )
            cutY_cent.add_data(
                master_center,
                master_center.get_data_coord(master_center.data.shape[0] // 2, "y"),
                label="master",
                color="red",
            )
            if cut_cent_clipping:
                this_cut = master_center.get_axis_cut(
                    cutY_cent.cut_ax, self.center_size // 2
                )
                cutY_cent.y_min, cutY_cent.y_max = this_clip(
                    this_cut, **clip_dict["cut_cent_clipping"]
                )
                # This expands by a factor of 2 the range of the percentile
                cutY_cent.y_min = -cutY_cent.y_max / 2 + 3.0 / 2.0 * cutY_cent.y_min
                cutY_cent.y_max = -cutY_cent.y_min / 2 + 3.0 / 2.0 * cutY_cent.y_max
                if cut_cent_highest_min:
                    cutY_cent.y_min = min(cut_cent_highest_min, cutY_cent.y_min)
                if cut_cent_lowest_max:
                    cutY_cent.y_max = max(cut_cent_lowest_max, cutY_cent.y_max)
                if cut_cent_min_span:
                    if raw_im is not None:
                        median_data = np.median(cutY_cent.data["raw"]["data"].data)
                    else:
                        median_data = np.median(cutY_cent.data["master"]["data"].data)
                    cutY_cent.y_min = min(
                        median_data - cut_cent_min_span / 2.0,
                        cutY_cent.y_min,
                    )
                    cutY_cent.y_max = max(
                        median_data + cut_cent_min_span / 2.0,
                        cutY_cent.y_max,
                    )

            p.assign_plot(cutY_cent, 2 + raw_im_space, 2)

            # Plot histogram
            histogram = HistogramPlot(
                master_data=master_im[master_im_ext].data,
                raw_data=raw_data.data if raw_im is not None else None,
                bins=self.hist_bins_max,
                title="Counts histogram",
                v_clip=hist_clipping,
                v_clip_kwargs=clip_dict["hist_clipping"],
            )
            # PIPE-11005 - Intelligently compute the number of bins for int plots
            if raw_im is not None:
                bin_per_val = self.hist_bins_max + 1  # Dummy val to start while
                vals_per_bin = 1
                rmin, rmax = histogram.get_vlim()
                while bin_per_val > self.hist_bins_max:
                    bin_per_val = (rmax - rmin + 1) // vals_per_bin
                    vals_per_bin += 1
                histogram.bins = bin_per_val

            p.assign_plot(histogram, 3 + raw_im_space, 1)

            # Text Plot
            text = TextPlot(columns=2, v_space=0.3)
            instru = master_im["PRIMARY"].header.get("INSTRUME", "N/A")
            text.add_data(text.metadata(master_im, master_im_ext))
            p.assign_plot(text, 0, 0, xext=2)

            # Collapse plot
            this_collapse_title = (
                f"{master_procatg} - collapse"
                if collapse_title is None
                else collapse_title
            )
            collapse = CollapsePlot(
                master_im[master_im_ext].data,
                collapse_dir_1,
                title=this_collapse_title,
                color="black",
                x_label="pixels",
            )
            if raw_im is not None:
                collapse.y_label = fetch_kw_or_default(
                    raw_im[raw_im_ext_i], "BUNIT", default="ADU"
                )
            else:
                collapse.y_label = fetch_kw_or_default(
                    master_im[master_im_ext], "BUNIT", default="ADU"
                )
            collapse.add_data(
                master_im[master_im_ext].data, collapse_dir_2, color="red"
            )
            if collapse_clipping:
                collapse.set_v_clip_method(
                    collapse_clipping, **clip_dict["collapse_clipping"]
                )
                [collapse.y_min, collapse.y_max] = collapse.get_vlim()

            p.assign_plot(collapse, 3 + raw_im_space, 2)

            input_files = [filedict["master_im"].filename()]
            if raw_im is not None:
                input_files.append(filedict["raw_im"].filename())
            panels[p] = {
                "master_im": master_im.filename(),
                "master_im_ext": master_im_ext,
                "master_procatg": master_procatg,
                "report_name": f"{instru}_{master_procatg.lower()}_{master_im_ext}",
                "report_description": f"Master cuts panel -  ({master_im.filename()}, "
                f"{master_im_ext})",
                "report_tags": [],
                "input_files": input_files,
            }

            if raw_im is not None:
                panels[p]["raw"] = raw_im.filename()
                panels[p]["raw_im_ext"] = raw_im_ext[i]

        return panels
