# SPDX-License-Identifier: BSD-3-Clause
import abc
import os
import re
from astropy.io import fits
from abc import ABCMeta
import logging

# master_report_logger = getLogger(__name__)
# master_report_self.logger.debug(f"Report logger instatiated "
#                            f"({master_report_self.logger.name}), "
#                            f"eff level "
#                            f"{master_report_self.logger.getEffectiveLevel()}")

logger = logging.getLogger(__name__)


def default_data_reader(filename):
    """
    A basic data reader for FITS files.

    Parameters
    ----------
    filename : str
        The name of the file to be opened.

    Returns
    -------
    :any:`astropy.io.fits.HDUList`
        The opened file object. It is opened in readonly mode.
    """
    return fits.open(filename, mode="readonly")


class AdariReportBase(metaclass=ABCMeta):
    """
    Base class to be used by all ADARI reports

    Parameters
    ----------
    name : str
        Name of the report
    """

    def __init__(self, name: str):
        self.name = name
        self.files = []
        self.inputs = []
        self.hdus = []
        self.data_readers = {}
        self.logger = logger
        self._version = None

    def version(self):
        """
        Return the version of this report.

        The version is a concatenation of the ADARI version used, and
        the version of the report (or 'unversioned' if `self.version` is
        `None`):, e.g.::

          <adari version number>/<report version number>
          <adari version number>/unversioned

        Returns
        -------
        version : str
            String representation of the report version
        """
        if self._version is None:
            raise AttributeError("Version has not been set for this report!")
        return f"{self._version}"

    def set_renderer(self, renderer):
        """
        Set the renderer to use in this report at runtime.

        Parameters:
        -----------
        renderer : str
            The name of the renderer
        """
        self.renderer = renderer

    def set_sof(self, sof):
        """
        Set the Set Of Frames
        The format of the file is as follows:

        /filepath/name1.fits    PRO_CATG1
        /filepath/name2.fits    PRO_CATG2
        ....

        Parameters:
        -----------
        sof : str
            Filename with the Set Of Frames
        """
        # Check if SOF file exists
        if not os.path.isfile(sof):
            err_msg = " SOF file does not exist or is not a regular file: {0} ".format(
                sof
            )
            self.logger.error(err_msg)
            return -1
        fbase = os.path.dirname(sof) + "/"

        try:
            sof_fd = open(sof, "r")
        except OSError:
            err_msg = " Problem with opening sof file with name: {0} ".format(sof)
            self.logger.error(err_msg)
            return -1

        # Create the list of files and categories that are listed in the SOF
        lines_sof = sof_fd.readlines()
        for x in lines_sof:
            xs = x.split()
            if len(xs) == 2 and not re.search("^#", x):
                fpath = xs[0]
                if not re.search("^\s*[\/\$]", fpath) and len(fbase):
                    fpath = fbase + fpath
                if re.search("^\s*\$", fpath):
                    m = re.search("^\s*\$(\w+)", fpath)
                    if m:
                        var = m.group(1)
                        resolv = os.environ.get(var)
                        fpath = resolv + "/" + fpath[len(var) + 2 :]
                self.files.append(fpath)
                self.inputs.append((fpath, xs[1]))
        sof_fd.close()

        self.logger.debug("Report SOF set")

    def set_hdus(self, files_list: list):
        """
        Convert the file lists from :any:`parse_sof` to a list of opened HDUs.

        For each dict in the input `files_list`, a new dict is formed which
        contains opened astropy :any:`HDUList` objects rather than file names.
        This new dict will be appended to the report's `hdus` attribute.
        There is a one-to-one mapping between the elements of the output of
        :any:`parse_sof` and the resulting list held in `self.hdus`.

        In it's default implementation, this function maps file types to
        file reader functions by looking up the mapping relationship in
        `self.data_readers`. If a file type isn't found in
        `self.data_readers`, a default reader is used, which simply
        opens the file.

        Importantly, this function returns a reference to the entire file,
        i.e., an :any:`HDUList`. It is up to the :any:`generate_panels`
        function (and any sub-functions it calls) to select the extension(s)
        to read. (This is done to allow a single run of a report to access,
        e.g., both the primary HDU, and one or more of the data extensions,
        without storing multiple references to the same file. It also allows
        the files to be closed cleanly at the end of the run.)

        Parameters
        ----------
        files_list : list of dicts
            The files_list to be read-in to the report. This should be the
            output of the :any:`parse_sof` function.
        """

        for filedict in files_list:
            hdudict = {}
            for t, fn in filedict.items():
                try:
                    reader = self.data_readers[t]
                except KeyError:
                    reader = default_data_reader

                if isinstance(fn, list) or isinstance(fn, tuple):
                    hdudict[t] = reader(*fn)
                elif isinstance(fn, dict):
                    hdudict[t] = reader(**fn)
                else:
                    hdudict[t] = reader(fn)
            self.hdus.append(hdudict)

        self.logger.debug("Have set report hdus property")

        return

    def close_hdus(self):
        """
        Close the open HDUs belonging to this report.
        """
        while len(self.hdus) > 0:
            hdudict = self.hdus.pop()
            for k in hdudict.keys():
                hdudict[k].close()  # Close the HDUList

        self.hdus = {}  # Throw away the now-useless dict

        return

    def get_artifacts(self):
        """
        Get the artifacts created by this report
        """
        artifacts = self.renderer.get_artifacts()
        for artifact in artifacts:
            if "input_files" not in artifact:
                artifact["input_files"] = self.files
        return artifacts

    @abc.abstractmethod
    def parse_sof(self):
        """
        Break the input SOF into individual components for each Panel.

        A given report may need to make intelligent decisions about
        which files in a given SOF need to be reported on, in what
        combination, and frequently in multiple combinations.

        Note that this function only filters SOFs at the *file* level.
        If further granularity is required (e.g., specific extensions,
        repeats over particular extensions, or special read-out requirements),
        this should be handled internally by :any:`generate_panels`.
        This is to ensure the file headers remain maximally accessible
        during :any:`Panel` creation.

        Returns
        -------
        list of dicts
            The return from this function should be a list of dicts. Each
            dict will contain the filenames necessary to create a single Panel
            for the given report. The report will use the keys "filetype..."
            in the run() method to get the proper filenames. Note that this is
            relevant for reports that inherit from masters in data_libs, since
            the parse_sof() in the instrument report should define map keys
            that are understood by the run() method in the master report.
            The dictionary takes the form of::

                {
                "filetype1": <relevant filename>,
                "filetype2": <relevant filename>,
                ...
                }
        """
        raise NotImplementedError("Individual reports need to implement " "parse_sof")

    @abc.abstractmethod
    def generate_panels(self, **kwargs):
        """
        Generate the :any:`Panel` (s) for a given report.

        Returns
        -------
        panels : dict of dicts
            The :any:`Panel` (s) to be rendered for this report. The dict takes
            the following format::

                { First panel : {
                    'report_name': '<report_name>',
                    'report_description': '<report_description>',
                    'report_tags': [<report tags>] },
                  Second panel : {
                    'report_name': '<report_name>',
                    'report_description': '<report_description>',
                    'report_tags': [<report tags>] },
                  }, ...
                }

        """
        raise NotImplementedError("Reports should implement generate_panels()")

    def run(self):
        """
        Do a complete run of this report, including rendering.

        This method will execute the following functions in sequence:
        parse_sof
        generate_panels

        It will then take the returned :any:`Panel` object(s), and render
        them using the requested renderer.
        """
        self.logger.debug("Report run has triggered")
        self.logger.debug("*** LOGGER INFORMATION ***")
        self.logger.debug(
            f"Logger level: {logging.getLevelName(self.logger.getEffectiveLevel())}"
        )
        self.logger.debug(f"Logger handlers: {self.logger.handlers}")
        self.logger.debug(
            f"Logger formatters: {list(h.formatter for h in self.logger.handlers)}"
        )
        self.logger.debug(f"Logger propagation setting: {self.logger.propagate}")

        # Parse files
        file_lists = self.parse_sof()

        # Convert the file lists into opened HDU objects
        self.set_hdus(file_lists)

        # Generate panels
        panel_list = self.generate_panels()

        # results list required for bokeh backend
        results = self.renderer.render_panels(panel_list)

        # Close out the used HDUs
        self.close_hdus()
        return results
